﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_BaseId.autogen.h>

#include "../../kern_Platform.h"
#include "../ARM/kern_RegisterAccess.h"
#include "../../kern_DebugSelect.h"
#include "../ARM/kern_KContext.h"
#include "../../kern_Kernel.h"
#include "kern_MemoryMap.h"
#include "../../kern_KProcess.h"
#include "../../kern_KSynchronization.h"
#include "../../kern_KTaggedAddress.h"
#include "../../kern_KScheduler.h"
#include "kern_Config.h"
#include "kern_ExceptionHandler.h"
#include "../../kern_DebugString.h"
#include "../ARM/kern_ExceptionContext.h"
#include "../../kern_KScopedSchedulingLock.h"
#include "../../kern_PageTableSelect.h"
#include "kern_KPageTableBody.h"
#include "../ARM/kern_MemoryCopy.h"
#include "kern_MemoryMap.h"
#include "../../kern_DpcManager.h"
#include "../../kern_MemoryCopySelect.h"
#include "../../kern_Utility.h"

#define KDEBUG_INFO_ONE     //KEventInfoの作成を1回に減少

using namespace nn::svc;
using namespace nn::kern::ARM;

namespace nn { namespace kern { namespace ARMv7A {
namespace
{
template <typename T>
    T* GetContextAtSuperVisorStackBottom(void* pBottom)
{
    return reinterpret_cast<T*>(reinterpret_cast<uintptr_t>(pBottom) - sizeof(KThread::ParamsOnStack)) - 1;
}

}

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KDebug);

uintptr_t KDebug::RetrivePc(const KThread& thread)
{
    return GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(thread.GetKernelStackBottom())->pc;
}

Result KDebug::GetThreadContext_core(ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    const ARM::ExceptionContext* abortContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pThread->GetKernelStackBottom());

    if (controlFlags & nn::svc::ContextFlag_General)
    {
        if (!pThread->IsCallingSvc() || pThread->GetSvcNo() == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            for (size_t i = 0; i < sizeof(abortContext->r) / sizeof(abortContext->r[0]); ++i)
            {
                pContext->r[i] = abortContext->r[i];
            }
            pContext->r[12] = abortContext->ip;
        }
    }

    if (controlFlags & nn::svc::ContextFlag_Control)
    {
        if (pThread->IsCallingSvc() && abortContext->write == 0)
        {
            if (abortContext->cpsr & HW_PSR_THUMB_STATE)
            {
                pContext->pc    = abortContext->pc - 2;
            }
            else
            {
                pContext->pc    = abortContext->pc - 4;
            }
        }
        else
        {
            pContext->pc    = abortContext->pc;
        }
        pContext->r[11] = abortContext->r[11];
        pContext->sp    = abortContext->sp_usr;
        pContext->lr    = abortContext->lr_usr;
        pContext->cpsr  = (abortContext->cpsr & HW_PSR_PL0_MASK);
        pContext->tpidr = abortContext->tpidr;
    }

    return GetFPUContext(pContext, pThread, controlFlags);
}

Result KDebug::SetThreadContext_core(const ThreadContext& context, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    ARM::ExceptionContext* abortContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pThread->GetKernelStackBottom());

    if (controlFlags & nn::svc::ContextFlag_General)
    {
        for (size_t i = 0; i < sizeof(abortContext->r) / sizeof(abortContext->r[0]); ++i)
        {
            abortContext->r[i] = context.r[i];
        }
        abortContext->ip = context.r[12];
    }

    if (controlFlags & nn::svc::ContextFlag_Control)
    {
        if (pThread->IsCallingSvc())
        {
            abortContext->write = 1;
        }

        abortContext->sp_usr  = context.sp;
        abortContext->lr_usr  = context.lr;
        abortContext->pc      = context.pc;
        abortContext->cpsr = ((context.cpsr & HW_PSR_PL0_MASK) | (abortContext->cpsr & ~HW_PSR_PL0_MASK));
        abortContext->tpidr = context.tpidr;
    }

    return SetFPUContext(context, pThread, controlFlags);
}

Result KDebug::GetFPUContext(ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pThread != &GetCurrentThread());
    if (!(controlFlags & (nn::svc::ContextFlag_FpuControl | nn::svc::ContextFlag_Fpu)))
    {
        return ResultSuccess();
    }

    KContext*  pThreadContext = pThread->GetContext();
    if (controlFlags & nn::svc::ContextFlag_FpuControl)
    {
        pContext->fpexc = pThreadContext->GetFpexc();
        pContext->fpexc |= HW_FPEXC_VFP_ENABLE;
        pContext->fpscr = pThreadContext->GetFpscr();
    }

    if (controlFlags & nn::svc::ContextFlag_Fpu)
    {
        const uint64_t* fpu = pThreadContext->GetVfpRegisters();
        for (size_t i = 0; i < pThreadContext->GetNumOfVfpRegisters(); i++)
        {
            pContext->fpuRegisters[i] = fpu[i];
        }
    }

    return ResultSuccess();
}

Result KDebug::SetFPUContext(const ThreadContext& context, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pThread != &GetCurrentThread());

    if (!(controlFlags & (nn::svc::ContextFlag_FpuControl | nn::svc::ContextFlag_Fpu)))
    {
        return ResultSuccess();
    }

    KContext* pThreadContext = pThread->GetContext();

    if (controlFlags & nn::svc::ContextFlag_FpuControl)
    {
        Bit32 fpexc;
        fpexc = context.fpexc & ~HW_FPEXC_VFP_ENABLE;
        pThreadContext->SetFpexc(fpexc);
        pThreadContext->SetFpscr(context.fpscr);
    }

    if (controlFlags & nn::svc::ContextFlag_Fpu)
    {
        pThreadContext->SetVfpRegisters(context.fpuRegisters);
    }

    return ResultSuccess();
}

void KDebug::SetPreviousPc()
{
    KDisableInterrupt di;
    KThread* pCurrent = &GetCurrentThread();
    NN_KERN_ASSERT(pCurrent->IsCallingSvc());
    ARM::ExceptionContext* swiContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pCurrent->GetKernelStackBottom());

    if (swiContext->write == 0)
    {
        //プログラムカウンタを1命令戻す
        if (swiContext->cpsr & HW_PSR_THUMB_STATE)
        {
            swiContext->pc -= 2;
        }
        else
        {
            swiContext->pc -= 4;
        }
        swiContext->write       = 1;
    }
}

void KDebug::PrintRegister(KThread* pThread)
{
#ifdef NN_KERN_FOR_DEVELOPMENT
    if (!pThread)
    {
        pThread = &GetCurrentThread();
    }
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->SuspendRequest(KThread::SuspendType_BackTrace);
                }
            }
        }

        const ARM::ExceptionContext* swiContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pThread->GetKernelStackBottom());
        NN_UNUSED(swiContext);
        NN_KERN_RELEASE_LOG("R8:    0x%08x\n", swiContext->r[8]);
        NN_KERN_RELEASE_LOG("R9:    0x%08x\n", swiContext->r[9]);
        NN_KERN_RELEASE_LOG("R10:   0x%08x\n", swiContext->r[10]);
        NN_KERN_RELEASE_LOG("R11:   0x%08x\n", swiContext->r[11]);
        NN_KERN_RELEASE_LOG("SP:    0x%08x\n", swiContext->sp_usr);
        NN_KERN_RELEASE_LOG("LR:    0x%08x\n", swiContext->lr_usr);
        NN_KERN_RELEASE_LOG("PC:    0x%08x\n", swiContext->pc - 4);
        NN_KERN_RELEASE_LOG("CPSR:  0x%08x\n", swiContext->cpsr);
        NN_KERN_RELEASE_LOG("TPIDR: 0x%08x\n", swiContext->tpidr);
        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->Resume(KThread::SuspendType_BackTrace);
                }
            }
        }
    }
#else
    NN_UNUSED(pThread);
#endif
}

void KDebug::PrintBacktrace(KThread* pThread)
{
#ifdef NN_KERN_FOR_DEVELOPMENT
    if (!pThread)
    {
        pThread = &GetCurrentThread();
    }

    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->SuspendRequest(KThread::SuspendType_BackTrace);
                }
            }
        }

        const ARM::ExceptionContext* swiContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pThread->GetKernelStackBottom());
        NN_KERN_RELEASE_LOG("User Backtrace\n");
        NN_KERN_RELEASE_LOG("   %p\n", swiContext->lr_usr);
        uintptr_t frameAddress = static_cast<uintptr_t>(swiContext->r[11]);
        for (int i = 0; i < 32; i++)
        {
            if (!frameAddress)
            {
                break;
            }
            if (frameAddress & 0x3)
            {
                break;
            }
            struct
            {
                uint32_t frame;
                uint32_t lr;
            } frame;
            {
                KMemoryInfo mi;
                PageInfo pi;

                KPhysicalAddress pa;
                KVirtualAddress va;

                if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress).IsFailure())
                {
                    break;
                }
                if (!(mi.state & KMemoryState_FlagsMemory))
                {
                    break;
                }
                if (mi.attribute & KMemoryAttribute_Uncached)
                {
                    break;
                }
                if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                {
                    break;
                }
                if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress))
                {
                    break;
                }
                if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                {
                    break;
                }
                va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                uint32_t v = *GetTypedPointer<uint32_t>(va);
                if (swiContext->sp_usr <= v && v < swiContext->sp_usr + NN_KERN_FINEST_PAGE_SIZE )
                {
                    // clang
                    if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress + 4).IsFailure())
                    {
                        break;
                    }
                    if (!(mi.state & KMemoryState_FlagsMemory))
                    {
                        break;
                    }
                    if (mi.attribute & KMemoryAttribute_Uncached)
                    {
                        break;
                    }
                    if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                    {
                        break;
                    }
                    frame.frame = *GetTypedPointer<uint32_t>(va);
                    if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress + 4))
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                    {
                        break;
                    }
                    va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                    frame.lr = *GetTypedPointer<uint32_t>(va);
                }
                else
                {
                    // gcc
                    if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress - 4).IsFailure())
                    {
                        break;
                    }
                    if (!(mi.state & KMemoryState_FlagsMemory))
                    {
                        break;
                    }
                    if (mi.attribute & KMemoryAttribute_Uncached)
                    {
                        break;
                    }
                    if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                    {
                        break;
                    }
                    frame.lr = *GetTypedPointer<uint32_t>(va);
                    if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress - 4))
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                    {
                        break;
                    }
                    va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                    frame.frame = *GetTypedPointer<uint32_t>(va);
                }
            }
            NN_KERN_RELEASE_LOG("   %p\n", frame.lr);
            frameAddress = frame.frame;
        }
        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->Resume(KThread::SuspendType_BackTrace);
                }
            }
        }
    }
#else
    NN_UNUSED(pThread);
#endif
}

void KDebug::GetCurrentBacktrace(uintptr_t* pOut, size_t num)
{
    KThread* pThread = &GetCurrentThread();
    KProcess* pCurrentProcess = pThread->GetParentPointer();
    size_t i = 0;
    if (pCurrentProcess)
    {
        const ARM::ExceptionContext* swiContext = GetContextAtSuperVisorStackBottom<ARM::ExceptionContext>(pThread->GetKernelStackBottom());
        uintptr_t frameAddress = static_cast<uintptr_t>(swiContext->r[11]);
        uintptr_t returnAddress = swiContext->lr_usr;
        bool isClang = false;
        while (i < num)
        {
            pOut[i++] = returnAddress;

            if (!frameAddress)
            {
                break;
            }
            if (frameAddress & 0x3)
            {
                break;
            }

            if (i == 0)
            {
                uint32_t v;
                if (!CopyMemoryFromUser(&v, reinterpret_cast<const void*>(frameAddress), sizeof(v)))
                {
                    break;
                }

                isClang = (swiContext->sp_usr <= v && v < swiContext->sp_usr + NN_KERN_FINEST_PAGE_SIZE);
            }

            struct
            {
                uint32_t frame;
                uint32_t lr;
            } frame;

            if (!CopyMemoryFromUser(&frame, reinterpret_cast<const void*>(frameAddress + (isClang ? 0: - 4)), sizeof(frame)))
            {
                break;
            }

            frameAddress = frame.frame;
            returnAddress = frame.lr;
        }
    }
    for (size_t j = i; j < num; j++)
    {
        pOut[j] = 0;
    }
}

Result KDebug::BreakWhenAttached(const nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize)
{
    return OnDebugEvent(nn::svc::DebugEvent_Exception, nn::svc::DebugException_UserBreak, RetrivePc(GetCurrentThread()), reason, pData, byteSize);
}

}}}
