﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../../kern_Platform.h"
#include "crt0.h"
#include "../../../kern_CPUSelect.h"
#include "kern_MemoryMap.h"
#include "../../../kern_InterruptControllerSelect.h"
#include "../../ARM/kern_RegisterAccess.h"

using namespace nn::kern;
#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
namespace nn { namespace kern { namespace DEV { namespace ARMv7A { namespace crt0 {
void WaitForCore0(int coreNo)   __attribute__((section(".start")));
void WakeUpOtherCore()          __attribute__((section(".start")));
void Step2(int coreNo)      __attribute__((section(".start")));
}}}}}
namespace nn {
namespace
{
inline uintptr_t GetPrivateRegionPhysicalAddress()__attribute__((section(".start")));
inline GicDistributer* GetGicDistributer()__attribute__((section(".start")));
inline GicCpuInterface* GetGicCpuInterface()__attribute__((section(".start")));
inline int32_t GetNumInterruptLines()__attribute__((section(".start")));
inline int32_t DivUp(int32_t x, int32_t divider)__attribute__((section(".start")));
void EnableInterrupt(int coreNo) __attribute__((section(".start")));
void DisableInterrupt(int coreNo)__attribute__((section(".start")));
void SendSoftwareInterrupt(Bit32 receiverList, Bit32 id) __attribute__((section(".start")));
void WaitSoftwareInterrupt(Bit32 senderNo, Bit32 id) __attribute__((section(".start")));
void __cpp_initialize__aeabi_() __attribute__((section(".start")));
int32_t AtomicIncrement(volatile int32_t* ptr) __attribute__((section(".start")));
int32_t AtomicDecrement(volatile int32_t* ptr) __attribute__((section(".start")));
int32_t g_GicInUse __attribute__((section(".start.data")));

extern "C" {
    extern void (*__init_array_start []) ();
    extern void (*__init_array_end []) ();
}
void __cpp_initialize__aeabi_()
{
    for (void (**f)() = __init_array_start; f < __init_array_end; ++f)
    {
        (*f)();
    }
}
#define __isb(x)        do { asm volatile("isb":::"memory"); } while (false)
#define __wfi(x)        do { asm volatile("wfi":::"memory"); } while (false)
#endif

enum CoreNo
{
    CORE_NO_0,
    CORE_NO_1,
    CORE_NO_2,
    CORE_NO_3
};

const Bit32 HW_OS_IDR_SW_LIST_CORE0              = (1u << CORE_NO_0);
const Bit32 HW_OS_IDR_SW_LIST_CORE1              = (1u << CORE_NO_1);
const Bit32 HW_OS_IDR_SW_LIST_CORE2              = (1u << CORE_NO_2);
const Bit32 HW_OS_IDR_SW_LIST_CORE3              = (1u << CORE_NO_3);

const Bit32 INTERRUPT_ID_AWAKE_CORE1             = 8;
const Bit32 INTERRUPT_ID_CORE1_WORK              = 8;
const Bit32 INTERRUPT_ID_CORE0_FINISH            = 9;
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
const int32_t GIC_DISTRIBUTER_OFFSET    = 0x1000;
const int32_t GIC_CPU_INTERFACE_OFFSET  = 0x0100;
#else
const int32_t GIC_DISTRIBUTER_OFFSET    = 0x1000;
const int32_t GIC_CPU_INTERFACE_OFFSET  = 0x2000;
#endif

const Bit32 GIC_SGIR_FILTER_LIST    = 0x0;

const Bit32 GIC_SGIR_FILTER_MASK    = 0x03;
const Bit32 GIC_SGIR_CORE_LIST_MASK = 0xFF;
const Bit32 GIC_SGIR_ID_MASK        = 0x0F;
const Bit32 GIC_IAR_SENDER_MASK     = 0x07;
const Bit32 GIC_IAR_ID_MASK         = 0x3FF;

const int GIC_SGIR_FILTER_SHIFT     = 24;
const int GIC_SGIR_CORE_LIST_SHIFT  = 16;
const int GIC_SGIR_ID_SHIFT         =  0;
const int GIC_IAR_SENDER_SHIFT      = 10;
const int GIC_IAR_ID_SHIFT          =  0;


inline uintptr_t GetPrivateRegionPhysicalAddress()
{
    uintptr_t a;
    HW_GET_CP15_CONFIGURATION_BASE_ADDRESS(a);
    return a;
}

inline GicDistributer* GetGicDistributer()
{
    return reinterpret_cast<GicDistributer*>(
            NN_KERN_V_ADDR_CORE_PRIVATE_REGION + GIC_DISTRIBUTER_OFFSET);
}

inline GicCpuInterface* GetGicCpuInterface()
{
    return reinterpret_cast<GicCpuInterface*>(
            NN_KERN_V_ADDR_CORE_PRIVATE_REGION + GIC_CPU_INTERFACE_OFFSET);
}

inline int32_t GetNumInterruptLines()
{
    GicDistributer* const pd = GetGicDistributer();
    return ((pd->typer >> 0) & 0x1F) * 32 + 32;
}

inline int32_t DivUp(int32_t x, int32_t divider)
{
    return (x + (divider - 1)) / divider;
}

int32_t AtomicIncrement(volatile int32_t* ptr)
{
    int32_t val;
    int result;
    do
    {
        asm volatile("ldrex %0, [%1]" : "=&r"(val) : "r"(ptr): "memory");
        val++;
        asm volatile("strex  %0, %1, [%2]" :"=&r"(result) :"r"(val), "r"(ptr): "memory");
    } while (result != 0);
    return val;
}
int32_t AtomicDecrement(volatile int32_t* ptr)
{
    int32_t val;
    int result;
    do
    {
        asm volatile("ldrex %0, [%1]" : "=&r"(val) : "r"(ptr): "memory");
        val--;
        asm volatile("strex  %0, %1, [%2]" :"=&r"(result) :"r"(val), "r"(ptr): "memory");
    } while (result != 0);
    return val;
}

/*---------------------------------------------------------------------------*
Name:         ConfigVFP

Description:  VFP の設定

Arguments:    None

Returns:      None
 *---------------------------------------------------------------------------*/
void ConfigVFP()
{
    // VFP のパーミッション設定 (フルアクセス)
    HW_SET_CP15_C1(2, HW_C1_VFP_AP_PACK( HW_C1_AP_PRIV, HW_C1_AP_PRIV ));

    // プリフェッチバッファフラッシュ
    //  CP 15 への反映待ち
    __isb(0xf);

    // VFP を無効化
    Bit32 val = 0;
#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
    asm volatile(".fpu vfp\n");
    asm volatile("fmxr fpexc, %0"::"r"(val):);
    asm volatile(".fpu softvfp\n");
#else
#error not defined NN_BUILD_CONFIG_COMPILER_
#endif
}

/*---------------------------------------------------------------------------*
Name:         EnableInterrupt

Description:  割り込みの有効化

Arguments:    coreNo:     実行中のコア番号

Returns:      None.

Note:         ソフトウェア割り込みを受信するだけなので、IRQビットは無効のま
ま(割り込みベクタには飛ばさない)
 *---------------------------------------------------------------------------*/
void EnableInterrupt(int coreNo);
void EnableInterrupt(int coreNo)
{
    GicDistributer* const pd = GetGicDistributer();
    GicCpuInterface* const pci = GetGicCpuInterface();

    // CPU インターフェイスの有効化
    pci->pmr = 0xFF;
    pci->bpr = 7;
    pci->ctlr = 0x00000003;

    // 以下 core 0 のみ
    if( coreNo == 0 )
    {
        // 全て割り込み無効
        //   ただしソフトウェア割り込みは無効にならない
        const int32_t numIntLines       = GetNumInterruptLines();
        const int32_t numIntLines_32    = DivUp(numIntLines, 32);

        for( int i = 0; i < numIntLines_32; ++i )
        {
            pd->icenabler[i]  = 0xFFFFFFFFu;
        }

        // ディストリビュータの有効化
        pd->ctlr = 0x00000003;
    }
    AtomicIncrement(&g_GicInUse);
}


/*---------------------------------------------------------------------------*
Name:         DisableInterrupt

Description:  割り込みの無効化

Arguments:    coreNo:     実行中のコア番号

Returns:      None.
 *---------------------------------------------------------------------------*/
void DisableInterrupt(int coreNo);
void DisableInterrupt(int coreNo)
{
    GicCpuInterface* const pci = GetGicCpuInterface();

    // 分散割り込みコントローラの無効化 (core 0 以外のときのみ)
    if(AtomicDecrement(&g_GicInUse) == 0)
    {
        GicDistributer* const pd = GetGicDistributer();

        // ディストリビュータの無効化
        pd->ctlr = 0x00000000;
    }

    // CPU インターフェイスの無効化
    pci->ctlr = 0x00000000;
}

void SendSoftwareInterrupt(Bit32 receiverList, Bit32 id)
{
    GicDistributer* const pd = GetGicDistributer();
    pd->sgir = ( ((GIC_SGIR_FILTER_LIST & GIC_SGIR_FILTER_MASK)    << GIC_SGIR_FILTER_SHIFT)
            | ((receiverList         & GIC_SGIR_CORE_LIST_MASK) << GIC_SGIR_CORE_LIST_SHIFT)
            | ((id                   & GIC_SGIR_ID_MASK)        << GIC_SGIR_ID_SHIFT) );
}

void WaitSoftwareInterrupt(Bit32 waitCores, Bit32 id);
void WaitSoftwareInterrupt(Bit32 waitCores, Bit32 id)
{
    Bit32 value;
    GicCpuInterface* const pci = GetGicCpuInterface();

    do
    {
        __wfi();

        value = pci->iar;
        pci->eoir = value;

        if (((value >> GIC_IAR_ID_SHIFT) & GIC_IAR_ID_MASK) == id)
        {
            waitCores &= ~(1 << ((value >> GIC_IAR_SENDER_SHIFT) & GIC_IAR_SENDER_MASK));
        }

    } while( waitCores != 0 );
}

}
}

namespace nn { namespace kern { namespace DEV { namespace ARMv7A { namespace crt0 {

void WaitForCore0(int coreNo)
{
    NN_UNUSED(coreNo);

    GicCpuInterface* const pci = reinterpret_cast<GicCpuInterface*>(
        GetPrivateRegionPhysicalAddress() + GIC_CPU_INTERFACE_OFFSET);

    {
        // CPU インターフェイスの有効化
        pci->pmr = 0xFF;
        pci->bpr = 7;
        pci->ctlr = 0x00000003;
    }

    {
        const Bit32 waitValue = ( (CORE_NO_0 << 10) | INTERRUPT_ID_AWAKE_CORE1 );
        Bit32 value;

        do
        {
            __wfi();

            value = pci->iar;
            pci->eoir = value;

        } while( value != waitValue );
    }

    {
        // CPU インターフェイスの無効化
        pci->ctlr = 0x00000000;
    }
}

void WakeUpOtherCore()
{
#if defined NN_BUILD_CONFIG_HARDWARE_BDSLIMX6
    // Core 1～3 のエントリアドレスを設定
    for (int i = 1; i < KCPU::NUM_CORE; i++)
    {
        *reinterpret_cast<volatile Bit32*>(NN_KERN_P_ADDR_SRC_REGISTER + 0x20 + i * 8) = reinterpret_cast<Bit32>(_start);
    }

    // Core 1～3 のリセット解除
    *reinterpret_cast<volatile Bit32*>(NN_KERN_P_ADDR_SRC_REGISTER) |= ((0x7 << 22) | (0x7 << 14));
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
    // エントリアドレスを設定
    *reinterpret_cast<volatile Bit32*>(0x6000F100) = reinterpret_cast<Bit32>(_start);
    asm volatile("dsb sy":::"memory");
    for (int i = 1; i < KCPU::NUM_CORE; i++)
    {
        if (*reinterpret_cast<volatile Bit32*>(0x60007000 + 0x18 + 8 * (i - 1)) & ((1u << 22) | (1u << 17)))
        {
            *reinterpret_cast<volatile Bit32*>(0x60007000 + 0x18 + 8 * (i - 1)) = ((1u << 3) | (1u << 0));
            asm volatile("dsb sy":::"memory");
        }
    }
#endif
}

/*---------------------------------------------------------------------------*
  Name:         Step2

  Description:  メモリ管理が有効になってから KernelMain 直前までの処理

  Arguments:    coreNo: 実行中のコア番号

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Step2(int coreNo)
{
    // VFP の設定
    ConfigVFP();

    // Jazelle 状態を無効化
    {
        Bit32 v;

        HW_INST_MRC(p14,7,c1,c0,0,v);
        v &= ~0x3;
        v |= 0x1;
        HW_INST_MCR(p14,7,c1,c0,0,v);
    }

    {
        Bit32 dfr0;

        HW_GET_CP15_DFR0(dfr0);
        switch ((dfr0 >> 24) & 0xF)
        {
        case 0:
        case 1:
            {
                HW_SET_CP15_PMCR(0);
                HW_SET_CP15_PMUSERENR(0);
                HW_SET_CP15_PMINTENCLR(0xFFFFFFFF);
            }
            break;
        default:
            break;
        }
    }

    // ソフトウェア割り込みを使用できるように
    EnableInterrupt(coreNo);

    // 他の core との同期(2 ～ 4コアを想定)
    if( coreNo == 0 )
    {
        // セクションの初期化
        {
            const size_t ziWords = NN_KERN_V_ADDR_CODE_MAIN_ZI_SIZE / 4;
            for( size_t w = 0; w < ziWords; ++w )
            {
                *(reinterpret_cast<Bit32*>(NN_KERN_V_ADDR_CODE_MAIN_ZI) + w) = 0;
            }
        }

        // C++ Static Initializers の呼び出し
        __cpp_initialize__aeabi_();

        Bit32 coreMask = 0;
        if (KCPU::NUM_CORE == 2)
        {
            coreMask = HW_OS_IDR_SW_LIST_CORE1;
        }
        else if (KCPU::NUM_CORE == 3)
        {
            coreMask = HW_OS_IDR_SW_LIST_CORE1 | HW_OS_IDR_SW_LIST_CORE2;
        }
        else if (KCPU::NUM_CORE == 4)
        {
            coreMask = HW_OS_IDR_SW_LIST_CORE1 | HW_OS_IDR_SW_LIST_CORE2 | HW_OS_IDR_SW_LIST_CORE3;
        }

        if (coreMask != 0)
        {
            // core1 を起こす (ID=1)
            SendSoftwareInterrupt(coreMask, INTERRUPT_ID_AWAKE_CORE1);

            // core1 からの応答を待つ (ID=1)
            WaitSoftwareInterrupt(coreMask, INTERRUPT_ID_CORE1_WORK);

            // core1 通知 (ID=2) ※KernelMain へのジャンプを許可
            SendSoftwareInterrupt(coreMask, INTERRUPT_ID_CORE0_FINISH);
        }
    }
    else
    {
        // core0 へ通知 (ID=1)
        SendSoftwareInterrupt(HW_OS_IDR_SW_LIST_CORE0, INTERRUPT_ID_CORE1_WORK);

        // core0 からの応答を待つ (ID=2)
        WaitSoftwareInterrupt(HW_OS_IDR_SW_LIST_CORE0, INTERRUPT_ID_CORE0_FINISH);
    }

    // 割り込みを無効に
    DisableInterrupt(coreNo);
}
} } } } }

