﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/TargetConfigs/build_Fpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_KContext.h"
#include "../../kern_Kernel.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_KScheduler.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_DebugSelect.h"
#include "../../kern_DpcManager.h"
#include "../../kern_Utility.h"
#include "kern_ExceptionContext.h"

#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
#define __yield()   do { asm volatile("yield":::"memory"); } while (false)
#endif

namespace nn { namespace kern {
    namespace ARM64 {

namespace
{
#if defined(NN_BUILD_CONFIG_FPU_FP_ARM64)
    void EnableFpu()
    {
        Bit64 cpacr;
        HW_GET_CPACR_EL1(cpacr);
        cpacr |= HW_CPACR_EL1_FEN_ENABLE;
        HW_SET_CPACR_EL1(cpacr);
        KCPU::InstructionMemoryBarrier();
    }
    __attribute__((unused))
    bool IsFpuEnabled()
    {
        Bit64 cpacr;
        HW_GET_CPACR_EL1(cpacr);
        return !!(cpacr & HW_CPACR_EL1_FEN_MASK);
    }
#endif

    KVirtualAddress
    InitializeKernelStackForUserModeThreadStarter(
        KVirtualAddress stack,
        KVirtualAddress userModeEntryPoint,
        KVirtualAddress userModeStackBottom,
        uintptr_t            param,
        bool            isAArch64
        )
    {
        Bit32 psr;
        ExceptionContext* pContext = GetTypedPointer<ExceptionContext>(stack) - 1;

        if (isAArch64)
        {
            psr = HW_PSR_EL0T_MODE;
        }
        else
        {
            psr = (((userModeEntryPoint & 0x1) == 0) ? HW_PSR_ARM_STATE: HW_PSR_THUMB_STATE) | HW_PSR_USR_MODE;
        }

        std::memset(pContext, 0, sizeof(ExceptionContext));
        pContext->pc = GetAsInteger(userModeEntryPoint);
        pContext->psr = psr;
        pContext->x[0] = param;
        if (isAArch64)
        {
            pContext->sp = GetAsInteger(userModeStackBottom);
        }
        else
        {
            pContext->x[13] = GetAsInteger(userModeStackBottom);
        }

        return reinterpret_cast<uintptr_t>(pContext);
    }

    KVirtualAddress
    InitializeKernelStackForSupervisorModeThreadStarter(
        KVirtualAddress stack,
        KVirtualAddress entryPoint,
        uintptr_t            param )
    {
        // カーネルスタックに
        //   stack - 8      r0      スレッド引数
        //   stack - 0      PC      スレッドのメイン関数へのポインタ
        // を積む
        Bit64* pStack = GetTypedPointer<Bit64>(stack);
        *--pStack = GetAsInteger(entryPoint);
        *--pStack = param;
        return reinterpret_cast<uintptr_t>(pStack);
    }
}

void UserModeThreadStarter();
void SupervisorModeThreadStarter();

void onThreadStart()
{
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
    {
        KEnableInterrupt interruptEnabled;
        KDebug::OnDebugEvent(nn::svc::DebugEvent_CreateThread, GetCurrentThread().GetId(), GetAsInteger(GetCurrentThread().GetThreadLocalRegionAddr()), GetCurrentThread().GetEntryAddress());
    }

    while (GetCurrentThread().IsDpcRegistered())
    {
        DpcManager::DpcHandler();
    }
    GetCurrentThread().ClearInExceptionHandler();
}

Result KContext::Initialize(KVirtualAddress initialPc, KVirtualAddress kernelSp, KVirtualAddress userSp, uintptr_t firstArg, bool isUserMode, bool is64Bit, bool isMain)
{
    KVirtualAddress pc;
    KVirtualAddress sp;

    NN_KERN_ASSERT(kernelSp != Null<KVirtualAddress>());

    if( isUserMode )
    {
        pc = reinterpret_cast<uintptr_t>(&UserModeThreadStarter);
        sp = InitializeKernelStackForUserModeThreadStarter(RoundDown(kernelSp,16), initialPc, RoundDown(userSp,16), firstArg, is64Bit);
    }
    else
    {
        NN_KERN_ASSERT(is64Bit);
        if(isMain)
        {
            pc = initialPc;
            sp = RoundDown(kernelSp, 16);
        }
        else
        {
            pc = reinterpret_cast<uintptr_t>(&SupervisorModeThreadStarter);
            sp = InitializeKernelStackForSupervisorModeThreadStarter(RoundDown(kernelSp,16), initialPc, firstArg);
        }
    }

    Bit64*  pStack = GetTypedPointer<Bit64>(sp);

    // LR (PC)
    m_Lr = GetAsInteger(pc);
    m_Sp = reinterpret_cast<Bit64>(pStack);

    // r19-r29
    for (size_t i = 0; i < sizeof(m_CpuRegisters) / sizeof(*m_CpuRegisters); ++i)
    {
        m_CpuRegisters[i] = 0;
    }

#if defined(NN_BUILD_CONFIG_FPU_FP_ARM64)
    // DN = 0, FZ = 0, Rmode = 0
    m_Fpcr = HW_FPCR_RMODE_NEAR;
    m_Fpsr = 0;
    m_Cpacr = 0;
    for(size_t i = 0; i < sizeof(m_FpuRegisters) / sizeof(*m_FpuRegisters); ++i)
    {
        m_FpuRegisters[i]  = 0;
    }
#endif

    m_Lock = isMain;

    return ResultSuccess();
}

Result KContext::Finalize()
{
    return ResultSuccess();
}

void KContext::SetFirstArgument(uintptr_t firstArg, uintptr_t secondArg)
{
    Bit64* pStack = reinterpret_cast<Bit64*>(m_Sp);
    pStack[0] = firstArg;
    pStack[1] = secondArg;
}

#if defined(NN_BUILD_CONFIG_FPU_FP_ARM64)
void KContext::FpuSwitchHandler(KThread* pNext)
{
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
    NN_KERN_ASSERT(!IsFpuEnabled());

    // VFP を有効に
    EnableFpu();

    // コンテキストを復帰
    KProcess* pProcess = pNext->GetParentPointer();
    NN_KERN_NULL_ASSERT(pProcess);
    if (pProcess->Is64Bit())
    {
        RestoreFpu64(*pNext->GetContext());
    }
    else
    {
        RestoreFpu32(*pNext->GetContext());
    }

#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    pNext->IncrementNumFpuSwitch();
#endif
}

void KContext::CloneFpuStatus()
{
    Bit64 fpcr;
    Bit64 fpsr;
    KCPU::InstructionMemoryBarrier();
    if (IsFpuEnabled())
    {
        asm volatile ("mrs %0, fpcr":"=r"(fpcr)::"memory");
        asm volatile ("mrs %0, fpsr":"=r"(fpsr)::"memory");
    }
    else
    {
        fpcr = GetCurrentThread().GetContext()->GetFpcr();
        fpsr = GetCurrentThread().GetContext()->GetFpsr();
    }
    SetFpcr(fpcr);
    SetFpsr(fpsr);
}

#endif

void KContext::OnThreadCoreChanged(KThread* pThread)
{
    NN_UNUSED(pThread);
}

void KContext::OnThreadTerminating(const KThread* pThread)
{
    NN_UNUSED(pThread);
}

const Bit128* KContext::GetFpuRegisters() const
{
    return m_FpuRegisters;
}

Bit32 KContext::GetFpsr() const
{
    return m_Fpsr;
}
Bit32 KContext::GetFpcr() const
{
    return m_Fpcr;
}
void KContext::SetFpuRegisters(const Bit128* x, bool is64Bit)
{
    if (is64Bit)
    {
        for (size_t i = 0; i < sizeof(m_FpuRegisters) / sizeof(*m_FpuRegisters); i++)
        {
            m_FpuRegisters[i] = x[i];
        }
    }
    else
    {
        for (size_t i = 0; i < (sizeof(m_FpuRegisters) / sizeof(*m_FpuRegisters)) / 2; i++)
        {
            m_FpuRegisters[i] = x[i];
        }
    }
}
void KContext::SetFpsr(Bit32 fpsr)
{
    m_Fpsr = fpsr;
}
void KContext::SetFpcr(Bit32 fpcr)
{
    m_Fpcr = fpcr;
}

    }
}}


