﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>

//----------------------------------------------------------------------
//                  Program Status Register
//----------------------------------------------------------------------

#define HW_PSR_CPU_MODE_MASK       0x1f // CPU mode

#define HW_PSR_USR_MODE            0x10 // User mode
#define HW_PSR_FIQ_MODE            0x11 // FIQ mode
#define HW_PSR_IRQ_MODE            0x12 // IRQ mode
#define HW_PSR_SVC_MODE            0x13 // Supervisor mode
#define HW_PSR_MON_MODE            0x16 // Monitor mode
#define HW_PSR_ABORT_MODE          0x17 // Abort mode (prefetch/Data)
#define HW_PSR_HYP_MODE            0x1a // Hyp mode
#define HW_PSR_UNDEF_MODE          0x1b // Undefined instruction mode
#define HW_PSR_SYS_MODE            0x1f // System mode

#define HW_PSR_ARM_STATE           0x0  // ARM state
#define HW_PSR_THUMB_STATE         0x20 // Thumb state
#define HW_PSR_STATE_MASK          0x20

#define HW_PSR_FIQ_DISABLE         0x40 // FIQ Disable
#define HW_PSR_IRQ_DISABLE         0x80 // IRQ Disable
#define HW_PSR_IRQ_FIQ_DISABLE     0xc0 // IRQ & FIQ Disable

#define HW_PSR_IRQ_DISABLE_SHIFT   7   // HW_PSR_IRQ_DISABLE == (1 << PSR_IRQ_DISABLE_SHIFT)

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
#define HW_PSR_IMPR_ABORT_DISABLE  0x00000100   // Imprecise Abort Disable
#define HW_PSR_DATA_LITTLE_ENDIAN  0x00000000   // Data Little Endian
#define HW_PSR_DATA_BIG_ENDIAN     0x00000200   // Data Big Endian

#define HW_PSR_SIMD_GE0            0x00010000   // SIMD greater-than-or-equal
#define HW_PSR_SIMD_GE1            0x00020000
#define HW_PSR_SIMD_GE2            0x00040000
#define HW_PSR_SIMD_GE3            0x00080000
#define HW_PSR_JAVA_STATE          0x01000000   // JAVA state
#endif // NN_BUILD_CONFIG_CPU_ARM_V7A

#define HW_PSR_Q_FLAG              0x08000000   // Sticky Overflow (Q) flag
#define HW_PSR_V_FLAG              0x10000000   // Overflow flag
#define HW_PSR_C_FLAG              0x20000000   // Carry/Borrow/Extend flag
#define HW_PSR_Z_FLAG              0x40000000   // Zero flag
#define HW_PSR_N_FLAG              0x80000000   // Negative/Less than flag

#define HW_PSR_PL0_MASK            0xFF0FFE20


//----------------------------------------------------------------------
//                  System Control Coprocessor
//----------------------------------------------------------------------

// Register c0.c0.0 : ID Code
// Register c0.c0.1 : Cache Type
// Register c0.c0.2 : TCM Type
// Register c0.c0.3 : TLB Type
// Register c0.c0.5 : CPU ID
#define HW_C0_MPE_ENABLE            0x80000000  // Multiprocessing Extensions
#define HW_C0_UNIPROCESSOR          0x40000000  // Indicates a Uniprocessor system
#define HW_C0_AP_CLUSTER_ID_MASK    0x00000f00  // Cluster ID
#define HW_C0_AP_CPU_ID_MASK        0x00000003  // CPU ID

#define HW_C0_AP_CLUSTER_ID_SFT     8
#define HW_C0_AP_CPU_ID_SFT         0


// Register 2.0.2 : Size of TTBR0 translation table

#define HW_C2_TTBR0_TABLE_SIZE_MASK 0x00000007  // The size of the first-level translation tables accessed by TTBR0
#define HW_C2_TTBR0_TABLE_SIZE_16KB 0x00000000
#define HW_C2_TTBR0_TABLE_SIZE_8KB  0x00000001
#define HW_C2_TTBR0_TABLE_SIZE_4KB  0x00000002
#define HW_C2_TTBR0_TABLE_SIZE_2KB  0x00000003
#define HW_C2_TTBR0_TABLE_SIZE_1KB  0x00000004
#define HW_C2_TTBR0_TABLE_SIZE_512B 0x00000005
#define HW_C2_TTBR0_TABLE_SIZE_256B 0x00000006
#define HW_C2_TTBR0_TABLE_SIZE_128B 0x00000007

#define HW_C2_2_PD0                 0x00000010
#define HW_C2_2_PD1                 0x00000020


// Register 2.0.0/1 : MMU L1 Table Base
#define HW_C2_0_T1_BASE_MASK_MAX    0xffffff80  // Pointer to the level one translation table when Boundary = 0 or VA = 0
#define HW_C2_0_T1_BASE_MASK_MIN    0xffffc000
#define HW_C2_1_T1_BASE_MASK        0xffffc000  //                                            when Boundary > 0 and VA > 0

// Register c1.c0.2 : Coprocessor Access Control Register
#define HW_C1_VFP_AP_PACK( vfp_float, vfp_double ) \
        ( ((vfp_float) << HW_C1_AP_CP10_SFT) \
        | ((vfp_double) << HW_C1_AP_CP11_SFT) )

#define HW_C1_AP_CP10_MASK          0x00300000  // Permit access to CP10
#define HW_C1_AP_CP11_MASK          0x00C00000  //                  CP11

#define HW_C1_AP_CP10_SFT           20
#define HW_C1_AP_CP11_SFT           22

#define HW_C1_AP_NA                 0   // Access denied
#define HW_C1_AP_SVR                1   // Supervisor access only
#define HW_C1_AP_PRIV               3   // Full access


// Register c1.c0.0 : System Control Register
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A15
    #define HW_C1_0_C0_0_VALID_MASK     0x72183c07
    #define HW_C1_0_C0_0_RESET          0x00c50078
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A7
    #define HW_C1_0_C0_0_VALID_MASK     0x72183c07
    #define HW_C1_0_C0_0_RESET          0x00c50078
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A9
    #define HW_C1_0_C0_0_RESET          0x00c50078
#else
    #error not defined NN_BUILD_CONFIG_CPU_
#endif

#define HW_C1_THUMB_EXCEPT          0x40000000  // Thumb Exception enable
#define HW_C1_FORCE_AP_BIT          0x20000000  // AP[0] used as Access Bit
#define HW_C1_TEX_CB_REMAP          0x10000000  // Remap registers are used for remapping
#define HW_C1_NMFI_FIQ              0x08000000  // FIQs behave as NMFIs
#define HW_C1_EXCEPT_LITTLE_ENDIAN  0x00000000  // CPSR E bit is set to 0 on taking an exception
#define HW_C1_EXCEPT_BIG_ENDIAN     0x02000000  // CPSR E bit is set to 1 on taking an exception
#define HW_C1_INTERRUPT_VEC_ENABLE  0x01000000  // Interrupt Vectors Enable bit
#define HW_C1_FAST_INTERRUPT_EBABLE 0x00200000  // Fast Interrupts configuration enable
#define HW_C1_USER_WRITABLE_IS_PXN  0x00100000
#define HW_C1_WRITABLE_IS_XN        0x00080000
#define HW_C1_HW_ACCESS_ENABLE      0x00020000  // Hardware Access Flag Enable
#define HW_C1_ROUND_ROBIN           0x00004000  // Round Robin bit
#define HW_C1_EXCEPT_VEC_UPPER      0x00002000  // High exception vectors selected
#define HW_C1_EXCEPT_VEC_LOWER      0x00000000  // Low exception vectors selected
#define HW_C1_IC_ENABLE             0x00001000  // Instruction Cache enabled
#define HW_C1_BR_PREDICT_ENABLE     0x00000800  // Branch prediction enable
#define HW_C1_SWP_ENABLE            0x00000400  // SWP/SWPB Enable bit
#define HW_C1_DC_ENABLE             0x00000004  // Data Cache enabled
#define HW_C1_ALIGN_FAULT_ENABLE    0x00000002  // Strict alignment fault checking enabled (Higher than UNALIGN)
#define HW_C1_MMU_ENABLE            0x00000001  // MMU enable

#define HW_C1_IC_ENABLE_SFT         12
#define HW_C1_DC_ENABLE_SFT         2



// Register c1.c0.1 : Auxiliary Control
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A15
    #define HW_C1_0_C0_1_VALID_MASK             0xFFFFFFFF
    #define HW_C1_0_C0_1_RESET                  0x00000000

    #define HW_C1_DELAY_EXCLUSIVE               0x80000000
    #define HW_C1_FORCE_MAIN_CLOCK              0x40000000
    #define HW_C1_FORCE_VFP_CLOCK               0x20000000
    #define HW_C1_WRITE_STREAMING_L1L2_12       0x00000000
    #define HW_C1_WRITE_STREAMING_L1L2_128      0x08000000
    #define HW_C1_WRITE_STREAMING_L1L2_512      0x10000000
    #define HW_C1_WRITE_STREAMING_L1L2_DISABLE  0x18000000
    #define HW_C1_WRITE_STREAMING_L1_4          0x00000000
    #define HW_C1_WRITE_STREAMING_L1_64         0x02000000
    #define HW_C1_WRITE_STREAMING_L1_128        0x04000000
    #define HW_C1_WRITE_STREAMING_L1_DISABLE    0x06000000
    #define HW_C1_NON_CACHEABLE_ENHANCEMENT     0x01000000
    #define HW_C1_FORCE_IN_ORDER_TO_LINE        0x00800000
    #define HW_C1_FORCE_IN_ORDER_TO_LOAD        0x00400000
    #define HW_C1_DISABLE_L2_TLB_PREFETCH       0x00200000
    #define HW_C1_DISABLE_STAGE2_TLB            0x00100000
    #define HW_C1_DISABLE_STAGE1_TLB            0x00080000
    #define HW_C1_DISABLE_STAGE1_TLB_PA         0x00040000
    #define HW_C1_DISABLE_TLB_OPT               0x00020000
    #define HW_C1_ENABLE_DEVICE_REPLAY          0x00010000
    #define HW_C1_FORCE_IN_ORDER_ISSUE          0x00008000
    #define HW_C1_FORCE_LIMIT_ONE_INST          0x00004000
    #define HW_C1_FLUSH_CP14_CP15               0x00002000
    #define HW_C1_FORCE_CP14_15_SHADOW          0x00001000
    #define HW_C1_LIMIT_ONE_INST                0x00000800
    #define HW_C1_FORCE_SERIALIZE               0x00000400
    #define HW_C1_DISABLE_FLAG_OPT              0x00000200
    #define HW_C1_WFI_AS_NOP                    0x00000100
    #define HW_C1_WFE_AS_NOP                    0x00000080
    #define HW_C1_SMP                           0x00000040
    #define HW_C1_PLD_AS_NOP                    0x00000020
    #define HW_C1_DISABLE_INDIRECT_PREDICTOR    0x00000010
    #define HW_C1_DISABLE_MICRO_BTB             0x00000008
    #define HW_C1_LIMIT_ONE_LOOP                0x00000004
    #define HW_C1_DISABLE_LOOP_BUFFER           0x00000002
    #define HW_C1_ENABLE_BTB_INVALIDATE         0x00000001

#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A7
    #define HW_C1_0_C0_1_VALID_MASK             0x1000FC40
    #define HW_C1_0_C0_1_RESET                  0x00006000

    #define HW_C1_DDI                           0x10000000
    #define HW_C1_DDVM                          0x00008000
    #define HW_C1_L1PCTL_DIS                    0x00000000
    #define HW_C1_L1PCTL_1                      0x00002000
    #define HW_C1_L1PCTL_2                      0x00004000
    #define HW_C1_L1PCTL_3                      0x00006000
    #define HW_C1_L1RADIS                       0x00001000
    #define HW_C1_L2RADIS                       0x00000800
    #define HW_C1_DODMBS                        0x00000400
    #define HW_C1_SMP                           0x00000040

#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A9
    #define HW_C1_PARITY_CHECK                  0x00000200  // Support for parity checking
    #define HW_C1_ALLOC_1WAY                    0x00000100  // Enable allocation in one cache way only
    #define HW_C1_CACHE_EXCL                    0x00000080  // Exclusive cache
    #define HW_C1_SMP_MODE                      0x00000040  // Symmetric Multi Processing mode
    #define HW_C1_AMP_MODE                      0x00000000  // Asymmetric Multi Processing mode
    #define HW_C1_WRITE_FULL_LINE               0x00000008  // Write full line of zeros mode
    #define HW_C1_L1_PREFETCH_ENABLE            0x00000004  // L1 prefetch enable
    #define HW_C1_L2_PREFETCH_ENABLE            0x00000002  // L2 prefetch enable
    #define HW_C1_HW_BROADCAST                  0x00000001  // Cache and TLB maintenance broadcast enable
    #define HW_C1_0_C0_1_RESET                  0x00000000
#else
    #error not defined NN_BUILD_CONFIG_CPU_
#endif

// Register 2.0.0/1 : MMU L1 Table Base
#define HW_C2_0_T1_BASE_MASK_MAX    0xffffff80  // Pointer to the level one translation table when Boundary = 0 or VA = 0
#define HW_C2_0_T1_BASE_MASK_MIN    0xffffc000
#define HW_C2_1_T1_BASE_MASK        0xffffc000  //                                            when Boundary > 0 and VA > 0
#define HW_C2_WALK_ON_SHARED_MEM    0x00000002  // Page table walk is to shared memory

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A15 || defined NN_BUILD_CONFIG_CPU_CORTEX_A9 || defined NN_BUILD_CONFIG_CPU_CORTEX_A7
    // L2 Cache cachable attributes for page table walking
    #define HW_C2_WALK_OUTER_RGN_PACK(rgn) \
        ((((rgn) & 0x1) << 3) | (((rgn) & 0x2) << 3))

    // L1 Cache cachable attributes for page table walking
    #define HW_C2_WALK_INNER_RGN_PACK(rgn) \
        ((((rgn) & 0x1) << 6) | (((rgn) & 0x2) >> 1))

    #define HW_C2_WALK_RGN_NC           0   // Non-cacheable
    #define HW_C2_WALK_RGN_WB_WA        1   // Write-Back Write-Allocate Cacheable
    #define HW_C2_WALK_RGN_WT           2   // Write-Through Cacheable
    #define HW_C2_WALK_RGN_WB           3   // Write-Back no Write-Allocate Cacheable
#else
    #error not defined NN_BUILD_CONFIG_CPU_
#endif


// Register 3 : Domain Access Control

#define HW_C3_DOMAIN_PACK( d0, d1, d2, d3, d4, d5, d6, d7, \
                           d8, d9, d10, d11, d12, d13, d14, d15 ) \
                                  \
        ( ((d0) << HW_C3_DM0_SFT) \
        | ((d1) << HW_C3_DM1_SFT) \
        | ((d2) << HW_C3_DM2_SFT) \
        | ((d3) << HW_C3_DM3_SFT) \
        | ((d4) << HW_C3_DM4_SFT) \
        | ((d5) << HW_C3_DM5_SFT) \
        | ((d6) << HW_C3_DM6_SFT) \
        | ((d7) << HW_C3_DM7_SFT) \
        | ((d8) << HW_C3_DM8_SFT) \
        | ((d9) << HW_C3_DM9_SFT) \
        | ((d10) << HW_C3_DM10_SFT) \
        | ((d11) << HW_C3_DM11_SFT) \
        | ((d12) << HW_C3_DM12_SFT) \
        | ((d13) << HW_C3_DM13_SFT) \
        | ((d14) << HW_C3_DM14_SFT) \
        | ((d15) << HW_C3_DM15_SFT) )

#define HW_C3_DOMAIN_ID_SFT( no )  ( (no) * 2 )

#define HW_C3_DM0_SFT               0   // Domain 0
#define HW_C3_DM1_SFT               2   //        1
#define HW_C3_DM2_SFT               4   //        2
#define HW_C3_DM3_SFT               6   //        3
#define HW_C3_DM4_SFT               8   //        4
#define HW_C3_DM5_SFT               10  //        5
#define HW_C3_DM6_SFT               12  //        6
#define HW_C3_DM7_SFT               14  //        7
#define HW_C3_DM8_SFT               16  //        8
#define HW_C3_DM9_SFT               18  //        9
#define HW_C3_DM10_SFT              20  //        10
#define HW_C3_DM11_SFT              22  //        11
#define HW_C3_DM12_SFT              24  //        12
#define HW_C3_DM13_SFT              26  //        13
#define HW_C3_DM14_SFT              28  //        14
#define HW_C3_DM15_SFT              30  //        15

#define HW_C3_DM_AP_NA              0   // Any access generates a domain fault
#define HW_C3_DM_AP_CLIENT          1   // Permit clients to access a domain
#define HW_C3_DM_AP_MANAGER         3   // Permit managers to access a domain


//----------------------------------------------------------------------
//                  VFP Control Registers
//----------------------------------------------------------------------

// Floating-Point Exception Register

#define HW_FPEXC_EXCEPTION_STATE    0x80000000  // in exception state
#define HW_FPEXC_VFP_ENABLE         0x40000000  // VFP enable

#define HW_FPEXC_VEC_ITR_MASK       0x00000700  // Vector iteration count

#define HW_FPEXC_INPUT_EXCEPTION    0x00000080  // Input exception flag
#define HW_FPEXC_POTENT_UNDERFLOW   0x00000008  // Potential underflow flag
#define HW_FPEXC_POTENT_OVERFLOW    0x00000004  // Potential overflow flag
#define HW_FPEXC_POTENT_INVALID     0x00000001  // Potential invalid operation flag

#define HW_FPEXC_VEC_ITR_SFT        8


// Floating-Point Status and Control Register

#define HW_FPSCR_N_FLAG             0x80000000  // Negative/Less than flag
#define HW_FPSCR_Z_FLAG             0x40000000  // Zero flag
#define HW_FPSCR_C_FLAG             0x20000000  // Carry/Borrow/Extend flag
#define HW_FPSCR_V_FLAG             0x10000000  // Overflow flag

#define HW_FPSCR_DEFAULT_NAN_MODE   0x02000000  // Default NaN mode (for std mode & fast mode)
#define HW_FPSCR_FLUSH_TO_ZERO_MODE 0x01000000  // Flush-to-zero mode (for std mode & fast mode)
#define HW_FPSCR_ROUND_MODE_MASK    0x00c00000  // Rounding mode
#define HW_FPSCR_VEC_STRIDE_MASK    0x00300000  // Vector stride
#define HW_FPSCR_VEC_LENGTH_MASK    0x00070000  // Vector length

#define HW_FPSCR_IDE_ENABLE         0x00008000  // Input Subnormal exception enable
#define HW_FPSCR_IXE_ENABLE         0x00001000  // Inexact exception enable
#define HW_FPSCR_UFE_ENABLE         0x00000800  // Underflow exception enable
#define HW_FPSCR_OFE_ENABLE         0x00000400  // Overflow exception enable
#define HW_FPSCR_DZE_ENABLE         0x00000200  // Division by Zero exception enable
#define HW_FPSCR_IOE_ENABLE         0x00000100  // Invalid Operation exception enable

#define HW_FPSCR_IDC                0x00000080  // Input Subnormal cumulative flag
#define HW_FPSCR_IXC                0x00000010  // Inexact cumulative flag
#define HW_FPSCR_UFC                0x00000008  // Underflow cumulative flag
#define HW_FPSCR_OFC                0x00000004  // Overflow cumulative flag
#define HW_FPSCR_DZC                0x00000002  // Division by Zero cumulative flag
#define HW_FPSCR_IOC                0x00000001  // Invalid Operation cumulative flag

#define HW_FPSCR_ROUND_MODE_SFT     22
#define HW_FPSCR_VEC_STRIDE_SFT     20
#define HW_FPSCR_VEC_LENGTH_SFT     16

#define HW_FPSCR_ROUND_NEAR_MODE    0           // Round to nearest mode
#define HW_FPSCR_ROUND_PLUS_MODE    1           // Round towards plus infinity mode
#define HW_FPSCR_ROUND_MINUS_MODE   2           // Round towards minus infinity mode
#define HW_FPSCR_ROUND_ZERO_MODE    3           // Round towards zero mode

#define HW_FPSCR_VEC_STRIDE_1       0           // Vector stride 1
#define HW_FPSCR_VEC_STRIDE_2       3           // Vector stride 2

#define HW_FPSCR_VEC_LENGTH_1       0           // Vector length 1
#define HW_FPSCR_VEC_LENGTH_2       1           // Vector length 2
#define HW_FPSCR_VEC_LENGTH_3       2           // Vector length 3
#define HW_FPSCR_VEC_LENGTH_4       3           // Vector length 4
#define HW_FPSCR_VEC_LENGTH_5       4           // Vector length 5
#define HW_FPSCR_VEC_LENGTH_6       5           // Vector length 6
#define HW_FPSCR_VEC_LENGTH_7       6           // Vector length 7
#define HW_FPSCR_VEC_LENGTH_8       7           // Vector length 8

#define HW_CNTKCTL_PL0PTEN              (1<<9)
#define HW_CNTKCTL_PL0VTEN              (1<<8)
#define HW_CNTKCTL_EVNTI_0              (0<<4)
#define HW_CNTKCTL_EVNTI_1              (1<<4)
#define HW_CNTKCTL_EVNTI_2              (2<<4)
#define HW_CNTKCTL_EVNTI_3              (3<<4)
#define HW_CNTKCTL_EVNTI_4              (4<<4)
#define HW_CNTKCTL_EVNTI_5              (5<<4)
#define HW_CNTKCTL_EVNTI_6              (6<<4)
#define HW_CNTKCTL_EVNTI_7              (7<<4)
#define HW_CNTKCTL_EVNTI_8              (8<<4)
#define HW_CNTKCTL_EVNTI_9              (9<<4)
#define HW_CNTKCTL_EVNTI_10             (10<<4)
#define HW_CNTKCTL_EVNTI_11             (11<<4)
#define HW_CNTKCTL_EVNTI_12             (12<<4)
#define HW_CNTKCTL_EVNTI_13             (13<<4)
#define HW_CNTKCTL_EVNTI_14             (14<<4)
#define HW_CNTKCTL_EVNTI_15             (15<<4)
#define HW_CNTKCTL_EVNTDIR              (1<<3)
#define HW_CNTKCTL_EVNTEN               (1<<2)
#define HW_CNTKCTL_PL0VCTEN             (1<<1)
#define HW_CNTKCTL_PL0PCTEN             (1<<0)

#define HW_PMXEVTYPER(el0, el1, type)   static_cast<uint32_t>(((!(el1) << 31)) | ((!(el0) << 30)) | (((type) & 0x3FF)))

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A15 || defined NN_BUILD_CONFIG_CPU_CORTEX_A7
#define HW_PM_TYPE_INST                 0x08
#define HW_PM_TYPE_CPU_CYCLES           0x11
#define HW_PM_TYPE_CACHE_REFILL         0x17

#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A9
#define HW_PM_TYPE_INST                 0x68
#define HW_PM_TYPE_CPU_CYCLES           0x11
#define HW_PM_TYPE_CACHE_REFILL         0x50
#endif

