﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Fpu.h>
#include <nn/nn_Common.h>
#include "../../kern_Result.h"
#include "../../kern_KContextBase.h"
#include "../../kern_KTaggedAddress.h"
#include <nn/nn_BitTypes.h>

namespace nn { namespace kern {

    class KThread;

    namespace ARM {

class KContext : public KContextBase
{
    friend class KContextICCHelper;

public:
    Bit32     m_CpuRegisters[8];
    Bit32     m_Sp;
    Bit32     m_Lr;
#if     NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS == 16
    Bit64     m_FpuRegisters[16];
#elif   NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS == 32
    Bit64     m_FpuRegisters[32];
#endif
#if     defined(NN_BUILD_CONFIG_FPU_VFP)
public:
    Bit32     m_Fpexc;
    Bit32     m_Fpscr;
#endif
    bool      m_Lock;

public:
    explicit KContext(){}

    Result Initialize(KVirtualAddress initialPc, KVirtualAddress kernelSp, KVirtualAddress userSp, uintptr_t firstArg, bool isUserMode, bool is64Bit, bool isMain);
    Result Finalize();
    void SetFirstArgument(uintptr_t firstArg, uintptr_t secondArg);

#if     defined(NN_BUILD_CONFIG_FPU_VFP)
    const size_t GetNumOfVfpRegisters() const
    {
        return sizeof(m_FpuRegisters) / sizeof(*m_FpuRegisters);
    }
    const Bit64*  GetVfpRegisters() const;
    Bit32   GetFpexc() const;
    Bit32   GetFpscr() const;
    static void SaveFpuContext(KThread* pThread);
    static void SaveCurrentFpuContext();

    void SetVfpRegisters(const Bit64* x);
    void SetFpexc(Bit32 fpexc);
    void SetFpscr(Bit32 fpscr);
    void CloneFpuStatus();

    static void VfpSwitchHandler(KThread* pNext);
    static void OnThreadCoreChanged(KThread* pThread);
    static void OnThreadTerminating(const KThread* pThread);
    static void BindInterrupt(int32_t coreNo);
#else
    static void OnThreadTerminating(const KThread* pThread) { NN_UNUSED(pThread); }
#endif

    const Bit32* GetCpuRegisters() const;
    void SetCpuRegister(uint32_t index, uint32_t val);

    static void Restore(const KContext& pContext);

private:
#if     defined(NN_BUILD_CONFIG_FPU_VFP)
    static void RestoreVfp(const KContext& pContext);
    static void UserModeThreadStarter();
    static void SupervisorModeThreadStarter();
#endif
};

    }
}}

