﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../kern_DebugSelect.h"
#include "../kern_KProcess.h"
#include "../kern_KThread.h"
#include "../kern_KPort.h"
#include "../kern_KEvent.h"
#include "../kern_KInterruptEvent.h"
#include "../kern_KPageGroup.h"
#include "../kern_KMemoryBlock.h"
#include "../kern_KSharedMemory.h"
#include "../kern_KTransferMemory.h"
#include "../kern_KCodeMemory.h"
#include "../kern_KDeviceAddressSpace.h"
#include "../kern_KSession.h"
#include "../kern_KLightSession.h"
#include "../kern_KObjectName.h"
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_DebugString.h"
#include "../kern_KSessionRequest.h"
#include "../kern_KResourceLimit.h"
#include "../kern_KSharedMemoryInfo.h"
#include "../kern_KPageBuffer.h"

#include "../kern_KMemoryLayout.h"
#include "../kern_Utility.h"
#include "kern_SystemControl.h"

namespace nn { namespace kern { namespace init {

template <typename T>
KProcessAddress InitializeAllocator(KProcessAddress a, size_t numObj)
{
    const size_t size = RoundUp(sizeof(T) * numObj, NN_ALIGNOF(uintptr_t));
    KProcessAddress begin = RoundUp(a, NN_ALIGNOF(T));

    if (size > 0)
    {
        NN_KERN_ABORT_UNLESS(begin + size - 1 <= (KMemoryLayout::GetSlabRegionEnd() - 1));
        T::InitializeAllocator(GetUntypedPointer(begin), size);
    }

    return begin + size;
}

template <typename T>
KProcessAddress InitializeSlab(KProcessAddress a, size_t numObj)
{
    const size_t size = RoundUp(sizeof(T) * numObj, NN_ALIGNOF(uintptr_t));
    KProcessAddress begin = RoundUp(a, NN_ALIGNOF(T));

    if (size > 0)
    {
        NN_KERN_ABORT_UNLESS(begin + size - 1 <= (KMemoryLayout::GetSlabRegionEnd() - 1));
        T::InitializeSlab(GetUntypedPointer(begin), size);
    }

    return begin + size;
}

#define NN_KERNEL_KERN_INITIALIZE_SLAB(a,b,c)                               \
    do                                                                      \
    {                                                                       \
        (c) = InitializeSlab<a>((c), NN_KERN_SLAB_OBJ_NUM_ ## b);           \
        NN_KERN_RELEASE_LOG("%-40s%7ld Byte  %p-%p\n", #a,                  \
                a::GetObjSize() * a::GetSlabSize(),                         \
                a::GetSlabAddr(),                                           \
                a::GetSlabAddr() + a::GetObjSize() * a::GetSlabSize() - 1); \
    }                                                                       \
    while (NN_STATIC_CONDITION(false))

#define NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(a,b,c)                          \
    do                                                                      \
    {                                                                       \
        (c) = InitializeAllocator<a>((c), NN_KERN_SLAB_OBJ_NUM_ ## b);      \
        NN_KERN_RELEASE_LOG("%-40s%7ld Byte  %p-%p\n", #a,                  \
                a::GetObjSize() * a::GetSlabSize(),                         \
                a::GetSlabAddr(),                                           \
                a::GetSlabAddr() + a::GetObjSize() * a::GetSlabSize() - 1); \
    }                                                                       \
    while (NN_STATIC_CONDITION(false))

#define NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(a,b,c)                                            \
    do                                                                                      \
    {                                                                                       \
        (c) += RoundUp(sizeof(a) * (NN_KERN_SLAB_OBJ_NUM_ ## b), NN_ALIGNOF(uintptr_t));    \
        (c) += NN_ALIGNOF(a);                                                               \
    }                                                                                       \
    while (NN_STATIC_CONDITION(false))

inline size_t CalcAllocatorsSize()
{
    size_t size = 0;
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KPageBuffer, PAGE_BUFFER, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KProcess, PROCESS, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KThread, THREAD, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KLinkedListNode, LINKED_LIST_NODE, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KEvent, EVENT, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KInterruptEvent, INTERRUPT_EVENT, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KInterruptEventTask, INTERRUPT_EVENT_TASK, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KPort, PORT, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KSharedMemory, SHARED_MEMORY, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KSharedMemoryInfo, SHARED_MEMORY_INFO, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KTransferMemory, TRANSFER_MEMORY, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KCodeMemory, CODE_MEMORY, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KDeviceAddressSpace, DEVICE_ADDRESS_SPACE, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KSession, SESSION, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KSessionRequest, SESSION_REQUEST, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KLightSession, LIGHT_SESSION, size);
#if NN_KERN_HAS_MMU
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KThreadLocalPage, THREAD_LOCAL_PAGE, size);
#endif  // if NN_KERN_HAS_MMU
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KObjectName, OBJECT_NAME, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KResourceLimit, RESOURCE_LIMIT, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KEventInfo, DEBUG_EVENT_OBJECT, size);
    NN_KERNEL_KERN_SUM_ALLOCATOR_SIZE(KDebug, DEBUG, size);

    size += NN_KERN_P_ADDR_SLAB_RESERVED_SIZE;

    return size;
}

inline void InitializeAllocators(size_t* pSize)
{
    KProcessAddress a = KMemoryLayout::GetSlabRegionBegin();

    enum SlabType
    {
        KPageBufferType,
        KProcessType,
        KThreadType,
        KLinkedListNodeType,
        KEventType,
        KInterruptEventType,
        KInterruptEventTaskType,
        KPortType,
        KSharedMemoryType,
        KSharedMemoryInfoType,
        KTransferMemoryType,
        KCodeMemoryType,
        KDeviceAddressSpaceType,
        KSessionType,
        KSessionRequestType,
        KLightSessionType,
        KThreadLocalPageType,
        KObjectNameType,
        KResourceLimitType,
        KEventInfoType,
        KDebugType,

        SlabTypeNum,
    };

    SlabType slabTypeArray[SlabTypeNum];
    for (int i = 0; i < SlabTypeNum; i++)
    {
        slabTypeArray[i] = static_cast<SlabType>(i);
    }
    for (int i = 0; i < SlabTypeNum; i++)
    {
        SlabType type = slabTypeArray[i];
        int r = KSystemControl::GetRandomValue(0, SlabTypeNum - 1);
        slabTypeArray[i] = slabTypeArray[r];
        slabTypeArray[r] = type;
    }

    size_t gapSizes[SlabTypeNum];
    const size_t reservedForGaps = NN_KERN_P_ADDR_SLAB_RESERVED_SIZE;

    {
        for (int i = 0; i < SlabTypeNum; i++)
        {
            gapSizes[i] = KSystemControl::GetRandomValue(0, reservedForGaps);
        }

        // Sort by increasing order
        for (int i = 0; i < SlabTypeNum; i++)
        {
            for (int j = i + 1; j < SlabTypeNum; j++)
            {
                if (gapSizes[i] > gapSizes[j])
                {
                    std::swap(gapSizes[i], gapSizes[j]);
                }
            }
        }
    }

    for (int i = 0; i < SlabTypeNum; i++)
    {
        a += i == 0 ? gapSizes[0] : gapSizes[i] - gapSizes[i - 1];

        SlabType type = slabTypeArray[i];
        switch (type)
        {
        case KPageBufferType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KPageBuffer, PAGE_BUFFER, a);
            }
            break;
        case KProcessType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KProcess, PROCESS, a);
            }
            break;
        case KThreadType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KThread, THREAD, a);
            }
            break;
        case KLinkedListNodeType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KLinkedListNode, LINKED_LIST_NODE, a);
            }
            break;
        case KEventType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KEvent, EVENT, a);
            }
            break;
        case KInterruptEventType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KInterruptEvent, INTERRUPT_EVENT, a);
            }
            break;
        case KInterruptEventTaskType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KInterruptEventTask, INTERRUPT_EVENT_TASK, a);
            }
            break;
        case KPortType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KPort, PORT, a);
            }
            break;
        case KSharedMemoryType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KSharedMemory, SHARED_MEMORY, a);
            }
            break;
        case KSharedMemoryInfoType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KSharedMemoryInfo, SHARED_MEMORY_INFO, a);
            }
            break;
        case KTransferMemoryType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KTransferMemory, TRANSFER_MEMORY, a);
            }
            break;
        case KCodeMemoryType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KCodeMemory, CODE_MEMORY, a);
            }
            break;
        case KDeviceAddressSpaceType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KDeviceAddressSpace, DEVICE_ADDRESS_SPACE, a);
            }
            break;
        case KSessionType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KSession, SESSION, a);
            }
            break;
        case KSessionRequestType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KSessionRequest, SESSION_REQUEST, a);
            }
            break;
        case KLightSessionType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KLightSession, LIGHT_SESSION, a);
            }
            break;
        case KThreadLocalPageType:
            {
#if NN_KERN_HAS_MMU
                NN_KERNEL_KERN_INITIALIZE_SLAB(KThreadLocalPage, THREAD_LOCAL_PAGE, a);
#endif  // if NN_KERN_HAS_MMU
            }
            break;
        case KObjectNameType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KObjectName, OBJECT_NAME, a);
            }
            break;
        case KResourceLimitType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KResourceLimit, RESOURCE_LIMIT, a);
            }
            break;
        case KEventInfoType:
            {
                NN_KERNEL_KERN_INITIALIZE_SLAB(KEventInfo, DEBUG_EVENT_OBJECT, a);
            }
            break;
        case KDebugType:
            {
                NN_KERNEL_KERN_INITIALIZE_ALLOCATOR(KDebug, DEBUG, a);
            }
            break;
        default:
            {
                NN_KERN_ABORT();
            }
            break;
        }
    }

    NN_KERN_RELEASE_LOG("   Slab allocate %6ld / %6ld\n", a - KMemoryLayout::GetSlabRegionBegin(), KMemoryLayout::GetSlabRegionSize());

    *pSize = a - KMemoryLayout::GetSlabRegionBegin();
}

}}}
