﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_Platform.h"
#include "kern_SystemControl.h"
#include "../../kern_Result.h"
#include "../../kern_Kernel.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_Main.h"
#include "../../kern_KProcess.h"
#include "kern_MemoryMap.h"
#include "../../kern_PageTableSelect.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../../kern_KLightMutex.h"
#include "../../kern_InitialProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_KTrace.h"
#include "../../kern_KSimpleLock.h"
#include "../../kern_Trace.h"
#include "../../kern_Utility.h"
#include "../../kern_KMemoryLayout.h"
#include "../../kern_KHeapArrange.h"

#include <cstring>
#include <random>
#include "kern_SecureMonitorCall.h"
#include "kern_SleepManager.h"

namespace nn { namespace kern {
namespace {
const size_t SecureRegionAlign = 0x00020000;
bool g_SecureRegionUsed = false;
KPhysicalAddress g_SecureRegionAddr = Null<KPhysicalAddress>();
size_t g_SecureRegionSize = 0;

const size_t AppletSecureMemorySize = 0x400000;
KSimpleLock g_AppletSecureMemoryLock;
bool g_AppletSecureMemoryUsed = false;
KVirtualAddress g_AppletSecureMemoryAddr = Null<KVirtualAddress>();

std::mt19937 g_Random;
KSimpleLock  g_RandomLock;
bool g_RandInitialized = false;
bool g_CallShowErrorOnPanic = false;

const struct
{
    KPhysicalAddress addr;
    size_t size;
}
g_MappableBlackList[] = {
    { NN_KERN_P_ADDR_MC_REGISTER,       NN_KERN_P_ADDR_MC_REGISTER_SIZE},
    { NN_KERN_P_ADDR_MC0_REGISTER,      NN_KERN_P_ADDR_MC0_REGISTER_SIZE},
    { NN_KERN_P_ADDR_MC1_REGISTER,      NN_KERN_P_ADDR_MC1_REGISTER_SIZE},
    { NN_KERN_P_ADDR_RTC_PMC_REGISTER,  NN_KERN_P_ADDR_RTC_PMC_REGISTER_SIZE},
    { 0x50040000,                       0x00020000},    // ARM registers (GIC, BPMP-Lite Cache)
    { 0x6000f000,                       0x00001000},    // Exception vectors
    { 0x6001dc00,                       0x00000400},    // IPATCH
};

// アクセス可能なMCレジスタのビット
const Bit8 g_McAccessible[] =
{
    0x00, 0x00, 0x20, 0x00, 0xF0, 0xFF, 0xF7, 0x01, // 0x000 - 0x0fc (0x000 - 0x01c, 0x020 - 0x03c, ...)
    0xCD, 0xFE, 0xC0, 0xFE, 0x00, 0x00, 0x00, 0x00, // 0x100 - 0x1fc
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6E, // 0x200 - 0x2fc
    0x30, 0x05, 0x06, 0xB0, 0x71, 0xC8, 0x43, 0x04, // 0x300 - 0x3fc
    0x80, 0xFF, 0x08, 0x80, 0x03, 0x38, 0x8E, 0x1F, // 0x400 - 0x4fc
    0xC8, 0xFF, 0xFF, 0x00, 0x0E, 0x00, 0x00, 0x00, // 0x500 - 0x5fc
    0xF0, 0x1F, 0x00, 0x30, 0xF0, 0x03, 0x03, 0x30, // 0x600 - 0x6fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x700 - 0x7fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x800 - 0x8fc
    0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x00, // 0x900 - 0x9fc
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xa00 - 0xafc
    0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0xFE, 0x0F, // 0xb00 - 0xbfc
    0x01, 0x00, 0x80, 0x00, 0x00, 0x08, 0x00, 0x00, // 0xc00 - 0xcfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xd00 - 0xdfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xe00 - 0xefc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  // 0xf00 - 0xffc
};

const Bit8 g_McAccessibleKernel[] =
{
    0x9F, 0x31, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x000 - 0x0fc (0x000 - 0x01c, 0x020 - 0x03c, ...)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x100 - 0x1fc
    0x00, 0xC0, 0x73, 0x3E, 0x6F, 0x00, 0x00, 0x00, // 0x200 - 0x2fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x300 - 0x3fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x400 - 0x4fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x500 - 0x5fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x600 - 0x6fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x700 - 0x7fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x800 - 0x8fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, // 0x900 - 0x9fc
    0x00, 0x00, 0x00, 0x00, 0xE4, 0xFF, 0xFF, 0x01, // 0xa00 - 0xafc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xb00 - 0xbfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xc00 - 0xcfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xd00 - 0xdfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xe00 - 0xefc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  // 0xf00 - 0xffc
};

bool IsSecureReadWriteRegister(nn::svc::PhysicalAddress physAddress)
{
    if (NN_KERN_P_ADDR_PMC_REGISTER <= physAddress && physAddress < NN_KERN_P_ADDR_PMC_REGISTER_END)
    {
        return true;
    }
    if (NN_KERN_P_ADDR_MC_REGISTER <= physAddress && physAddress < NN_KERN_P_ADDR_MC_REGISTER_END)
    {
        size_t offset = physAddress - NN_KERN_P_ADDR_MC_REGISTER;
        if (NN_LIKELY(g_McAccessible[(offset / 4) / 8] & (1u << ((offset / 4) % 8))))
        {
            return true;
        }
    }
    if (NN_KERN_P_ADDR_MC0_REGISTER <= physAddress && physAddress < NN_KERN_P_ADDR_MC0_REGISTER_END)
    {
        size_t offset = physAddress - NN_KERN_P_ADDR_MC0_REGISTER;
        if (NN_LIKELY(g_McAccessible[(offset / 4) / 8] & (1u << ((offset / 4) % 8))))
        {
            return true;
        }
    }
    if (NN_KERN_P_ADDR_MC1_REGISTER <= physAddress && physAddress < NN_KERN_P_ADDR_MC1_REGISTER_END)
    {
        size_t offset = physAddress - NN_KERN_P_ADDR_MC1_REGISTER;
        if (NN_LIKELY(g_McAccessible[(offset / 4) / 8] & (1u << ((offset / 4) % 8))))
        {
            return true;
        }
    }
    return false;
}

bool IsSecureReadWriteRegisterForKernel(nn::svc::PhysicalAddress physAddress)
{
    if (NN_KERN_P_ADDR_MC_REGISTER <= physAddress && physAddress < NN_KERN_P_ADDR_MC_REGISTER_END)
    {
        size_t offset = physAddress - NN_KERN_P_ADDR_MC_REGISTER;
        if (NN_LIKELY(g_McAccessibleKernel[(offset / 4) / 8] & (1u << ((offset / 4) % 8))))
        {
            return true;
        }
    }
    return false;
}

volatile Bit32* GetRegisterAddess(nn::svc::PhysicalAddress physAddress)
{
    NN_UNUSED(physAddress);
    return nullptr;
}

bool SetSecureRegion(KPhysicalAddress phys, size_t size)
{
    static KSimpleLock g_SmcLock;

    if (GetAsInteger(phys) & (SecureRegionAlign - 1))
    {
        return false;
    }
    if (size & (SecureRegionAlign - 1))
    {
        return false;
    }

    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_SmcLock);

    if (size == 0)
    {
        if (!g_SecureRegionUsed)
        {
            return false;
        }
        if (phys != g_SecureRegionAddr)
        {
            return false;
        }
        g_SecureRegionUsed = false;
        g_SecureRegionAddr = Null<KPhysicalAddress>();
        g_SecureRegionSize = 0;
        NX::smc::SetKernelCarveoutRegion(1, GetAsInteger(phys), size);
    }
    else
    {
        if (g_SecureRegionUsed)
        {
            return false;
        }
        g_SecureRegionUsed = true;
        g_SecureRegionAddr = phys;
        g_SecureRegionSize = size;
        NX::smc::SetKernelCarveoutRegion(1, GetAsInteger(phys), size);
    }
    return true;
}

uint64_t GetApplicationSize()
{
    NX::smc::ConfigItemData config = {};
    NX::smc::GetConfig(&config, NX::smc::ConfigItem_MemoryMode);
    Bit64 memoryMode = config.data[0];
    const Bit64 memoryArrangeMask = 0x3F;
    switch (memoryMode & memoryArrangeMask)
    {
    case 0x01: return 3285ll * 1024 * 1024; // 4G
    case 0x02: return 2048ll * 1024 * 1024; // 4GInternal;
    case 0x03: return 3285ll * 1024 * 1024; // 4GSdk
    case 0x11: return 4916ll * 1024 * 1024; // 6G
    case 0x12: return 3285ll * 1024 * 1024; // 6GInternal
    case 0x21: return 4916ll * 1024 * 1024; // 8GInternal
    default:   return 3285ll * 1024 * 1024; // 未知は 4G
    }
}

uint64_t GetAppletSize()
{
    NX::smc::ConfigItemData config = {};
    NX::smc::GetConfig(&config, NX::smc::ConfigItem_MemoryMode);
    Bit64 memoryMode = config.data[0];
    const Bit64 memoryArrangeMask = 0x3F;
    switch (memoryMode & memoryArrangeMask)
    {
    case 0x01: return  507ll * 1024 * 1024; // 4G
    case 0x02: return 1554ll * 1024 * 1024; // 4GInternal;
    case 0x03: return  448ll * 1024 * 1024; // 4GSdk
    case 0x11: return  562ll * 1024 * 1024; // 6G
    case 0x12: return 2193ll * 1024 * 1024; // 6GInternal
    case 0x21: return 2193ll * 1024 * 1024; // 8GInternal
    default:   return  507ll * 1024 * 1024; // 未知は 4G
    }
}

uint64_t GetPhysicalMemorySize()
{
    NX::smc::ConfigItemData config = {};
    NX::smc::GetConfig(&config, NX::smc::ConfigItem_KernelFlags);
    nn::util::BitPack32 kernelFlags = { static_cast<Bit32>(config.data[0]) };
    switch (kernelFlags.Get<NX::smc::KernelFlags::MemorySize>())
    {
    case NX::smc::MemorySize_4G: return 0x100000000ul;
    case NX::smc::MemorySize_6G: return 0x180000000ul;
    case NX::smc::MemorySize_8G: return 0x200000000ul;
    default:                 return 0x100000000ul;
    }
}

uint64_t GetRealPhysicalMemorySize()
{
#define NN_MC_EMEM_CFG0_OFFSET (0x50)
    Bit32 ememCfg0;
    KSystemControl::ReadWriteRegisterFromKernel(&ememCfg0, (NN_KERN_P_ADDR_MC_REGISTER + NN_MC_EMEM_CFG0_OFFSET), 0, 0);
    return (ememCfg0 & 0x3FFF) * 0x00100000ul;
}

size_t GetHeapSize(size_t physMemorySize, size_t realMemorySize)
{
    NN_KERN_ABORT_UNLESS(KMemoryLayout::GetHeapRegionSize() + physMemorySize >= realMemorySize);
    return KMemoryLayout::GetHeapRegionSize() + (physMemorySize - realMemorySize);
}

void InitializeAppletSecureMemory()
{
    static_assert(AppletSecureMemorySize % SecureRegionAlign == 0, "");

    NN_KERN_ABORT_UNLESS(g_AppletSecureMemoryAddr == Null<KVirtualAddress>());

    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, AppletSecureMemorySize));

    Bit32 allocateOption = KMemoryManager::MakeAllocateOption(KMemoryManager::Region_SecureSystem, KMemoryManager::From_Front);
    g_AppletSecureMemoryAddr = Kernel::GetKernelHeapManager().AllocateContinuous(AppletSecureMemorySize / KMemoryManager::PageSize, 1, allocateOption);
    if (g_AppletSecureMemoryAddr == Null<KVirtualAddress>())
    {
        NN_KERNEL_PANIC("Unable to allocate applet secure memory\n");
    }

    Kernel::GetKernelHeapManager().Open(g_AppletSecureMemoryAddr, AppletSecureMemorySize / KMemoryManager::PageSize);
}

Result AllocateSecureMemoryForApplet(KVirtualAddress* pAddr, size_t size)
{
    if ((size & (KMemoryManager::PageSize - 1)) != 0)
    {
        return nn::svc::ResultInvalidSize();
    }

    if (size > AppletSecureMemorySize)
    {
        return nn::svc::ResultOutOfMemory();
    }

    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_AppletSecureMemoryLock);

    NN_KERN_ABORT_UNLESS(g_AppletSecureMemoryAddr != Null<KVirtualAddress>());

    if (g_AppletSecureMemoryUsed)
    {
        return nn::svc::ResultOutOfMemory();
    }

    g_AppletSecureMemoryUsed = true;

    *pAddr = g_AppletSecureMemoryAddr;

    return ResultSuccess();
}

void FreeSecureMemoryForApplet(KVirtualAddress address, size_t size)
{
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_AppletSecureMemoryLock);

    NN_KERN_ABORT_UNLESS(address == g_AppletSecureMemoryAddr);
    NN_KERN_ABORT_UNLESS(size <= AppletSecureMemorySize);
    NN_KERN_ABORT_UNLESS((size & (KMemoryManager::PageSize - 1)) == 0);
    NN_KERN_ABORT_UNLESS(g_AppletSecureMemoryUsed);

    g_AppletSecureMemoryUsed = false;
}
}

uint64_t KSystemControl::GetRandomValue(uint64_t randMin, uint64_t randMax)
{
    std::uniform_int_distribution<uint64_t> rand(randMin, randMax);
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_RandomLock);
    if (NN_UNLIKELY(!g_RandInitialized))
    {
        Bit64 seed;
        GenerateRandom(&seed, 1);
        g_Random.seed(static_cast<uint64_t>(seed));
        g_RandInitialized = true;
    }
    return rand(g_Random);
}

void KSystemControl::GenerateRandom(Bit64* pBuffer, size_t num)
{
    NN_KERN_ABORT_UNLESS(0 < num && num <= 7);
    NX::smc::GenerateRandomBytes(pBuffer, num * sizeof(Bit64));
}

void KSystemControl::GetHeapArrange(KHeapArrange* p)
{
    size_t realMemorySize = GetRealPhysicalMemorySize();
    size_t physMemorySize = GetPhysicalMemorySize();
    NN_KERN_ABORT_UNLESS(physMemorySize <= realMemorySize);
    bool useFirstHalf = (physMemorySize * 2 <= realMemorySize);

    size_t managementSize = KMemoryManager::CalcManagementAreaSize(GetHeapSize(physMemorySize, realMemorySize));
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);

    NN_KERN_ABORT_UNLESS(GetApplicationSize() > KMemoryLayout::GetDebugRegionPhysicalSize());
    p->applicationSize = GetApplicationSize() - KMemoryLayout::GetDebugRegionPhysicalSize();
    p->appletSize = GetAppletSize();

    // 最小限必要な NonSecure system のサイズ
    size_t nonSecureSystemMinSize = NN_SVC_NONSECURE_SYSTEM_MEMORY_SIZE;
    // NX では 非セキュアな初期プログラムをアプリ領域で動かす。
    // 非セキュアな初期プログラムは boot プロセスだけなので、アプリが起動する前には終了する前提。
    NN_KERN_ABORT_UNLESS(GetInitialProcessesNonSecureMemorySize() <= p->applicationSize);

    // 最小限必要な NonSecure system のサイズから secureRegionEnd を計算
    KVirtualAddress nonSecureEnd  = KMemoryLayout::GetHeapRegionEnd() - (useFirstHalf? (realMemorySize - physMemorySize): ((realMemorySize - physMemorySize) / 2));
    KVirtualAddress nonSecureAddr = nonSecureEnd - p->applicationSize - p->appletSize - nonSecureSystemMinSize;
    KVirtualAddress secureRegionEnd = RoundDown(nonSecureAddr - (useFirstHalf? 0: ((realMemorySize - physMemorySize) / 2)), SecureRegionAlign);

    NN_KERN_ABORT_UNLESS(KMemoryLayout::GetHeapRegionBegin() <= nonSecureAddr   && nonSecureAddr   <= KMemoryLayout::GetHeapRegionEnd() - 1);
    NN_KERN_ABORT_UNLESS(KMemoryLayout::GetHeapRegionBegin() <= secureRegionEnd && secureRegionEnd <= KMemoryLayout::GetHeapRegionEnd() - 1);

    // secureRegionEnd が carveout 設定可能領域から出ないよう調整
    size_t          carveoutRegionMaxSize = (512 * 1024 * 1024) - (128 * 1024);
    KVirtualAddress carveoutRegionMaxEnd  = RoundDown(KMemoryLayout::ToLinearVirtualAddress(KMemoryLayout::GetKernelRegionPhysicalBegin()) + carveoutRegionMaxSize, SecureRegionAlign);
    if (secureRegionEnd > carveoutRegionMaxEnd)
    {
        secureRegionEnd = carveoutRegionMaxEnd;
    }

    KVirtualAddress mainMemoryVirtualAddr = KMemoryLayout::ToLinearVirtualAddress(NN_KERN_P_ADDR_MAIN_MEMORY);
    KVirtualAddress backAddr = mainMemoryVirtualAddr + realMemorySize / 2;

    // secure と non-seucre が連続しないように配置する
    p->managementAddr = secureRegionEnd - managementSize;
    p->managementSize = managementSize;
    p->secureSystemAddr = KMemoryLayout::GetHeapRegionBegin();
    p->secureSystemSize = p->managementAddr - p->secureSystemAddr;

    nonSecureAddr = secureRegionEnd + (useFirstHalf? 0: ((realMemorySize - physMemorySize) / 2));
    NN_KERN_ABORT_UNLESS(nonSecureEnd  == KMemoryLayout::GetHeapRegionEnd() - (useFirstHalf? (realMemorySize - physMemorySize): ((realMemorySize - physMemorySize) / 2)));
    size_t nonSecureSize = nonSecureEnd - nonSecureAddr;
    NN_KERN_ABORT_UNLESS(p->applicationSize + p->appletSize <= nonSecureSize);
    p->applicationAddr = nonSecureEnd - p->applicationSize;
    p->appletAddr = p->applicationAddr - p->appletSize;
    p->nonSecureSystemAddr = nonSecureAddr;
    p->nonSecureSystemSize = p->appletAddr - p->nonSecureSystemAddr;

    p->secureSystemBackAddr     = p->secureSystemAddr    + p->secureSystemSize;
    p->nonSecureSystemBackAddr  = p->nonSecureSystemAddr + p->nonSecureSystemSize;
    p->appletBackAddr           = p->appletAddr          + p->appletSize;
    p->applicationBackAddr      = p->applicationAddr     + p->applicationSize;

    if (p->secureSystemAddr < backAddr && backAddr < p->secureSystemAddr + p->secureSystemSize)
    {
        p->secureSystemBackAddr = backAddr;
    }
    if (p->nonSecureSystemAddr < backAddr && backAddr < p->nonSecureSystemAddr + p->nonSecureSystemSize)
    {
        p->nonSecureSystemBackAddr = backAddr;
    }
    if (p->appletAddr < backAddr && backAddr < p->appletAddr + p->appletSize)
    {
        p->appletBackAddr = backAddr;
    }
    if (p->applicationAddr < backAddr && backAddr < p->applicationAddr + p->applicationSize)
    {
        p->applicationBackAddr = backAddr;
    }
}

// NX では 非セキュアな初期プログラムをアプリ領域で動かす。
// 非セキュアな初期プログラムは boot プロセスだけなので、アプリが起動する前には終了する前提。
KMemoryManager::Region KSystemControl::GetCreateProcessMemoryRegion()
{
    return KMemoryManager::Region_Application;
}

void KSystemControl::SystemInitialize0()
{
    {
        NX::smc::ConfigItemData config = {};
        NX::smc::GetConfig(&config, NX::smc::ConfigItem_IsDevelopmentFunctionEnabled);
        KTargetSystem::SetDevelopmentHardware(config.data[0]);

        if (KTargetSystem::IsDevelopmentHardware())
        {
            KTargetSystem::EnablePrint(true);
        }
    }
    {
        NX::smc::ConfigItemData config = {};
        NX::smc::GetConfig(&config, NX::smc::ConfigItem_KernelFlags);
        nn::util::BitPack32 kernelFlags = { static_cast<Bit32>(config.data[0]) };
        KTargetSystem::EnableUserExceptionHandler(kernelFlags.Get<NX::smc::KernelFlags::EnableUserExceptionHandler>());
        KTargetSystem::EnableNonZeroFillMemory(kernelFlags.Get<NX::smc::KernelFlags::EnableNonZeroFillMemory>());
        g_CallShowErrorOnPanic = kernelFlags.Get<NX::smc::KernelFlags::CallShowErrorOnPanic>();
        KTargetSystem::EnableUserPmuAccess(kernelFlags.Get<NX::smc::KernelFlags::EnablePmuAccess>());
    }
    {
        NX::smc::ConfigItemData config = {};
        NX::smc::GetConfig(&config, NX::smc::ConfigItem_DisableDescVerification);
        KTargetSystem::EnableInternalDebug(!!config.data[0]);
    }
    {
        KHeapArrange arrange;
        GetHeapArrange(&arrange);

        KPhysicalAddress secureBegin = KMemoryLayout::GetKernelRegionPhysicalBegin();
        KPhysicalAddress secureEnd = KMemoryLayout::ToLinearPhysicalAddress(std::max(GetAsInteger(arrange.managementAddr + arrange.managementSize), GetAsInteger(arrange.secureSystemAddr + arrange.secureSystemSize)));
        NX::smc::SetKernelCarveoutRegion(0, GetAsInteger(secureBegin), secureEnd - secureBegin);
        NN_LOG("Set Carveout Region %p - %p\n", secureBegin, secureEnd - 1);
    }

    KResourceLimit& resourceLimit = Kernel::GetSystemResourceLimit();
    KAutoObject::Create(&resourceLimit);
    resourceLimit.Initialize();
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_PhysicalMemoryMax,      GetPhysicalMemorySize()).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_ThreadCountMax,         NN_KERN_SLAB_OBJ_NUM_THREAD).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_EventCountMax,          NN_KERN_SLAB_OBJ_NUM_EVENT).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_TransferMemoryCountMax, NN_KERN_SLAB_OBJ_NUM_TRANSFER_MEMORY).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_SessionCountMax,        NN_KERN_SLAB_OBJ_NUM_SESSION).IsSuccess());

    NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, KMemoryLayout::GetKernelRegionPhysicalSize()));
    if (NN_KERN_P_ADDR_RESERVED_LO_SIZE)
    {
        NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, NN_KERN_P_ADDR_RESERVED_LO_SIZE));
    }
    if (NN_KERN_P_ADDR_RESERVED_HI_SIZE)
    {
        NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, NN_KERN_P_ADDR_RESERVED_HI_SIZE));
    }
}

void KSystemControl::SystemInitialize1()
{
    NX::SleepManager::Initialize();

    InitializeAppletSecureMemory();

#if defined NN_KERN_ENABLE_KERNEL_TRACE
    InitializeTrace();
#endif
#if defined NN_KERN_ENABLE_KTRACE
    KTrace::Initialize(KMemoryLayout::ToLinearVirtualAddress(KMemoryLayout::GetDebugRegionPhysicalBegin()), NN_KERN_KTRACE_BUFFER_SIZE);
#endif
}

void KSystemControl::CallSecureMonitor64(Bit64* pBuf)
{
    Bit64 x = pBuf[0];

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();
    const int numMapRegs = 7;

    KPageGroup pg[numMapRegs];

    for (int i = 0; i < numMapRegs; i++)
    {
        int regNo = i + 1;
        if ((1ul << (regNo + 8)) & x)
        {
            pg[i].Initialize(pageTable.GetBlockInfoManager());
            KProcessAddress va = pBuf[regNo];

            Result result = pageTable.MakePageGroupAndOpen(
                    &pg[i], GetAsInteger(va) & ~static_cast<size_t>(NN_KERN_FINEST_PAGE_SIZE - 1), 1,
                    0, 0,
                    KMemoryPermission_UserReadWrite, KMemoryPermission_UserReadWrite,
                    0, 0);
            if (result.IsSuccess())
            {
                KPageGroup::BlockInfoList::const_iterator it = pg[i].GetBlockBeginIter();
                NN_KERN_ASSERT(it != pg[i].GetBlockEndIter());
                NN_KERN_ASSERT(it->GetNumPages() == 1);

                KPhysicalAddress pa = pageTable.GetPageTable().GetHeapPhysicalAddress(it->GetBlockAddr());

                pBuf[regNo] = GetAsInteger(pa) | (GetAsInteger(va) & static_cast<size_t>(NN_KERN_FINEST_PAGE_SIZE - 1));
            }
            else
            {
                pBuf[regNo] = 0;
            }
        }
    }

    NX::smc::SecureMonitorCall(pBuf);

    for (int i = 0; i < numMapRegs; i++)
    {
        pg[i].Close();
        pg[i].Finalize();
    }
}

void KSystemControl::SleepSystem()
{
    NN_LOG("SleepSystem called\n");
    NX::SleepManager::SleepSystem();
}

void KSystemControl::ReadWriteRegisterFromKernel(Bit32* pOut, nn::svc::PhysicalAddress physAddress, Bit32 mask, Bit32 value)
{
    NN_KERN_ABORT_UNLESS((physAddress & 0x3) == 0);
    NN_KERN_ABORT_UNLESS(IsSecureReadWriteRegisterForKernel(physAddress));
    NN_KERN_ABORT_UNLESS(NX::smc::ReadWriteRegister(pOut, physAddress, mask, value));
}

Result KSystemControl::ReadWriteRegister(Bit32* pOut, nn::svc::PhysicalAddress physAddress, Bit32 mask, Bit32 value)
{
    if (NN_UNLIKELY(physAddress & 0x3))
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (IsSecureReadWriteRegister(physAddress))
    {
        if (NN_UNLIKELY(!NX::smc::ReadWriteRegister(pOut, physAddress, mask, value)))
        {
            return nn::svc::ResultInvalidAddress();
        }
    }
    else
    {
        volatile Bit32* pRegister = GetRegisterAddess(physAddress);

        if (NN_UNLIKELY(!pRegister))
        {
            return nn::svc::ResultInvalidAddress();
        }

        if (mask == 0)
        {
            *pOut = *pRegister;
        }
        else if (mask == 0xFFFFFFFF)
        {
            *pOut = 0;
            *pRegister = value;
        }
        else
        {
            *pOut = *pRegister;
            *pRegister = (*pOut & ~mask) | (value & mask);
        }
    }

    return ResultSuccess();
}

void KSystemControl::StopSystem()
{
    if (g_CallShowErrorOnPanic)
    {
        NX::smc::ShowError(0xF00);
    }
    for (;;) {}
}

bool KSystemControl::IsIoMappablePhysicalAddress(KPhysicalAddress physAddr, size_t size)
{
    if (physAddr > physAddr + size)
    {
        return false;
    }

    for (size_t i = 0; i < NN_ARRAY_SIZE(g_MappableBlackList); i++)
    {
        KPhysicalAddress testAddr = g_MappableBlackList[i].addr;
        size_t testSize           = g_MappableBlackList[i].size;

        if (!((physAddr + size  - 1 < testAddr) || (testAddr + testSize - 1 < physAddr)))
        {
            return false;
        }
    }

    return true;
}

bool KSystemControl::IsStaticMappablePhysicalAddress(KPhysicalAddress physAddr, size_t size)
{
    if (!(physAddr < physAddr + size))
    {
        return false;
    }

#if defined NN_KERN_ENABLE_KERNEL_TRACE || defined NN_KERN_ENABLE_KTRACE
    if (KMemoryLayout::InDebugRegionPhysical(physAddr) && KMemoryLayout::InDebugRegionPhysical(physAddr + size - 1))
    {
        return true;
    }
#endif

    return false;
}

size_t KSystemControl::ComputeMemoryRequirementForSecureMemory(size_t size, KMemoryManager::Region region)
{
    if (region == KMemoryManager::Region_Applet)
    {
        return 0;
    }

    return size;
}

Result KSystemControl::AllocateSecureMemory(KVirtualAddress* pAddr, size_t size, KMemoryManager::Region region)
{
    if (region == KMemoryManager::Region_Applet)
    {
        return AllocateSecureMemoryForApplet(pAddr, size);
    }

    size_t align = (region == KMemoryManager::Region_SecureSystem? KMemoryManager::PageSize: SecureRegionAlign);

    if ((size & (align - 1)) != 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    Bit32 allocateOption = KMemoryManager::MakeAllocateOption(region, KMemoryManager::From_Front);
    KVirtualAddress va = Kernel::GetKernelHeapManager().AllocateContinuous(size / KMemoryManager::PageSize, align / KMemoryManager::PageSize, allocateOption);

    if (va == Null<KVirtualAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }

    Kernel::GetKernelHeapManager().Open(va, size / KMemoryManager::PageSize);

    if (region != KMemoryManager::Region_SecureSystem)
    {
        KPhysicalAddress pa = KPageTable::GetHeapPhysicalAddress(va);
        NN_KERN_ABORT_UNLESS(pa != Null<KPhysicalAddress>());
        if (!SetSecureRegion(pa, size))
        {
            Kernel::GetKernelHeapManager().Close(va, size / KMemoryManager::PageSize);
            return nn::svc::ResultOutOfMemory();
        }
    }

    *pAddr = va;

    return ResultSuccess();
}

void KSystemControl::FreeSecureMemory(KVirtualAddress addr, size_t size, KMemoryManager::Region region)
{
    if (region == KMemoryManager::Region_Applet)
    {
        FreeSecureMemoryForApplet(addr, size);
        return;
    }

    size_t align = (region == KMemoryManager::Region_SecureSystem? KMemoryManager::PageSize: SecureRegionAlign);

    NN_KERN_ABORT_UNLESS((GetAsInteger(addr) & (align - 1)) == 0);
    NN_KERN_ABORT_UNLESS(((size) & (align - 1)) == 0);

    if (region != KMemoryManager::Region_SecureSystem)
    {
        NN_KERN_ABORT_UNLESS(g_SecureRegionSize == size);
        KPhysicalAddress pa = KPageTable::GetHeapPhysicalAddress(addr);
        NN_KERN_ABORT_UNLESS(pa != Null<KPhysicalAddress>());
        NN_KERN_ABORT_UNLESS(g_SecureRegionAddr == pa);
        NN_KERN_ABORT_UNLESS(SetSecureRegion(pa, 0));
    }

    Kernel::GetKernelHeapManager().Close(addr, size / KMemoryManager::PageSize);
}

}}

