﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_Platform.h"
#include "kern_SystemControl.h"
#include "../../kern_Result.h"
#include "../../kern_Kernel.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_Main.h"
#include "../../kern_KProcess.h"
#include "kern_MemoryMap.h"
#include "../../kern_PageTableSelect.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../../kern_KScopedSchedulingLock.h"
#include "../../kern_KLightMutex.h"
#include "../../kern_KLightConditionvariable.h"
#include "../../kern_KThread.h"
#include "../../kern_DevicePageTableSelect.h"
#include "../../kern_Utility.h"
#include <cstring>
#include <random>
#include "kern_SecureMonitorCall.h"
#include "kern_SleepManager.h"

namespace nn { namespace kern { namespace NX { namespace SleepManager {

namespace {
struct CpuRegisters
{
public:
    void Save();
    void Restore() const;
private:
    // mair, sctlr, ttbr1は スリープハンドラ内で保存復帰する
    Bit64 ttbr0;
    Bit64 tcr;
    Bit64 elr_el1;
    Bit64 sp_el0;
    Bit64 spsr_el1;
    Bit64 daif;
    Bit64 cpacr_el1;
    Bit64 vbar_el1;
    Bit64 csselr_el1;
    Bit64 cntp_ctl_el0;
    Bit64 cntp_cval_el0;
    Bit64 cntkctl_el1;
    Bit64 tpidr_el0;
    Bit64 tpidrro_el0;

    // debug
    Bit64 mdscr;
    Bit64 contextidr;
    Bit64 dbgwcr[16];
    Bit64 dbgwvr[16];
    Bit64 dbgbcr[16];
    Bit64 dbgbvr[16];

    // pmu
    Bit64 pmccfiltr_el0;
    Bit64 pmccntr_el0;
    Bit64 pmcntenset_el0;
    Bit64 pmcr_el0;
    Bit64 pmevcntr_el0[31];
    Bit64 pmevtyper_el0[31];
    Bit64 pmintenset_el1;
    Bit64 pmovsset_el0;
    Bit64 pmselr_el0;
    Bit64 pmuserenr_el0;
};

Bit64           g_SleepProcessCpus = 0;
KThreadQueue    g_SleepRequestWaiterQueue;
KPhysicalAddress g_SleepBufferPhyscalAddress[KCPU::NUM_CORE];
struct CpuRegisters g_CpuRegisters[KCPU::NUM_CORE];
KLightMutex     g_Mutex;
KLightMutex             g_SyncMutex;
KLightConditionvariable g_SyncCv;

void CpuRegisters::Save()
{
    // レジスタ保存

    HW_GET_TTBR0_EL1(ttbr0);
    HW_GET_TCR_EL1(tcr);
    HW_GET_TPIDR_EL0(tpidr_el0);
    HW_GET_ELR_EL1(elr_el1);
    HW_GET_SP_EL0(sp_el0);
    HW_GET_SPSR_EL1(spsr_el1);
    HW_GET_DAIF(daif);
    HW_GET_CPACR_EL1(cpacr_el1);
    HW_GET_VBAR_EL1(vbar_el1);
    HW_GET_CSSELR_EL1(csselr_el1);
    HW_GET_CNTP_CTL_EL0(cntp_ctl_el0);
    HW_GET_CNTP_CVAL_EL0(cntp_cval_el0);
    HW_GET_CNTKCTL_EL1(cntkctl_el1);
    HW_GET_THREAD_ID_USER_READ_ONLY(tpidrro_el0);

    // pmu
    HW_GET_PMCR_EL0(pmcr_el0);
    HW_SET_PMCR_EL0(0ul); // 停止する
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();
    HW_GET_PMUSERENR_EL0(pmuserenr_el0);
    HW_GET_PMSELR_EL0(pmselr_el0);
    HW_GET_PMCCFILTR_EL0(pmccfiltr_el0);
    HW_GET_PMCNTENSET_EL0(pmcntenset_el0);
    HW_GET_PMINTENSET_EL1(pmintenset_el1);
    HW_GET_PMOVSSET_EL0(pmovsset_el0);
    HW_GET_PMCCNTR_EL0(pmccntr_el0);

    switch ((pmcr_el0 >> 11) & 0x1F)
    {
    case 31:
        {
            HW_GET_PMEVCNTR_EL0(30, pmevcntr_el0[30]);
            HW_GET_PMEVTYPER_EL0(30, pmevtyper_el0[30]);
        } NN_FALL_THROUGH;
    case 30:
        {
            HW_GET_PMEVCNTR_EL0(29, pmevcntr_el0[29]);
            HW_GET_PMEVTYPER_EL0(29, pmevtyper_el0[29]);
        } NN_FALL_THROUGH;
    case 29:
        {
            HW_GET_PMEVCNTR_EL0(28, pmevcntr_el0[28]);
            HW_GET_PMEVTYPER_EL0(28, pmevtyper_el0[28]);
        } NN_FALL_THROUGH;
    case 28:
        {
            HW_GET_PMEVCNTR_EL0(27, pmevcntr_el0[27]);
            HW_GET_PMEVTYPER_EL0(27, pmevtyper_el0[27]);
        } NN_FALL_THROUGH;
    case 27:
        {
            HW_GET_PMEVCNTR_EL0(26, pmevcntr_el0[26]);
            HW_GET_PMEVTYPER_EL0(26, pmevtyper_el0[26]);
        } NN_FALL_THROUGH;
    case 26:
        {
            HW_GET_PMEVCNTR_EL0(25, pmevcntr_el0[25]);
            HW_GET_PMEVTYPER_EL0(25, pmevtyper_el0[25]);
        } NN_FALL_THROUGH;
    case 25:
        {
            HW_GET_PMEVCNTR_EL0(24, pmevcntr_el0[24]);
            HW_GET_PMEVTYPER_EL0(24, pmevtyper_el0[24]);
        } NN_FALL_THROUGH;
    case 24:
        {
            HW_GET_PMEVCNTR_EL0(23, pmevcntr_el0[23]);
            HW_GET_PMEVTYPER_EL0(23, pmevtyper_el0[23]);
        } NN_FALL_THROUGH;
    case 23:
        {
            HW_GET_PMEVCNTR_EL0(22, pmevcntr_el0[22]);
            HW_GET_PMEVTYPER_EL0(22, pmevtyper_el0[22]);
        } NN_FALL_THROUGH;
    case 22:
        {
            HW_GET_PMEVCNTR_EL0(21, pmevcntr_el0[21]);
            HW_GET_PMEVTYPER_EL0(21, pmevtyper_el0[21]);
        } NN_FALL_THROUGH;
    case 21:
        {
            HW_GET_PMEVCNTR_EL0(20, pmevcntr_el0[20]);
            HW_GET_PMEVTYPER_EL0(20, pmevtyper_el0[20]);
        } NN_FALL_THROUGH;
    case 20:
        {
            HW_GET_PMEVCNTR_EL0(19, pmevcntr_el0[19]);
            HW_GET_PMEVTYPER_EL0(19, pmevtyper_el0[19]);
        } NN_FALL_THROUGH;
    case 19:
        {
            HW_GET_PMEVCNTR_EL0(18, pmevcntr_el0[18]);
            HW_GET_PMEVTYPER_EL0(18, pmevtyper_el0[18]);
        } NN_FALL_THROUGH;
    case 18:
        {
            HW_GET_PMEVCNTR_EL0(17, pmevcntr_el0[17]);
            HW_GET_PMEVTYPER_EL0(17, pmevtyper_el0[17]);
        } NN_FALL_THROUGH;
    case 17:
        {
            HW_GET_PMEVCNTR_EL0(16, pmevcntr_el0[16]);
            HW_GET_PMEVTYPER_EL0(16, pmevtyper_el0[16]);
        } NN_FALL_THROUGH;
    case 16:
        {
            HW_GET_PMEVCNTR_EL0(15, pmevcntr_el0[15]);
            HW_GET_PMEVTYPER_EL0(15, pmevtyper_el0[15]);
        } NN_FALL_THROUGH;
    case 15:
        {
            HW_GET_PMEVCNTR_EL0(14, pmevcntr_el0[14]);
            HW_GET_PMEVTYPER_EL0(14, pmevtyper_el0[14]);
        } NN_FALL_THROUGH;
    case 14:
        {
            HW_GET_PMEVCNTR_EL0(13, pmevcntr_el0[13]);
            HW_GET_PMEVTYPER_EL0(13, pmevtyper_el0[13]);
        } NN_FALL_THROUGH;
    case 13:
        {
            HW_GET_PMEVCNTR_EL0(12, pmevcntr_el0[12]);
            HW_GET_PMEVTYPER_EL0(12, pmevtyper_el0[12]);
        } NN_FALL_THROUGH;
    case 12:
        {
            HW_GET_PMEVCNTR_EL0(11, pmevcntr_el0[11]);
            HW_GET_PMEVTYPER_EL0(11, pmevtyper_el0[11]);
        } NN_FALL_THROUGH;
    case 11:
        {
            HW_GET_PMEVCNTR_EL0(10, pmevcntr_el0[10]);
            HW_GET_PMEVTYPER_EL0(10, pmevtyper_el0[10]);
        } NN_FALL_THROUGH;
    case 10:
        {
            HW_GET_PMEVCNTR_EL0(9, pmevcntr_el0[9]);
            HW_GET_PMEVTYPER_EL0(9, pmevtyper_el0[9]);
        } NN_FALL_THROUGH;
    case 9:
        {
            HW_GET_PMEVCNTR_EL0(8, pmevcntr_el0[8]);
            HW_GET_PMEVTYPER_EL0(8, pmevtyper_el0[8]);
        } NN_FALL_THROUGH;
    case 8:
        {
            HW_GET_PMEVCNTR_EL0(7, pmevcntr_el0[7]);
            HW_GET_PMEVTYPER_EL0(7, pmevtyper_el0[7]);
        } NN_FALL_THROUGH;
    case 7:
        {
            HW_GET_PMEVCNTR_EL0(6, pmevcntr_el0[6]);
            HW_GET_PMEVTYPER_EL0(6, pmevtyper_el0[6]);
        } NN_FALL_THROUGH;
    case 6:
        {
            HW_GET_PMEVCNTR_EL0(5, pmevcntr_el0[5]);
            HW_GET_PMEVTYPER_EL0(5, pmevtyper_el0[5]);
        } NN_FALL_THROUGH;
    case 5:
        {
            HW_GET_PMEVCNTR_EL0(4, pmevcntr_el0[4]);
            HW_GET_PMEVTYPER_EL0(4, pmevtyper_el0[4]);
        } NN_FALL_THROUGH;
    case 4:
        {
            HW_GET_PMEVCNTR_EL0(3, pmevcntr_el0[3]);
            HW_GET_PMEVTYPER_EL0(3, pmevtyper_el0[3]);
        } NN_FALL_THROUGH;
    case 3:
        {
            HW_GET_PMEVCNTR_EL0(2, pmevcntr_el0[2]);
            HW_GET_PMEVTYPER_EL0(2, pmevtyper_el0[2]);
        } NN_FALL_THROUGH;
    case 2:
        {
            HW_GET_PMEVCNTR_EL0(1, pmevcntr_el0[1]);
            HW_GET_PMEVTYPER_EL0(1, pmevtyper_el0[1]);
        } NN_FALL_THROUGH;
    case 1:
        {
            HW_GET_PMEVCNTR_EL0(0, pmevcntr_el0[0]);
            HW_GET_PMEVTYPER_EL0(0, pmevtyper_el0[0]);
        } NN_FALL_THROUGH;
    default:
    case 0:
        break;
    }

    // debug register
    Bit64 dfr0;
    HW_GET_ID_AA64DFR0_EL1(dfr0);
    int wrps = ((dfr0 >> 20) & 0xF);
    int brps = ((dfr0 >> 12) & 0xF);

    HW_GET_MDSCR_EL1(mdscr);
    HW_GET_CONTEXTIDR_EL1(contextidr);

    switch (wrps)
    {
    case 0xF:
        {
            HW_GET_DBGWCR_EL1(15, dbgwcr[15]);
            HW_GET_DBGWVR_EL1(15, dbgwvr[15]);
        } NN_FALL_THROUGH;
    case 0xE:
        {
            HW_GET_DBGWCR_EL1(14, dbgwcr[14]);
            HW_GET_DBGWVR_EL1(14, dbgwvr[14]);
        } NN_FALL_THROUGH;
    case 0xD:
        {
            HW_GET_DBGWCR_EL1(13, dbgwcr[13]);
            HW_GET_DBGWVR_EL1(13, dbgwvr[13]);
        } NN_FALL_THROUGH;
    case 0xC:
        {
            HW_GET_DBGWCR_EL1(12, dbgwcr[12]);
            HW_GET_DBGWVR_EL1(12, dbgwvr[12]);
        } NN_FALL_THROUGH;
    case 0xB:
        {
            HW_GET_DBGWCR_EL1(11, dbgwcr[11]);
            HW_GET_DBGWVR_EL1(11, dbgwvr[11]);
        } NN_FALL_THROUGH;
    case 0xA:
        {
            HW_GET_DBGWCR_EL1(10, dbgwcr[10]);
            HW_GET_DBGWVR_EL1(10, dbgwvr[10]);
        } NN_FALL_THROUGH;
    case 0x9:
        {
            HW_GET_DBGWCR_EL1(9, dbgwcr[9]);
            HW_GET_DBGWVR_EL1(9, dbgwvr[9]);
        } NN_FALL_THROUGH;
    case 0x8:
        {
            HW_GET_DBGWCR_EL1(8, dbgwcr[8]);
            HW_GET_DBGWVR_EL1(8, dbgwvr[8]);
        } NN_FALL_THROUGH;
    case 0x7:
        {
            HW_GET_DBGWCR_EL1(7, dbgwcr[7]);
            HW_GET_DBGWVR_EL1(7, dbgwvr[7]);
        } NN_FALL_THROUGH;
    case 0x6:
        {
            HW_GET_DBGWCR_EL1(6, dbgwcr[6]);
            HW_GET_DBGWVR_EL1(6, dbgwvr[6]);
        } NN_FALL_THROUGH;
    case 0x5:
        {
            HW_GET_DBGWCR_EL1(5, dbgwcr[5]);
            HW_GET_DBGWVR_EL1(5, dbgwvr[5]);
        } NN_FALL_THROUGH;
    case 0x4:
        {
            HW_GET_DBGWCR_EL1(4, dbgwcr[4]);
            HW_GET_DBGWVR_EL1(4, dbgwvr[4]);
        } NN_FALL_THROUGH;
    case 0x3:
        {
            HW_GET_DBGWCR_EL1(3, dbgwcr[3]);
            HW_GET_DBGWVR_EL1(3, dbgwvr[3]);
        } NN_FALL_THROUGH;
    case 0x2:
        {
            HW_GET_DBGWCR_EL1(2, dbgwcr[2]);
            HW_GET_DBGWVR_EL1(2, dbgwvr[2]);
        } NN_FALL_THROUGH;
    case 0x1:
        {
            HW_GET_DBGWCR_EL1(1, dbgwcr[1]);
            HW_GET_DBGWVR_EL1(1, dbgwvr[1]);

            HW_GET_DBGWCR_EL1(0, dbgwcr[0]);
            HW_GET_DBGWVR_EL1(0, dbgwvr[0]);
        } NN_FALL_THROUGH;
    default:
        break;
    }

    switch (brps)
    {
    case 0xF:
        {
            HW_GET_DBGBCR_EL1(15, dbgbcr[15]);
            HW_GET_DBGBVR_EL1(15, dbgbvr[15]);
        } NN_FALL_THROUGH;
    case 0xE:
        {
            HW_GET_DBGBCR_EL1(14, dbgbcr[14]);
            HW_GET_DBGBVR_EL1(14, dbgbvr[14]);
        } NN_FALL_THROUGH;
    case 0xD:
        {
            HW_GET_DBGBCR_EL1(13, dbgbcr[13]);
            HW_GET_DBGBVR_EL1(13, dbgbvr[13]);
        } NN_FALL_THROUGH;
    case 0xC:
        {
            HW_GET_DBGBCR_EL1(12, dbgbcr[12]);
            HW_GET_DBGBVR_EL1(12, dbgbvr[12]);
        } NN_FALL_THROUGH;
    case 0xB:
        {
            HW_GET_DBGBCR_EL1(11, dbgbcr[11]);
            HW_GET_DBGBVR_EL1(11, dbgbvr[11]);
        } NN_FALL_THROUGH;
    case 0xA:
        {
            HW_GET_DBGBCR_EL1(10, dbgbcr[10]);
            HW_GET_DBGBVR_EL1(10, dbgbvr[10]);
        } NN_FALL_THROUGH;
    case 0x9:
        {
            HW_GET_DBGBCR_EL1(9, dbgbcr[9]);
            HW_GET_DBGBVR_EL1(9, dbgbvr[9]);
        } NN_FALL_THROUGH;
    case 0x8:
        {
            HW_GET_DBGBCR_EL1(8, dbgbcr[8]);
            HW_GET_DBGBVR_EL1(8, dbgbvr[8]);
        } NN_FALL_THROUGH;
    case 0x7:
        {
            HW_GET_DBGBCR_EL1(7, dbgbcr[7]);
            HW_GET_DBGBVR_EL1(7, dbgbvr[7]);
        } NN_FALL_THROUGH;
    case 0x6:
        {
            HW_GET_DBGBCR_EL1(6, dbgbcr[6]);
            HW_GET_DBGBVR_EL1(6, dbgbvr[6]);
        } NN_FALL_THROUGH;
    case 0x5:
        {
            HW_GET_DBGBCR_EL1(5, dbgbcr[5]);
            HW_GET_DBGBVR_EL1(5, dbgbvr[5]);
        } NN_FALL_THROUGH;
    case 0x4:
        {
            HW_GET_DBGBCR_EL1(4, dbgbcr[4]);
            HW_GET_DBGBVR_EL1(4, dbgbvr[4]);
        } NN_FALL_THROUGH;
    case 0x3:
        {
            HW_GET_DBGBCR_EL1(3, dbgbcr[3]);
            HW_GET_DBGBVR_EL1(3, dbgbvr[3]);
        } NN_FALL_THROUGH;
    case 0x2:
        {
            HW_GET_DBGBCR_EL1(2, dbgbcr[2]);
            HW_GET_DBGBVR_EL1(2, dbgbvr[2]);
        } NN_FALL_THROUGH;
    case 0x1:
        {
            HW_GET_DBGBCR_EL1(1, dbgbcr[1]);
            HW_GET_DBGBVR_EL1(1, dbgbvr[1]);
        } NN_FALL_THROUGH;
    default:
        break;
    }

    HW_GET_DBGBCR_EL1(0, dbgbcr[0]);
    HW_GET_DBGBVR_EL1(0, dbgbvr[0]);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();
    HW_SET_MDSCR_EL1(0ul);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();
}

void CpuRegisters::Restore() const
{
    Bit64 dfr0;
    HW_GET_ID_AA64DFR0_EL1(dfr0);
    int wrps = ((dfr0 >> 20) & 0xF);
    int brps = ((dfr0 >> 12) & 0xF);

    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    HW_SET_MDSCR_EL1(0ul);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    HW_SET_OSLAR_EL1(0ul);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    switch (wrps)
    {
    case 0xF:
        {
            HW_SET_DBGWCR_EL1(15, dbgwcr[15]);
            HW_SET_DBGWVR_EL1(15, dbgwvr[15]);
        } NN_FALL_THROUGH;
    case 0xE:
        {
            HW_SET_DBGWCR_EL1(14, dbgwcr[14]);
            HW_SET_DBGWVR_EL1(14, dbgwvr[14]);
        } NN_FALL_THROUGH;
    case 0xD:
        {
            HW_SET_DBGWCR_EL1(13, dbgwcr[13]);
            HW_SET_DBGWVR_EL1(13, dbgwvr[13]);
        } NN_FALL_THROUGH;
    case 0xC:
        {
            HW_SET_DBGWCR_EL1(12, dbgwcr[12]);
            HW_SET_DBGWVR_EL1(12, dbgwvr[12]);
        } NN_FALL_THROUGH;
    case 0xB:
        {
            HW_SET_DBGWCR_EL1(11, dbgwcr[11]);
            HW_SET_DBGWVR_EL1(11, dbgwvr[11]);
        } NN_FALL_THROUGH;
    case 0xA:
        {
            HW_SET_DBGWCR_EL1(10, dbgwcr[10]);
            HW_SET_DBGWVR_EL1(10, dbgwvr[10]);
        } NN_FALL_THROUGH;
    case 0x9:
        {
            HW_SET_DBGWCR_EL1(9, dbgwcr[9]);
            HW_SET_DBGWVR_EL1(9, dbgwvr[9]);
        } NN_FALL_THROUGH;
    case 0x8:
        {
            HW_SET_DBGWCR_EL1(8, dbgwcr[8]);
            HW_SET_DBGWVR_EL1(8, dbgwvr[8]);
        } NN_FALL_THROUGH;
    case 0x7:
        {
            HW_SET_DBGWCR_EL1(7, dbgwcr[7]);
            HW_SET_DBGWVR_EL1(7, dbgwvr[7]);
        } NN_FALL_THROUGH;
    case 0x6:
        {
            HW_SET_DBGWCR_EL1(6, dbgwcr[6]);
            HW_SET_DBGWVR_EL1(6, dbgwvr[6]);
        } NN_FALL_THROUGH;
    case 0x5:
        {
            HW_SET_DBGWCR_EL1(5, dbgwcr[5]);
            HW_SET_DBGWVR_EL1(5, dbgwvr[5]);
        } NN_FALL_THROUGH;
    case 0x4:
        {
            HW_SET_DBGWCR_EL1(4, dbgwcr[4]);
            HW_SET_DBGWVR_EL1(4, dbgwvr[4]);
        } NN_FALL_THROUGH;
    case 0x3:
        {
            HW_SET_DBGWCR_EL1(3, dbgwcr[3]);
            HW_SET_DBGWVR_EL1(3, dbgwvr[3]);
        } NN_FALL_THROUGH;
    case 0x2:
        {
            HW_SET_DBGWCR_EL1(2, dbgwcr[2]);
            HW_SET_DBGWVR_EL1(2, dbgwvr[2]);
        } NN_FALL_THROUGH;
    case 0x1:
        {
            HW_SET_DBGWCR_EL1(1, dbgwcr[1]);
            HW_SET_DBGWVR_EL1(1, dbgwvr[1]);

            HW_SET_DBGWCR_EL1(0, dbgwcr[0]);
            HW_SET_DBGWVR_EL1(0, dbgwvr[0]);
        } NN_FALL_THROUGH;
    default:
        break;
    }

    switch (brps)
    {
    case 0xF:
        {
            HW_SET_DBGBCR_EL1(15, dbgbcr[15]);
            HW_SET_DBGBVR_EL1(15, dbgbvr[15]);
        } NN_FALL_THROUGH;
    case 0xE:
        {
            HW_SET_DBGBCR_EL1(14, dbgbcr[14]);
            HW_SET_DBGBVR_EL1(14, dbgbvr[14]);
        } NN_FALL_THROUGH;
    case 0xD:
        {
            HW_SET_DBGBCR_EL1(13, dbgbcr[13]);
            HW_SET_DBGBVR_EL1(13, dbgbvr[13]);
        } NN_FALL_THROUGH;
    case 0xC:
        {
            HW_SET_DBGBCR_EL1(12, dbgbcr[12]);
            HW_SET_DBGBVR_EL1(12, dbgbvr[12]);
        } NN_FALL_THROUGH;
    case 0xB:
        {
            HW_SET_DBGBCR_EL1(11, dbgbcr[11]);
            HW_SET_DBGBVR_EL1(11, dbgbvr[11]);
        } NN_FALL_THROUGH;
    case 0xA:
        {
            HW_SET_DBGBCR_EL1(10, dbgbcr[10]);
            HW_SET_DBGBVR_EL1(10, dbgbvr[10]);
        } NN_FALL_THROUGH;
    case 0x9:
        {
            HW_SET_DBGBCR_EL1(9, dbgbcr[9]);
            HW_SET_DBGBVR_EL1(9, dbgbvr[9]);
        } NN_FALL_THROUGH;
    case 0x8:
        {
            HW_SET_DBGBCR_EL1(8, dbgbcr[8]);
            HW_SET_DBGBVR_EL1(8, dbgbvr[8]);
        } NN_FALL_THROUGH;
    case 0x7:
        {
            HW_SET_DBGBCR_EL1(7, dbgbcr[7]);
            HW_SET_DBGBVR_EL1(7, dbgbvr[7]);
        } NN_FALL_THROUGH;
    case 0x6:
        {
            HW_SET_DBGBCR_EL1(6, dbgbcr[6]);
            HW_SET_DBGBVR_EL1(6, dbgbvr[6]);
        } NN_FALL_THROUGH;
    case 0x5:
        {
            HW_SET_DBGBCR_EL1(5, dbgbcr[5]);
            HW_SET_DBGBVR_EL1(5, dbgbvr[5]);
        } NN_FALL_THROUGH;
    case 0x4:
        {
            HW_SET_DBGBCR_EL1(4, dbgbcr[4]);
            HW_SET_DBGBVR_EL1(4, dbgbvr[4]);
        } NN_FALL_THROUGH;
    case 0x3:
        {
            HW_SET_DBGBCR_EL1(3, dbgbcr[3]);
            HW_SET_DBGBVR_EL1(3, dbgbvr[3]);
        } NN_FALL_THROUGH;
    case 0x2:
        {
            HW_SET_DBGBCR_EL1(2, dbgbcr[2]);
            HW_SET_DBGBVR_EL1(2, dbgbvr[2]);
        } NN_FALL_THROUGH;
    case 0x1:
        {
            HW_SET_DBGBCR_EL1(1, dbgbcr[1]);
            HW_SET_DBGBVR_EL1(1, dbgbvr[1]);
        } NN_FALL_THROUGH;
    default:
        break;
    }

    HW_SET_DBGBCR_EL1(0, dbgbcr[0]);
    HW_SET_DBGBVR_EL1(0, dbgbvr[0]);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    HW_SET_CONTEXTIDR_EL1(contextidr);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    HW_SET_MDSCR_EL1(mdscr);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    // pmu
    // 一旦初期化
    HW_SET_PMUSERENR_EL0(0ul);
    HW_SET_PMCR_EL0((1ul << 2) | (1ul << 1));
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();
    HW_SET_PMOVSCLR_EL0(0xFFFFFFFFul);
    HW_SET_PMINTENCLR_EL1(0xFFFFFFFFul);
    HW_SET_PMCNTENCLR_EL0(0xFFFFFFFFul);

    switch ((pmcr_el0 >> 11) & 0x1F)
    {
    case 31:
        {
            HW_SET_PMEVCNTR_EL0(30, pmevcntr_el0[30]);
            HW_SET_PMEVTYPER_EL0(30, pmevtyper_el0[30]);
        } NN_FALL_THROUGH;
    case 30:
        {
            HW_SET_PMEVCNTR_EL0(29, pmevcntr_el0[29]);
            HW_SET_PMEVTYPER_EL0(29, pmevtyper_el0[29]);
        } NN_FALL_THROUGH;
    case 29:
        {
            HW_SET_PMEVCNTR_EL0(28, pmevcntr_el0[28]);
            HW_SET_PMEVTYPER_EL0(28, pmevtyper_el0[28]);
        } NN_FALL_THROUGH;
    case 28:
        {
            HW_SET_PMEVCNTR_EL0(27, pmevcntr_el0[27]);
            HW_SET_PMEVTYPER_EL0(27, pmevtyper_el0[27]);
        } NN_FALL_THROUGH;
    case 27:
        {
            HW_SET_PMEVCNTR_EL0(26, pmevcntr_el0[26]);
            HW_SET_PMEVTYPER_EL0(26, pmevtyper_el0[26]);
        } NN_FALL_THROUGH;
    case 26:
        {
            HW_SET_PMEVCNTR_EL0(25, pmevcntr_el0[25]);
            HW_SET_PMEVTYPER_EL0(25, pmevtyper_el0[25]);
        } NN_FALL_THROUGH;
    case 25:
        {
            HW_SET_PMEVCNTR_EL0(24, pmevcntr_el0[24]);
            HW_SET_PMEVTYPER_EL0(24, pmevtyper_el0[24]);
        } NN_FALL_THROUGH;
    case 24:
        {
            HW_SET_PMEVCNTR_EL0(23, pmevcntr_el0[23]);
            HW_SET_PMEVTYPER_EL0(23, pmevtyper_el0[23]);
        } NN_FALL_THROUGH;
    case 23:
        {
            HW_SET_PMEVCNTR_EL0(22, pmevcntr_el0[22]);
            HW_SET_PMEVTYPER_EL0(22, pmevtyper_el0[22]);
        } NN_FALL_THROUGH;
    case 22:
        {
            HW_SET_PMEVCNTR_EL0(21, pmevcntr_el0[21]);
            HW_SET_PMEVTYPER_EL0(21, pmevtyper_el0[21]);
        } NN_FALL_THROUGH;
    case 21:
        {
            HW_SET_PMEVCNTR_EL0(20, pmevcntr_el0[20]);
            HW_SET_PMEVTYPER_EL0(20, pmevtyper_el0[20]);
        } NN_FALL_THROUGH;
    case 20:
        {
            HW_SET_PMEVCNTR_EL0(19, pmevcntr_el0[19]);
            HW_SET_PMEVTYPER_EL0(19, pmevtyper_el0[19]);
        } NN_FALL_THROUGH;
    case 19:
        {
            HW_SET_PMEVCNTR_EL0(18, pmevcntr_el0[18]);
            HW_SET_PMEVTYPER_EL0(18, pmevtyper_el0[18]);
        } NN_FALL_THROUGH;
    case 18:
        {
            HW_SET_PMEVCNTR_EL0(17, pmevcntr_el0[17]);
            HW_SET_PMEVTYPER_EL0(17, pmevtyper_el0[17]);
        } NN_FALL_THROUGH;
    case 17:
        {
            HW_SET_PMEVCNTR_EL0(16, pmevcntr_el0[16]);
            HW_SET_PMEVTYPER_EL0(16, pmevtyper_el0[16]);
        } NN_FALL_THROUGH;
    case 16:
        {
            HW_SET_PMEVCNTR_EL0(15, pmevcntr_el0[15]);
            HW_SET_PMEVTYPER_EL0(15, pmevtyper_el0[15]);
        } NN_FALL_THROUGH;
    case 15:
        {
            HW_SET_PMEVCNTR_EL0(14, pmevcntr_el0[14]);
            HW_SET_PMEVTYPER_EL0(14, pmevtyper_el0[14]);
        } NN_FALL_THROUGH;
    case 14:
        {
            HW_SET_PMEVCNTR_EL0(13, pmevcntr_el0[13]);
            HW_SET_PMEVTYPER_EL0(13, pmevtyper_el0[13]);
        } NN_FALL_THROUGH;
    case 13:
        {
            HW_SET_PMEVCNTR_EL0(12, pmevcntr_el0[12]);
            HW_SET_PMEVTYPER_EL0(12, pmevtyper_el0[12]);
        } NN_FALL_THROUGH;
    case 12:
        {
            HW_SET_PMEVCNTR_EL0(11, pmevcntr_el0[11]);
            HW_SET_PMEVTYPER_EL0(11, pmevtyper_el0[11]);
        } NN_FALL_THROUGH;
    case 11:
        {
            HW_SET_PMEVCNTR_EL0(10, pmevcntr_el0[10]);
            HW_SET_PMEVTYPER_EL0(10, pmevtyper_el0[10]);
        } NN_FALL_THROUGH;
    case 10:
        {
            HW_SET_PMEVCNTR_EL0(9, pmevcntr_el0[9]);
            HW_SET_PMEVTYPER_EL0(9, pmevtyper_el0[9]);
        } NN_FALL_THROUGH;
    case 9:
        {
            HW_SET_PMEVCNTR_EL0(8, pmevcntr_el0[8]);
            HW_SET_PMEVTYPER_EL0(8, pmevtyper_el0[8]);
        } NN_FALL_THROUGH;
    case 8:
        {
            HW_SET_PMEVCNTR_EL0(7, pmevcntr_el0[7]);
            HW_SET_PMEVTYPER_EL0(7, pmevtyper_el0[7]);
        } NN_FALL_THROUGH;
    case 7:
        {
            HW_SET_PMEVCNTR_EL0(6, pmevcntr_el0[6]);
            HW_SET_PMEVTYPER_EL0(6, pmevtyper_el0[6]);
        } NN_FALL_THROUGH;
    case 6:
        {
            HW_SET_PMEVCNTR_EL0(5, pmevcntr_el0[5]);
            HW_SET_PMEVTYPER_EL0(5, pmevtyper_el0[5]);
        } NN_FALL_THROUGH;
    case 5:
        {
            HW_SET_PMEVCNTR_EL0(4, pmevcntr_el0[4]);
            HW_SET_PMEVTYPER_EL0(4, pmevtyper_el0[4]);
        } NN_FALL_THROUGH;
    case 4:
        {
            HW_SET_PMEVCNTR_EL0(3, pmevcntr_el0[3]);
            HW_SET_PMEVTYPER_EL0(3, pmevtyper_el0[3]);
        } NN_FALL_THROUGH;
    case 3:
        {
            HW_SET_PMEVCNTR_EL0(2, pmevcntr_el0[2]);
            HW_SET_PMEVTYPER_EL0(2, pmevtyper_el0[2]);
        } NN_FALL_THROUGH;
    case 2:
        {
            HW_SET_PMEVCNTR_EL0(1, pmevcntr_el0[1]);
            HW_SET_PMEVTYPER_EL0(1, pmevtyper_el0[1]);
        } NN_FALL_THROUGH;
    case 1:
        {
            HW_SET_PMEVCNTR_EL0(0, pmevcntr_el0[0]);
            HW_SET_PMEVTYPER_EL0(0, pmevtyper_el0[0]);
        } NN_FALL_THROUGH;
    default:
    case 0:
        break;
    }

    HW_SET_PMUSERENR_EL0(pmuserenr_el0);
    HW_SET_PMSELR_EL0(pmselr_el0);
    HW_SET_PMCCFILTR_EL0(pmccfiltr_el0);
    HW_SET_PMCNTENSET_EL0(pmcntenset_el0);
    HW_SET_PMINTENSET_EL1(pmintenset_el1);
    HW_SET_PMOVSSET_EL0(pmovsset_el0);
    HW_SET_PMCCNTR_EL0(pmccntr_el0);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    HW_SET_PMCR_EL0(pmcr_el0);
    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();

    // レジスタ復帰
    HW_SET_TTBR0_EL1(ttbr0);
    HW_SET_TCR_EL1(tcr);
    HW_SET_TPIDR_EL0(tpidr_el0);
    HW_SET_ELR_EL1(elr_el1);
    HW_SET_SP_EL0(sp_el0);
    HW_SET_SPSR_EL1(spsr_el1);
    HW_SET_DAIF(daif);
    HW_SET_CPACR_EL1(cpacr_el1);
    HW_SET_VBAR_EL1(vbar_el1);
    HW_SET_CSSELR_EL1(csselr_el1);
    HW_SET_CNTP_CTL_EL0(cntp_ctl_el0);
    HW_SET_CNTP_CVAL_EL0(cntp_cval_el0);
    HW_SET_CNTKCTL_EL1(cntkctl_el1);
    HW_SET_THREAD_ID_USER_READ_ONLY(tpidrro_el0);

    KCPU::DataSynchronizationBarrier();
    KCPU::InstructionMemoryBarrier();
    KCPU::InvalidateEntireTlb();
}

void WakeUpCore(int coreNo, KPhysicalAddress entryPhys, Bit64 contexId)
{
    // PSCI CPU On
    //  Target CPU
    //      Bits [40:63]: Must be zero
    //      Bits [32:39] Aff3 : Match Aff3 of target core MPIDR
    //      Bits [24:31] Must be zero
    //      Bits [16:23] Aff2 : Match Aff2 of target core MPIDR
    //      Bits [8:15] Aff1 : Match Aff1 of target core MPIDR
    //      Bits [0:7] Aff0 : Match Aff0 of target core MPIDR

    // 同じクラスタの CPU1～CPU3をOnにする
    Bit64 mpidr;
    HW_GET_MPIDR_EL1(mpidr);
    Bit64 mpidrAttr = (mpidr & ~(HW_MPIDR_EL1_AFF0_MASK | 0xFFFFFF00FF000000ul));

    smc::CpuOn(mpidrAttr | coreNo, GetAsInteger(entryPhys), contexId);
}

void WaitForSubCoreOff()
{
    uint64_t PMC_PWRGATE_STATUS = NN_KERN_P_ADDR_PMC_REGISTER + 0x38;
    enum
    {
        PMC_PWRGATE_STATUS_CE123_MASK = (7 << 9), // bit mask for bit 9, 10, 11
    };

    Bit32 v;
    do
    {
        smc::ReadWriteRegister(&v, PMC_PWRGATE_STATUS, 0, 0);
    }
    while ((v & PMC_PWRGATE_STATUS_CE123_MASK) != 0);
}

void SleepManagerThread(uintptr_t sleepBuffer)
{
    int coreNo = GetCurrentCpuNo();
    KPhysicalAddress sleepBufferPhys;
    KPhysicalAddress resumeEntryPhys;
    Result result;

    {
        NN_KERN_ABORT_UNLESS(Kernel::GetKernelPageTable().GetPhysicalAddress(&sleepBufferPhys, KProcessAddress(sleepBuffer)));
        g_SleepBufferPhyscalAddress[coreNo] = sleepBufferPhys;
        NN_KERN_ABORT_UNLESS(Kernel::GetKernelPageTable().GetPhysicalAddress(&resumeEntryPhys, KProcessAddress(&nn::kern::KSystemControl::ResumeEntry)));
    }

    for (;;)
    {
        {
            KScopedLightLock locker(&g_SyncMutex);
            while ((g_SleepProcessCpus & (1ull << coreNo)) == 0)
            {
                g_SyncCv.Wait(&g_SyncMutex, -1);
            }
        }

        {
            Bit64 tcr;

            KDisableInterrupt di;

            // レジスタ保存
            g_CpuRegisters[coreNo].Save();

            HW_GET_TCR_EL1(tcr);
            HW_SET_TCR_EL1(tcr | HW_TCR_EPD0);
            KCPU::DataSynchronizationBarrier();
            KCPU::InstructionMemoryBarrier();

            HW_SET_TTBR0_EL1(Kernel::GetKernelPageTable().GetTtbr0(coreNo));
            KCPU::DataSynchronizationBarrier();
            KCPU::InstructionMemoryBarrier();
            HW_SET_TCR_EL1(tcr & ~HW_TCR_EPD0);
            KCPU::DataSynchronizationBarrier();
            KCPU::InstructionMemoryBarrier();

            KCPU::InvalidateEntireTlb();

            KCPU::SynchronizeAllCore();

            if (coreNo == 0)
            {
                KDevicePageTable::Sleep();
            }

            KCPU::SynchronizeAllCore();

            Kernel::GetInterruptManager().Save(coreNo);

            NN_LOG("%s:%d Sleep Core%d ARG=%p\n", __func__,__LINE__, coreNo, GetAsInteger(sleepBufferPhys));

            KCPU::SynchronizeAllCore();
            if (coreNo != 0)
            {
                nn::kern::KSystemControl::CpuSleepHandler(GetAsInteger(sleepBufferPhys), GetAsInteger(resumeEntryPhys));
            }
            else
            {
                WaitForSubCoreOff();

                NN_LOG("%s:%d Enter Sc7\n", __func__,__LINE__);
                SavePrintRegisters();
                nn::kern::KSystemControl::CpuSleepHandler(GetAsInteger(sleepBufferPhys), GetAsInteger(resumeEntryPhys));
                RestorePrintRegisters();
                NN_LOG("%s:%d Leave Sc7\n", __func__,__LINE__);

                // cpu 1～3を復帰
                for (int i = 1; i < KCPU::NUM_CORE; i++)
                {
                    WakeUpCore(i, resumeEntryPhys, GetAsInteger(g_SleepBufferPhyscalAddress[i]));
                }
            }

            NN_LOG("%s:%d Running Core%d\n", __func__,__LINE__, coreNo);

            KCPU::SynchronizeAllCore();
            Kernel::GetInterruptManager().Restore(coreNo);
            KCPU::SynchronizeAllCore();

            if (coreNo == 0)
            {
                KDevicePageTable::Wakeup();
            }

            KCPU::SynchronizeAllCore();

            // レジスタ復帰
            g_CpuRegisters[coreNo].Restore();
        }

        {
            KScopedLightLock locker(&g_SyncMutex);
            g_SleepProcessCpus &= ~(1ull << coreNo);
            if (g_SleepProcessCpus == 0)
            {
                g_SyncCv.Broadcast();
            }
        }
    }
}
}

void Initialize()
{
    static Bit64 g_SleepBuffer[KCPU::NUM_CORE][128]  __attribute__ ((aligned(16)));
    for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
    {
        NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_ThreadCountMax, 1));

        const int priority = 2;

        KThread* pThread = KThread::Create();
        NN_KERN_ABORT_UNLESS(pThread != nullptr);
        NN_KERN_ABORT_IF_FAILED(InitializeThreadForKernel(pThread, &SleepManagerThread, reinterpret_cast<uintptr_t>(&g_SleepBuffer[coreNo][0]), priority, coreNo));
        KThread::Register(pThread);
        pThread->Run();
    }
}

void SleepSystem()
{
    KDevicePageTable::Lock();

    {
        KScopedLightLock locker(&g_Mutex);

        {
            KScopedLightLock locker(&g_SyncMutex);
            NN_KERN_ABORT_UNLESS(g_SleepProcessCpus == 0);

            for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
            {
                g_SleepProcessCpus |= (1ull << coreNo);
            }
            g_SyncCv.Broadcast();

            while (g_SleepProcessCpus != 0)
            {
                g_SyncCv.Wait(&g_SyncMutex, -1);
            }
        }
    }

    KDevicePageTable::Unlock();
}

}}}}

