﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>

/**
 * @file This file contains private and local resolver options
 */

namespace nn { namespace socket {

/**
 * @brief This enum class defines a set of @ref ResolverOptionKey
 * value for Get, Set, and Request functions but are not part of
 * the public SDK.
 */
typedef enum nnsocketResolverOptionPrivateKey
{

    /**
     * @brief
     * This constant defines the minimum boundary for Private key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_GetMinimumValue = 40000,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverGetOption ResolverGetOption and, if enabled
     * for the build, this function gets the current set of DNS
     * addresses.
     *
     *@details
     * To use this functionality you must set the pointer value to
     * a structure of @ref PrivateDnsAddressArrayArgument.
     *
     * @code
     * struct PrivateDnsAddressArrayArgument argument = { 0 };
     * ResolverOption option = {
     *     .key = ResolverOptionKey::GetDnsServerAddressesPointer,
     *     .type = ResolverOptionType_Pointer,
     *     .size = sizeof(struct PrivateSetTimeToLiveForHostnameArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * nn::socket::
     * GetResolverOption(ResolverOptionKey::GetDnsServerAddressesPointer,
     *                   option);
     * @endcode
     */
    nnsocketResolverOptionPrivateKey_GetDnsServerAddressesPointer = 40001,

    /**
     * @brief
     * This constant option is passed to ResolverGetOption and, if
     * enabled for the build. You pass a hostname string in and, on
     * success, the function return an integer representing the count.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_GetCacheEntryCountForHostnameString = 40002,

    /**
     * @brief
     * This constant option is passed to ResolverGetOption and, if
     * enabled for the build. You pass a uint32_t in and, on
     * success, the function returns an integer representing the count.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_GetCacheEntryCountForIpUnsigned32 = 40003,

    /**
     * @brief
     * This constant defines the maximum boundary for Private key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_GetMaximumValue = 40004,


    /**
     * @brief
     * This constant defines the minimum boundary for Private key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_SetMinimumValue = 50000,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverGetOption ResolverGetOption and, if enabled
     * for the build, this function gets the current set of DNS
     * addresses.
     *
     *@details
     * To use this functionality you must set the pointer value to
     * a structure of @ref PrivateDnsAddressArrayArgument.
     *
     *@code
     * nn::socket::InetPton(AF_INET, "1.2.3.4", &argument.addresses.s_addr[0]);
     * argument.addresses[0].sin_port = nn::socket::InetHtons(53);
     *
     * ResolverOption option = {
     *     .key = ResolverOptionKey::GetDnsServerAddressesPointer,
     *     .type = ResolverOptionType_Pointer,
     *     .size = sizeof(struct PrivateSetTimeToLiveForHostnameArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * // then make a resolver call such as:
     * nn::socket::
     * GetResolverOption(ResolverOptionKey::GetDnsServerAddressesPointer,
     *                   option);
     * @endcode
     */
    nnsocketResolverOptionPrivateKey_SetDnsServerAddressesPointer = 50001,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverSetOption ResolverSetOption and, if enabled
     * for the build, this function sets the current set of DNS
     * addresses.
     *
     *@details
     * To use this functionality you must set the pointer value to
     * a structure of @ref PrivateSetTimeToLiveForHostnameArgument
     * PrivateSetTimeToLiveForHostnameArgument.
     *
     * @code
     * const char* host = "www.nintendo.com";
     * struct PrivateSetTimeToLiveForHostnameArgument argument;
     * argument.ttl = 60;
     * memset(argument.hostname, host, strlen(host)+1);
     * ResolverOption option = {
     *     .key = ResolverOptionKey::PrivateSetTimeToLiveForHostnamePointer,
     *     .type = ResolverOptionType_Pointer,
     *     .size = sizeof(struct PrivateSetTimeToLiveForHostnameArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * nn::socket::SetResolverOption(option);
     * @endcode
     */
    nnsocketResolverOptionPrivateKey_SetTimeToLiveForHostnamePointer = 50002,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverSetOption ResolverSetOption and, if enabled
     * for the build, it will flush the entire DNS cache.
     *
     * @code
     * ResolverOption option = {
     *     .key = ResolverOptionKey::PrivateSetFlushCacheBoolean,
     *     .type = ResolverOptionType_Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = true
     * };
     *
     * nn::socket::SetResolverOption(option);
     * @endcode
     */
    nnsocketResolverOptionPrivateKey_SetFlushCacheBoolean = 50003,

    /**
     * @brief
     * This constant defines the maximum boundary for Private key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_SetMaximumValue = 50004,

    /**
     * @brief
     * This constant defines the minimum boundary for Local key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_RequestMinimumValue = 60000,

    /**
     * @brief
     * This constant defines the maximum boundary for Local key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionPrivateKey_RequestMaximumValue = 60001,
} nnsocketResolverOptionPrivateKey;

typedef enum nnsocketResolverOptionLocalKey
{
    /**
     * @brief
     * This constant defines the minimum boundary for Local key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionLocalKey_MinimumValue = 70000,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * ResolverGetOption and @ref ResolverSetOption ResolverSetOption.
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetHostByName GetHostByName.
     */
    nnsocketResolverOptionLocalKey_GetHostByNameBufferSizeUnsigned64  = 70001,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * ResolverGetOption and @ref ResolverSetOption ResolverSetOption.
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetHostByAddr.
     */
    nnsocketResolverOptionLocalKey_GetHostByAddrBufferSizeUnsigned64  = 70002,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * ResolverGetOption and @ref ResolverSetOption. The data value
     * parameter is an unsigned 64-bit value (uint64_t) and that that
     * value is used to determine the size in bytes of the service
     * framework buffer used with @ref HStrError
     */
    nnsocketResolverOptionLocalKey_GetHostErrorStringBufferSizeUnsigned64 = 70003,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * and @ref ResolverSetOption. The data value parameter is an
     * unsigned 64-bit value (uint64_t) and that that value is used to
     * determine the size in bytes of the service framework buffer used
     * for @ref GAIStrError
     */
    nnsocketResolverOptionLocalKey_GaiErrorStringBufferSizeUnsigned64  = 70004,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * ResolverGetOption and @ref ResolverSetOption ResolverSetOption.
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetAddrInfo GetAddrInfo.
     */
    nnsocketResolverOptionLocalKey_GetAddrInfoBufferSizeUnsigned64 = 70005,

    /**
     * @brief This constant is used with @ref ResolverGetOption
     * ResolverGetOption and @ref ResolverSetOption ResolverSetOption.
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetAddrInfo GetAddrInfo
     * 'hints' parameter buffer size.
     */
    nnsocketResolverOptionLocalKey_GetAddrInfoHintsBufferSizeUnsigned64 = 70006,

    /**
     * @brief
     * This constant defines the maximum boundary for Local key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionLocalKey_MaximumValue = 70007,
} nnsocketResolverOptionLocalKey;

/**
 * @brief this global constant sets the maximum length of a hostname
 * string used with @ref PrivateSetTimeToLiveForHostnameArgument
 * PrivateSetTimeToLiveForHostnameArgument.
 */
const unsigned nnsocketMaximumHostnameStringLength = 255;

/**
 * @brief This structure is the private parameter used with @ref
 * ResolverSetOption ResolverSetOption to update the TTL for all
 * answer records matching the hostname provided.
 */
struct nnsocketPrivateSetTimeToLiveForHostnameArgument
{
    uint32_t ttl;
    char hostname[MaximumHostnameStringLength];
};

/**
 * @brief This global constant sets the maximum number of addresses
 * that might be in the resolver DNS address array.
 */
const unsigned nnsocketMaximumDnsHosts = 2;

/**
 * @brief This structure is the private parameter used with @ref
 * ResolverGetOption ResolverGetOption and @ref ResolverSetOption
 * ResolverSetOption to set or get a copy of the resolver DNS address
 * array.
 */
struct nnsocketPrivateDnsAddressArrayArgument
{
    uint32_t count;
    struct sockaddr_in addresses[MaximumDnsHosts];
};

}}; // nn::socket
