﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   イベントに関する C 言語用 API の宣言
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_EventCommon.h>
#include <nnc/os/os_EventTypes.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

//---------------------------------------------------------------------------
//  C 向けの宣言
//---------------------------------------------------------------------------

//! @name イベント関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeEvent(nnosEventType* event, bool initiallySignaled, nnosEventClearMode autoClear);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトを破棄します。
 *
 * @details
 *  詳細は nn::os::FinalizeEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosFinalizeEvent(nnosEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトがシグナル状態になるまで待機します。
 *
 * @details
 *  詳細は nn::os::WaitEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosWaitEvent(nnosEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトがシグナル状態かをポーリングします。
 *
 * @details
 *  詳細は nn::os::TryWaitEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool nnosTryWaitEvent(nnosEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトがシグナル状態になるまで時限付きで待機します。
 *
 * @details
 *  詳細は nn::os::TimedWaitEvent() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool nnosTimedWaitEvent(nnosEventType* event, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトを即座にシグナル状態にします。
 *
 * @details
 *  詳細は nn::os::SignalEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSignalEvent(nnosEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosEventType オブジェクトを非シグナル状態にクリアします。
 *
 * @details
 *  詳細は nn::os::ClearEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosClearEvent(nnosEventType* event);

//! @}


//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、event と関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, EventType* event) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForEvent(nnosMultiWaitHolderType* multiWaitHolder, nnosEventType* event);

//! @}

