﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向けアドレス変換に関する C 言語用公開ヘッダファイル
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/nn_Result.h>
#include <nnc/dd/dd_Macro.h>

//--------------------------------------------------------------------------
//  for C
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
/**
 * @brief   物理メモリアドレスを表す型です
 */
typedef uint64_t nnddPhysicalAddress;

//--------------------------------------------------------------------------
/**
 * @brief   仮想メモリ領域に対応する物理メモリ領域を取得するための構造体です
 */
typedef struct nnddPhysicalMemoryInfo
{
    nnddPhysicalAddress   physicalAddress;
    uintptr_t             virtualAddress;
    size_t                size;

    uintptr_t             _endVirtualAddress;

} nnddPhysicalMemoryInfo;

//! @name アドレス変換関連 API
//! @{

/**
 * @brief  PhysicalMemoryInfo オブジェクトを初期化します
 *
 * @details
 *  詳細は nn::dd::InitializePhysicalMemoryInfo() を参照してください。
 */
NN_DD_EXTERN_C void nnddInitializePhysicalMemoryInfo( nnddPhysicalMemoryInfo* pOutInfo, const void* addr, size_t size );

/**
 * @brief   仮想メモリ領域に対応する物理メモリ情報を得ます
 *
 * @details
 *  詳細は nn::dd::QueryNextPhysicalMemoryInfo() を参照してください。
 */
NN_DD_EXTERN_C nnResult nnddQueryNextPhysicalMemoryInfo( nnddPhysicalMemoryInfo* pOutInfo );

/**
 * @brief  単体領域で管理されている物理メモリ領域の物理アドレスを得ます
 *
 * @details
 *  詳細は nn::dd::QuerySinglePhysicalAddress() を参照してください。
 */
NN_DD_EXTERN_C nnResult nnddQuerySinglePhysicalAddress( nnddPhysicalAddress* pOutAddress, const void* addr, size_t size );

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域の物理アドレスを得ます
 *
 * @details
 *  詳細は nn::dd::GetPhysicalAddress() を参照してください。
 */
NN_DD_EXTERN_C inline nnddPhysicalAddress nnddGetPhysicalAddress( const nnddPhysicalMemoryInfo* pmi )
{
    return pmi->physicalAddress;
}

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域に対応する仮想アドレスを得ます
 *
 * @details
 *  詳細は nn::dd::GetVirtualAddress() を参照してください。
 */
NN_DD_EXTERN_C inline const void* nnddGetVirtualAddress( const nnddPhysicalMemoryInfo* pmi )
{
    return (void*)(pmi->virtualAddress);
}

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域のサイズを得ます
 *
 * @details
 *  詳細は nn::dd::GetSize() を参照してください。
 */
NN_DD_EXTERN_C inline size_t nnddGetSize( const nnddPhysicalMemoryInfo* pmi )
{
    return pmi->size;
}

//! @}
