﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <type_traits>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_MemoryFence.h>

namespace nn { namespace util {

/**
 * @brief   値の読み書きをロックなしにアトミックに行うための型です。
 * @tparam  T   扱う型
 *
 * @details
 *  値の読み込みにおいて、書き込み途中でない正しい値が取得できることを保証します。
 *
 *  値の書き込みを複数スレッドで行うには、別途排他処理が必要な点に注意してください。
 *
 *  本クラスは、ミューテックスや std::atomic による排他制御を行っておらず、
 *  プロセスをまたぐ共有メモリの読み書きなどにも利用可能です。
 *  2つのプロセス間において、片方が値を書き込み、もう一方がその値を安全に読み込むことが可能です。
 */
template <typename T>
struct LockFreeAtomicType
{
    volatile uint32_t _counter;
    T _value[2];

#if !defined( NN_BUILD_CONFIG_COMPILER_GCC )
    NN_STATIC_ASSERT(std::is_trivially_copyable<T>::value);
#endif
};

/**
 * @brief   値を書き込みます。
 *
 * @param[in]   p       書き込み先となるLockFreeAtomicTypeを指すポインタ
 * @param[in]   value   書き込む値
 *
 * @details
 *  LoadFromLockFreeAtomicType() に対してはスレッドセーフです。
 *
 *  StoreToLockFreeAtomicType() に対してはスレッドアンセーフです。
 *  よって、StoreToLockFreeAtomicType() を複数スレッドで実行するには、別途排他制御が必要です。
 */
template <typename T>
void StoreToLockFreeAtomicType(LockFreeAtomicType<T>* p, const T& value) NN_NOEXCEPT
{
    auto counter = p->_counter;
    counter++;
    p->_value[counter % 2] = value;

    nn::os::FenceMemoryStoreStore();

    p->_counter = counter;
}

/**
 * @brief   値を読み込みます。
 *
 * @param[in]   p   読み込み元となるLockFreeAtomicTypeを指すポインタ
 *
 * @details
 *  StoreToLockFreeAtomicType() を行ったことがない場合、返る値は不定です。
 *
 *  本関数はスレッドセーフです。
 */
template <typename T>
T LoadFromLockFreeAtomicType(const LockFreeAtomicType<T>* p) NN_NOEXCEPT
{
    for(;;)
    {
        auto counter = p->_counter;
        auto ret = p->_value[counter % 2];

        nn::os::FenceMemoryLoadLoad();

        if(counter == p->_counter)
        {
            return ret;
        }
    }
}

}}
