﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <nn/nn_Result.h>
#include <nn/socket.h>

 /**
 * @file
 * @brief TSC Arp API
 *
 * @details This file defines the public TSC
 *          arp API (C++ language).
 */

namespace nn { namespace tsc {

    /**
    *   @brief      Add a static arp entry for the provided address
    *   @details    Adds a static arp entry for the provided address. Will replace any previously existing entry.
    *   @pre        There is an active interface.
    *
    *   @param[in]  ipAddress                Address for which to add an arp entry
    *   @param[in]  pInputHardwareAddress    Pointer to the 6 byte hardware address to associate with the provided ip address.
    *   @param[in]  hardwareAddressSize      Size of the hardware address buffer. Should be 6 bytes.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultInvalidAddressSize           Passed hardware address size is invalid (not 6 bytes)
    *   @retval     ResultIpv4AddressInvalid           Passed Ip or hardware address is invalid. (Non-standard address or address is not on subnet)
    *   @retval     ResultIfRemoved                    An interface does not exist
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result AddStaticArpEntry(nn::socket::InAddr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize);
    Result AddStaticArpEntry(in_addr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize);

    /**
    *   @brief      Remove an arp entry with the provided address
    *   @details    Removes an arp entry with the provided address.
    *   @pre        There is an active interface.
    *
    *   @param[in]  ipAddress                Address of arp entry to remove.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
    *   @retval     ResultIfRemoved                    An interface does not exist
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result RemoveArpEntry(nn::socket::InAddr ipAddress);
    Result RemoveArpEntry(in_addr ipAddress);

    /**
    *   @brief      Look up the arp entry with the provided ip address.
    *   @details    looks up an arp entry with the provided ip address, and returns the associated hardware address.
    *   @pre        There is an active interface.
    *
    *   @param[out] pOutputHardwareAddress  Pointer to the buffer to fill with the looked up arp hardware address.
    *   @param[in]  hardwareAddressSize     Size of the provided hardware address buffer.
    *   @param[in]  ipAddress               Address of arp entry to look up.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
    *   @retval     ResultIfRemoved                    An interface does not exist
    *   @retval     ResultInvalidAddressSize           Provided size of the hardware address buffer is not valid.
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, nn::socket::InAddr ipAddress);
    Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, in_addr ipAddress);

    /**
    *   @brief      Look up the arp entry with the provided hardware address.
    *   @details    looks up an arp entry with the provided hardware address, and returns the associated ip address.
    *   @pre        There is an active interface.
    *
    *   @param[out] pOutputIPAddress    Pointer to the buffer to fill with the looked up arp ip address.
    *   @param[in]  ipAddressSize       Size of the provided ip address buffer.
    *   @param[in]  pHardwareAddress    Pointer to the hardware address of the arp entry to look up.
    *   @param[in]  hardwareAddressSize Size of the provided hardware address buffer.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultArpEntryNotFound             Failed to find an arp entry for the provided IP Address
    *   @retval     ResultIfRemoved                    An interface does not exist
    *   @retval     ResultInvalidAddressSize           Provided size of the hardware or ip address buffer is not valid.
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result LookupArpEntry(nn::socket::InAddr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize);
    Result LookupArpEntry(in_addr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize);

    /**
    *   @brief      Clear all arp entries.
    *   @details    Clears all arp entries in the arp table.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result ClearArpEntries();

    /**
    *   @brief      Clear all arp entries for a given interface.
    *   @details    Clears all arp entries associated with the provided interface.
    *
    *   @param[in]  pIfName   Pointer to the interface name
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result ClearArpEntries(const char * ifName);

    /**
    *   @brief      Prints all arp entries.
    *   @details    Prints all arp entries in the arp table.
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultInvalidInternalLogic         There was an unexpected error from cfg layer.
    */
    Result PrintArpEntries();

}}
