﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <stdint.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

//==============================================================================

namespace nn { namespace tma {  namespace file_io {
        enum OpenMode
        {
            OpenMode_Read = 1 << 0,
            OpenMode_Write = 1 << 1,
        };

        typedef void*                       FileHandle;
        typedef void*                       DirectoryHandle;

        void Initialize();

        void Finalize();

        nn::Result  OpenFile                ( FileHandle*                   pOutHandle,
                                              const char*                   pPath,
                                              uint32_t                      openMode );

        //nn::fs0 signature
        nn::Result  ReadFile                ( int32_t*                      pOutValue,
                                              FileHandle                    handle,
                                              int64_t                       offset,
                                              void*                         pBuffer,
                                              int64_t                       size );

        //nn::fs signature
        nn::Result  ReadFile                ( size_t*                       pOutValue,
                                              FileHandle                    handle,
                                              int64_t                       offset,
                                              void*                         pBuffer,
                                              size_t                        size,
                                              const nn::fs::ReadOption&     option );

        //nn::fs0 signature
        nn::Result  WriteFile               ( int32_t*                      pOutValue,
                                              FileHandle                    handle,
                                              int64_t                       offset,
                                              const void*                   pBuffer,
                                              int64_t                       size,
                                              bool                          flush );

        //nn::fs signature
        nn::Result  WriteFile               ( FileHandle                    handle,
                                              int64_t                       offset,
                                              const void*                   pBuffer,
                                              int64_t                       size,
                                              const nn::fs::WriteOption&    option);

        nn::Result  GetFileSize             ( int64_t*                      pOutValue,
                                              FileHandle                    handle );

        nn::Result  SetFileSize             ( FileHandle                    handle,
                                              int64_t                       size );

        nn::Result GetFileTimeStamp         ( uint64_t*                     pCreateTime,
                                              uint64_t*                     pAccessTime,
                                              uint64_t*                     pModifyTime,
                                              const char*                   pathName );

        nn::Result  FlushFile               ( FileHandle                    handle );

        nn::Result  SetPriorityForFile      ( FileHandle                    handle,
                                              int32_t                       priority );

        nn::Result  GetPriorityForFile      ( int32_t*                      pOutValue,
                                              FileHandle                    handle );

        void        CloseFile               ( FileHandle                    handle );

        nn::Result  FileExists              ( bool*                         pOutValue,
                                              const char*                   pPath );

        nn::Result  DeleteFile              ( const char*                   pPath );

        nn::Result  RenameFile              ( const char*                   pFromName,
                                              const char*                   pToName );

        nn::Result  GetIOType               ( nn::fs::DirectoryEntryType*   pOutType,
                                              const char*                   pPath );

        nn::Result  OpenDirectory           ( DirectoryHandle*          pOutHandle,
                                              const char*               pathName,
                                              fs::OpenDirectoryMode     OpenMode = fs::OpenDirectoryMode_File );

        nn::Result  GetEntryCount           ( int64_t*                  pOutValue,
                                              DirectoryHandle           handle );

        nn::Result  ReadDirectory           ( int64_t*                  pOutValue,
                                              nn::fs::DirectoryEntry    pEntries[],
                                              DirectoryHandle           handle,
                                              int64_t                   numEntries );

        nn::Result  SetPriorityForDirectory ( DirectoryHandle           handle,
                                              int32_t                   priority );

        nn::Result  GetPriorityForDirectory ( int32_t*                  pOutValue,
                                              DirectoryHandle           handle );

        void        CloseDirectory          ( DirectoryHandle           handle );

        nn::Result  DirectoryExists         ( bool*                     pOutValue,
                                              const char*               pathName );

        nn::Result  CreateDirectory         ( const char*               pathName );

        nn::Result  DeleteDirectory         ( const char*               pathName,
                                              bool                      recursively );

        nn::Result  RenameDirectory         ( const char*               fromName,
                                              const char*               ToName );

        nn::Result  CreateFile              ( const char*               pPath,
                                              int64_t                   size );

//==============================================================================
}}} // namespace
//==============================================================================
