﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/spl/spl_Types.h>

namespace nn { namespace spl {

    namespace detail
    {
        const char PortName[]       = "spl:";
        const char PortNameRandom[] = "csrng";
        const char PortNameCrypto[] = "spl:mig";
        const char PortNameFs[]     = "spl:fs";
        const char PortNameSsl[]    = "spl:ssl";
        const char PortNameEs[]     = "spl:es";
        const char PortNameManu[]   = "spl:manu";
    }


    //! @name 初期化
    //! @{

    void Initialize() NN_NOEXCEPT;
    void InitializeForCrypto() NN_NOEXCEPT;
    void InitializeForFs() NN_NOEXCEPT;
    void InitializeForSsl() NN_NOEXCEPT;
    void InitializeForEs() NN_NOEXCEPT;
    void InitializeForManu() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    //! @}


    //! @name RSA
    //! @{

    Result ModularExponentiate(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pBase,
        size_t      baseSize,
        const void* pExponent,
        size_t      exponentSize,
        const void* pModulus,
        size_t      modulusSize ) NN_NOEXCEPT;

    //! @}


    //! @name AES
    //! @{

    Result AllocateAesKeySlot(
        int* pSlotIndex ) NN_NOEXCEPT;

    Result DeallocateAesKeySlot(
        int slotIndex ) NN_NOEXCEPT;

    Result GenerateAesKek(
        AccessKey*          pAccessKey,
        const void*         pKeySource,
        size_t              keySourceSize,
        int                 generation,
        int                 option ) NN_NOEXCEPT;

    Result LoadAesKey(
        int                 slotIndex,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT;

    Result GenerateAesKey(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT;

    Result GenerateSpecificAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       purpose  ) NN_NOEXCEPT;

    Result ComputeCtr(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize,
        const void* pInitialCounter,
        size_t      initialCounterSize ) NN_NOEXCEPT;

    Result ComputeCmac(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize ) NN_NOEXCEPT;

    Result ExtractDrmDeviceCertEccKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT;

    Result ExtractSslClientCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT;

    Result ExtractNfpEccBn128Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT;

    Result ExtractNfpEccP160Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT;

    Result DecryptAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       option  ) NN_NOEXCEPT;

    Result ReencryptNfpEccBn128Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptNfpEccP160Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptDrmDeviceCertEccKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptGcKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptEsDeviceKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptSslClientCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    Result ReencryptDrmDeviceCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT;

    //! @}


    //! @name システム設定
    //! @{

    Result GetConfig(
        Bit64*      pOut,
        ConfigItem  key ) NN_NOEXCEPT;

    Result SetConfig(
        ConfigItem  key,
        Bit64       value ) NN_NOEXCEPT;

    bool IsDevelopment() NN_NOEXCEPT;

    inline bool GetConfigBool(ConfigItem key) NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, key) );
        return v != 0;
    }

    inline HardwareType GetHardwareType() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_HardwareType) );
        return static_cast<HardwareType>(v);
    }

    inline HardwareState GetHardwareState() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_HardwareState) );
        return static_cast<HardwareState>(v);
    }

    inline DramId GetDramId() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_DramId) );
        return static_cast<DramId>(v);
    }

    // Device ID の下位 56 ビットを取得
    inline Bit64 GetDeviceIdLow() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_DeviceId) );
        return v;
    }

    inline RetailInteractiveDisplayState GetRetailInteractiveDisplayState() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_RetailInteractiveDisplayState) );
        return static_cast<RetailInteractiveDisplayState>(v);
    }

    inline Regulator GetRegulator() NN_NOEXCEPT
    {
        Bit64 v;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetConfig(&v, ConfigItem_Regulator) );
        return static_cast<Regulator>(v);
    }

    Result SetBootReason(BootReasonValue bootReason) NN_NOEXCEPT;

    Result GetBootReason(BootReasonValue* pOut) NN_NOEXCEPT;

    inline BootReasonValue GetBootReason() NN_NOEXCEPT
    {
        BootReasonValue bootReason;
        NN_ABORT_UNLESS_RESULT_SUCCESS( GetBootReason(&bootReason) );
        return bootReason;
    }

    // 初期プロセスなど settings を利用できないプロセスのみが利用
    // それ以外のプロセスでの利用は禁止
    SocType GetSocType() NN_NOEXCEPT;

    Result GetPackage2Hash(
        void*       pOutBuffer,
        size_t      bufferSize) NN_NOEXCEPT;

    //! @}


    //! @name その他
    //! @{

    Result  GenerateRandomBytes(
        void*   pOutBuffer,
        size_t  bufferSize ) NN_NOEXCEPT;

    Result  DecryptAndStoreGcKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT;

    Result  DecryptGcMessage(
        size_t*     pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize) NN_NOEXCEPT;

    Result DecryptAndStoreSslClientCertKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT;

    Result ModularExponentiateWithSslClientCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT;

    Result DecryptAndStoreDrmDeviceCertKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT;

    Result ModularExponentiateWithDrmDeviceCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT;

    Result LoadEsDeviceKey(
        const void*      pData,
        size_t           dataSize) NN_NOEXCEPT;

    Result PrepareEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT;

    Result PrepareEsArchiveKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT;

    Result PrepareCommonEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pKeySource,
        size_t           keySourceSize,
        int              generation) NN_NOEXCEPT;

    Result LoadPreparedAesKey(
        int              slotIndex,
        const AccessKey& accessKey) NN_NOEXCEPT;

    //! @}



}}  // namespace nn::spl

