﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Types.h>

namespace nn {
namespace socket {

typedef uint32_t TcpSeq;

/*
 * TCP header.
 * Per RFC 793, September, 1981.
 */
enum class TcpHdrFlag : uint8_t
{
    Th_Fin    = 0x01,
    Th_Syn    = 0x02,
    Th_Rst    = 0x04,
    Th_Push   = 0x08,
    Th_Ack    = 0x10,
    Th_Urg    = 0x20,
    Th_Ece    = 0x40,
    Th_Cwr    = 0x80,
    Th_Flags  = (Th_Fin|Th_Syn|Th_Rst|Th_Push|Th_Ack|Th_Urg|Th_Ece|Th_Cwr)
};

const char PrintThFlags[] = "\20\1FIN\2SYN\3RST\4PUSH\5ACK\6URG\7ECE\10CWR";

struct TcpHdr
{
    uint16_t    th_sport;               /* source port */
    uint16_t    th_dport;               /* destination port */
    TcpSeq      th_seq;                 /* sequence number */
    TcpSeq      th_ack;                 /* acknowledgement number */
#if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE )
    uint8_t     th_x2:4,                /* (unused) */
                th_off:4;               /* data offset */
#else
    uint8_t     th_off:4,               /* data offset */
                th_x2:4;                /* (unused) */
#endif
    TcpHdrFlag  th_flags;
    uint16_t    th_win;                 /* window */
    uint16_t    th_sum;                 /* checksum */
    uint16_t    th_urp;                 /* urgent pointer */
};

enum class TcpOpt : uint32_t
{
    TcpOpt_Eol              = 0,
    TcpOpt_Pad              = 0,        /* padding after EOL */
    TcpOpt_Nop              = 1,
    TcpOpt_MaxSeg           = 2,
    TcpOpt_Window           = 3,
    TcpOpt_Sack_Permitted   = 4,
    TcpOpt_Sack             = 5,
    TcpOpt_Timestamp        = 8,
    TcpOpt_Signature        = 19       /* Keyed MD5: RFC 2385 */
};

const uint32_t TcpOlen_Eol = 1;
const uint32_t TcpOlen_Pad = 1;
const uint32_t TcpOlen_Nop = 1;
const uint32_t TcpOlen_MaxSeg = 4;
const uint32_t TcpOlen_Window = 3;
const uint32_t TcpOlen_Sack_Permitted = 2;
const uint32_t TcpOlen_SackHdr = 2;
const uint32_t TcpOlen_Sack = 8;        /* 2*sizeof(TcpSeq) */
const uint32_t TcpOlen_Timestamp = 10;
const uint32_t TcpOlen_Tstamp_Appa = (TcpOlen_Timestamp + 2); /* appendix A */
const uint32_t TcpOlen_Signature = 18;

typedef TcpSeq Tcp6Seq;
typedef TcpHdr Tcp6Hdr;

/* Miscellaneous constants */
const uint32_t Max_Sack_Blks  = 6;        /* Max # SACK blocks stored at receiver side */
const uint32_t Tcp_Max_Sack   = 4;        /* MAX # SACKs sent in any segment */

/*
 * The default maximum segment size (MSS) to be used for new TCP connections
 * when path MTU discovery is not enabled.
 *
 * RFC879 derives the default MSS from the largest datagram size hosts are
 * minimally required to handle directly or through IP reassembly minus the
 * size of the IP and TCP header.  With IPv6 the minimum MTU is specified
 * in RFC2460.
 *
 * For IPv4 the MSS is 576 - sizeof(TcpIpHdr)
 * For IPv6 the MSS is IPV6_MMTU - sizeof(Ip6Hdr) - sizeof(TcpHdr)
 *
 * We use explicit numerical definition here to avoid header pollution.
 */
const uint32_t Tcp_Mss   = 536;
const uint32_t Tcp6_Mss  = 1220;

/*
 * Limit the lowest MSS we accept for path MTU discovery and the TCP SYN MSS
 * option.  Allowing low values of MSS can consume significant resources and
 * be used to mount a resource exhaustion attack.
 * Connections requesting lower MSS values will be rounded up to this value
 * and the IP_DF flag will be cleared to allow fragmentation along the path.
 *
 * See tcp_subr.c tcp_minmss SYSCTL declaration for more comments.  Setting
 * it to "0" disables the minmss check.
 *
 * The default value is fine for TCP across the Internet's smallest official
 * link MTU (256 bytes for AX.25 packet radio).  However, a connection is very
 * unlikely to come across such low MTU interfaces these days (anno domini 2003).
 */
const uint32_t Tcp_MinMss           = 216;
const uint32_t Tcp_MaxWin           = 65535;        /* largest value for (unscaled) window */
const uint32_t Ttcp_Client_Snd_Wnd  = 4096;         /* dflt send window for T/TCP client */
const uint32_t Tcp_Max_WinShift     = 14;           /* maximum window shift */
const uint32_t Tcp_MaxBurst         = 4;            /* maximum segments in a burst */
const uint32_t Tcp_MaxHlen          = (0xf << 2);   /* max length of header in bytes */
const uint32_t Tcp_MaxOlen          = (Tcp_MaxHlen - sizeof(TcpHdr));  /* max space left for options */

}} /* nn::socket */
