﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace socket {

/*
 * Interface Control Message Protocol Definitions.
 * Per RFC 792, September 1981.
 */

/**
    Type of message.
*/
enum class IcmpType : uint8_t
{
    Icmp_EchoReply		    =  0,   ///< echo reply
    Icmp_Unreach            =  3,   ///< dest unreachable, codes:
    Icmp_SourceQuench	    =  4,   ///< packet lost, slow down
    Icmp_Redirect		    =  5,   ///< shorter route, codes:
    Icmp_AltHostAddr	    =  6,   ///< alternate host address
    Icmp_Echo               =  8,   ///< echo service
    Icmp_RouterAdvert       =  9,   ///< router advertisement
    Icmp_RouterSolicit      = 10,   ///< router solicitation
    Icmp_TimXceed           = 11,   ///< time exceeded, code:
    Icmp_ParamProb          = 12,   ///< ip header bad
    Icmp_Tstamp             = 13,   ///< timestamp request
    Icmp_TstampReply        = 14,   ///< timestamp reply
    Icmp_IReq               = 15,   ///< information request
    Icmp_IReqReply          = 16,   ///< information reply
    Icmp_MaskReq            = 17,   ///< address mask request
    Icmp_MaskReply          = 18,   ///< address mask reply
    Icmp_TraceRoute         = 30,   ///< traceroute
    Icmp_DataConvErr        = 31,   ///< data conversion error
    Icmp_Mobile_Redirect    = 32,   ///< mobile host redirect
    Icmp_IpV6_WhereAreYou   = 33,   ///< IPv6 where-are-you
    Icmp_IpV6_IAmHere       = 34,   ///< IPv6 i-am-here
    Icmp_Mobile_RegRequest  = 35,   ///< mobile registration req
    Icmp_Mobile_RegReply    = 36,   ///< mobile registration reply
    Icmp_Skip               = 39,   ///< SKIP
    Icmp_Photuris           = 40    ///< Photuris
};

/**
    Type sub code.
*/
enum class IcmpCode : uint8_t
{
    Icmp_Unreach_Net                    =  0,   ///< bad net
    Icmp_Unreach_Host                   =  1,   ///< bad host
    Icmp_Unreach_Protocol               =  2,   ///< bad protocol
    Icmp_Unreach_Port                   =  3,   ///< bad port
    Icmp_Unreach_NeedFrag               =  4,   ///< IP_DF caused drop
    Icmp_Unreach_SrcFail                =  5,   ///< src route failed
    Icmp_Unreach_Net_Unknown            =  6,   ///< unknown net
    Icmp_Unreach_Host_Unknown           =  7,   ///< unknown host
    Icmp_Unreach_Isolated               =  8,   ///< src host isolated
    Icmp_Unreach_Net_Prohib             =  9,   ///< prohibited access
    Icmp_Unreach_Host_Prohib            = 10,   ///< ditto
    Icmp_Unreach_TosNet                 = 11,   ///< bad tos for net
    Icmp_Unreach_TosHost                = 12,   ///< bad tos for host
    Icmp_Unreach_Filter_Prohib          = 13,   ///< admin prohib
    Icmp_Unreach_Host_Precedence        = 14,   ///< host prec vio.
    Icmp_Unreach_Precedence_Cutoff      = 15,   ///< prec cutoff
    Icmp_Redirect_Net                   =  0,   ///< for network
    Icmp_Redirect_Host                  =  1,   ///< for host
    Icmp_Redirect_TosNet                =  2,   ///< for tos and net
    Icmp_Redirect_TosHost               =  3,   ///< for tos and host
    Icmp_RouterAdvert_Normal            =  0,   ///< normal advertisement
    Icmp_RouterAdvert_NoRoute_Common    = 16,   ///< selective routing
    Icmp_TimXceed_InTrans               =  0,   ///< ttl==0 in transit
    Icmp_TimXceed_Reass                 =  1,   ///< ttl==0 in reass
    Icmp_ParamProb_ErrAtPtr             =  0,   ///< error at param ptr
    Icmp_ParamProb_OptAbsent            =  1,   ///< req. opt. absent
    Icmp_ParamProb_Length               =  2,   ///< bad length
    Icmp_Photuris_Unknown_Index         =  1,   ///< unknown sec index
    Icmp_Photuris_Auth_Failed           =  2,   ///< auth failed
    Icmp_Photuris_Decrypt_Failed        =  3    ///< decrypt failed
};

/*
 * Internal of an ICMP Router Advertisement
 */
struct IcmpRaAddr
{
    uint32_t ira_addr;
    uint32_t ira_preference;
};

/*
* Structure of an icmp header.
*/
struct IcmpHdr {
    IcmpType	icmp_type;		/* type of message, see below */
    IcmpCode	icmp_code;		/* type sub code */
    uint16_t	icmp_cksum;		/* ones complement cksum of struct */
};

struct Icmp
{
    IcmpType icmp_type;		        ///< type of message, see below
    IcmpCode icmp_code;		        ///< type sub code
    uint16_t icmp_cksum;		    ///< ones complement cksum of struct
    union {
        uint8_t ih_pptr;			///< ICMP_PARAMPROB
        InAddr ih_gwaddr;	        ///< ICMP_REDIRECT
        struct ih_idseq {
            uint16_t	icd_id;	    ///< network format
            uint16_t	icd_seq;    ///< network format
        } ih_idseq;
        int ih_void;

        /* ICMP_UNREACH_NEEDFRAG -- Path MTU Discovery (RFC1191) */
        struct ih_pmtu {
            uint16_t ipm_void;	    ///< network format
            uint16_t ipm_nextmtu;	///< network format
        } ih_pmtu;

        struct ih_rtradv {
            uint8_t irt_num_addrs;
            uint8_t irt_wpa;
            uint16_t irt_lifetime;
        } ih_rtradv;
    } icmp_hun;
#define	icmp_pptr	icmp_hun.ih_pptr
#define	icmp_gwaddr	icmp_hun.ih_gwaddr
#define	icmp_id		icmp_hun.ih_idseq.icd_id
#define	icmp_seq	icmp_hun.ih_idseq.icd_seq
#define	icmp_void	icmp_hun.ih_void
#define	icmp_pmvoid	icmp_hun.ih_pmtu.ipm_void
#define	icmp_nextmtu	icmp_hun.ih_pmtu.ipm_nextmtu
#define	icmp_num_addrs	icmp_hun.ih_rtradv.irt_num_addrs
#define	icmp_wpa	icmp_hun.ih_rtradv.irt_wpa
#define	icmp_lifetime	icmp_hun.ih_rtradv.irt_lifetime
    union {
        struct id_ts {          ///< ICMP Timestamp */
            /*
             * The next 3 fields are in network format,
             * milliseconds since 00:00 GMT
             */
            uint32_t its_otime; ///< Originate
            uint32_t its_rtime; ///< Receive
            uint32_t its_ttime; ///< Transmit
        } id_ts;
        struct id_ip  {
            Ip idi_ip;
            /* options and then 64 bits of data */
        } id_ip;
        IcmpRaAddr id_radv;
        uint32_t id_mask;
        char	id_data[1];
    } icmp_dun;
#define	icmp_otime	icmp_dun.id_ts.its_otime
#define	icmp_rtime	icmp_dun.id_ts.its_rtime
#define	icmp_ttime	icmp_dun.id_ts.its_ttime
#define	icmp_ip		icmp_dun.id_ip.idi_ip
#define	icmp_radv	icmp_dun.id_radv
#define	icmp_mask	icmp_dun.id_mask
#define	icmp_data	icmp_dun.id_data
};

/*
 * Lower bounds on packet lengths for various types.
 * For the error advice packets must first insure that the
 * packet is large enough to contain the returned ip header.
 * Only then can we do the check to see if 64 bits of packet
 * data have been returned, since we need to check the returned
 * ip header length.
 */
const uint32_t Icmp_MinLen      = 8;                              /* abs minimum */
const uint32_t Icmp_TsLen       = 8 + 3 * sizeof(uint32_t);       /* timestamp */
const uint32_t Icmp_MaskLen     = 12;                             /* address mask */
const uint32_t Icmp_AdvLenMin   = 8 + sizeof(nn::socket::Ip) + 8; /* min */

inline uint32_t Icmp_AdvLen(Icmp *p)
{
    return 8 + (p->icmp_ip.ip_hl << 2) + 8;
}

const uint32_t Icmp_MaxType = 40;

inline bool Icmp_InfoType(IcmpType type)
{
    return (type == IcmpType::Icmp_EchoReply || type == IcmpType::Icmp_Echo ||
            type == IcmpType::Icmp_RouterAdvert || type == IcmpType::Icmp_RouterSolicit ||
            type == IcmpType::Icmp_Tstamp || type == IcmpType::Icmp_TstampReply ||
            type == IcmpType::Icmp_IReq || type == IcmpType::Icmp_IReqReply ||
            type == IcmpType::Icmp_MaskReq || type == IcmpType::Icmp_MaskReply);
}

}} /* nn::socket */
