﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Types.h>

namespace nn {
namespace socket {

/*
 * Definitions for internet protocol version 4.
 *
 * Per RFC 791, September 1981.
 */
const uint32_t IpVersion = 4;

/*
 * Structure of an internet header, naked of options.
 */
enum class IpFlag : uint16_t
{
    Ip_Rf        = 0x8000,                  /* reserved fragment flag */
    Ip_Df        = 0x4000,                  /* dont fragment flag */
    Ip_Mf        = 0x2000,                  /* more fragments flag */
    Ip_OffMask   = 0x1fff                   /* mask for fragmenting bits */
};

struct Ip
{
#if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE )
    uint8_t         ip_hl:4,        ///< header length
                    ip_v:4;         ///< version
#else
    uint8_t         ip_v:4,         ///< version
                    ip_hl:4;        ///< header length
#endif
    uint8_t         ip_tos;         ///< type of service
    uint16_t        ip_len;         ///< total length
    uint16_t        ip_id;          ///< identification
    uint16_t        ip_off;         ///< fragment offset field
    uint8_t         ip_ttl;         ///< time to live
    uint8_t         ip_p;           ///< protocol, cast to nn::socket::Protocol
    uint16_t        ip_sum;         ///< checksum
    InAddr          ip_src,ip_dst;  ///< source and dest address
} /*__packed*/ NN_ALIGNAS(4);

const uint32_t Ip_MaxPacket = 65535; /* maximum packet size */

/*
 * Definitions for options.
 */
enum class IpOpt : uint8_t
{
    IpOpt_Control               = 0x00,
    IpOpt_Reserved1             = 0x20,
    IpOpt_DebMeas               = 0x40,
    IpOpt_Reserved2             = 0x60,

    IpOpt_Eol                   = 0,        /* end of option list */
    IpOpt_Nop                   = 1,        /* no operation */

    IpOpt_Rr                    = 7,        /* record packet route */
    IpOpt_Ts                    = 68,       /* timestamp */
    IpOpt_Security              = 130,      /* provide s,c,h,tcc */
    IpOpt_Lsrr                  = 131,      /* loose source route */
    IpOpt_Eso                   = 133,      /* extended security */
    IpOpt_CIpSo                 = 134,      /* commerical security */
    IpOpt_SatId                 = 136,      /* satnet id */
    IpOpt_Ssrr                  = 137,      /* strict source route */
    IpOpt_Ra                    = 148,      /* router alert */
/*
 * Offsets to fields in options other than EOL and NOP.
 */
    IpOpt_OptVal                = 0,        /* option ID */
    IpOpt_Olen                  = 1,        /* option length */
    IpOpt_Offset                = 2,        /* offset within option */
    IpOpt_MinOff                = 4,        /* min value of above */
};

inline IpOpt operator & (IpOpt a, uint8_t b) NN_NOEXCEPT
{
    return static_cast<IpOpt>(static_cast<uint8_t>(a) & b);
}

inline IpOpt IpOpt_Copied(IpOpt o)
{
    return o & 0x80;
}

inline IpOpt IpOpt_Class(IpOpt o)
{
    return o & 0x60;
}

inline IpOpt IpOpt_Number(IpOpt o)
{
    return o & 0x1f;
}

/*
 * Time stamp option structure.
 */
struct IpTimestamp
{
    uint8_t     ipt_code;                   /* IPOPT_TS */
    uint8_t     ipt_len;                    /* size of structure (variable) */
    uint8_t     ipt_ptr;                    /* index of current entry */
#if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE )
    uint8_t     ipt_flg:4,                  /* flags, see below */
                ipt_oflw:4;                 /* overflow counter */
#else
    uint8_t     ipt_oflw:4,                 /* overflow counter */
                ipt_flg:4;                  /* flags, see below */
#endif
    union ipt_timestamp
    {
        uint32_t    ipt_time[1];            /* network format */
        struct      ipt_ta
        {
            InAddr   ipt_addr;
            uint32_t ipt_time;              /* network format */
        } ipt_ta[1];
    } ipt_timestamp;
};

/* Flag bits for ipt_flg. */
enum class IptFlag : uint8_t
{
    IpOpt_Ts_TsOnly = 0,       /* timestamps only */
    IpOpt_Ts_TsAndAddr = 1,       /* timestamps and addresses */
    IpOpt_Ts_PreSpec = 3,       /* specified modules only */
};

/* Bits for security (not byte swapped). */
enum class IpsFlag : uint16_t
{
    IpOpt_Secur_Unclass         = 0x0000,
    IpOpt_Secur_ConfId          = 0xf135,
    IpOpt_Secur_Efto            = 0x789a,
    IpOpt_Secur_Mmmm            = 0xbc4d,
    IpOpt_Secur_Restr           = 0xaf13,
    IpOpt_Secur_Secret          = 0xd788,
    IpOpt_Secur_TopSecret       = 0x6bc5,
};

/*
 * Internet implementation parameters.
 */
const uint32_t MaxTtl       = 255;  /* maximum time to live (seconds) */
const uint32_t IpDefTtl     = 64;   /* default ttl, from RFC 1340 */
const uint32_t IpFragTtl    = 60;   /* time to live for frags, slowhz */
const uint32_t IpTtlDec     = 1;    /* subtracted when forwarding */
const uint32_t Ip_Mss       = 576;  /* default maximum segment size */

/*
 * This is the real IPv4 pseudo header, used for computing the TCP and UDP
 * checksums. For the Internet checksum, struct ipovly can be used instead.
 * For stronger checksums, the real thing must be used.
 */
struct IpPseudo
{
    InAddr      ippseudo_src;       /* source internet address */
    InAddr      ippseudo_dst;       /* destination internet address */
    uint8_t     ippseudo_pad;       /* pad, must be zero */
    uint8_t     ippseudo_p;         /* protocol */
    uint16_t    ippseudo_len;       /* protocol length */
};

}} /* nn::socket */
