﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/sys/socket.h>

namespace nn {
namespace socket {

/*
 * Length of interface external name, including terminating '\0'.
 * Note: this is the same size as a generic device's external name.
 */
const int If_NameSize = 16;
const int IfNamSiz    = If_NameSize;
const int If_MaxUnit  = 0x7fff; /* historical value */

/*
 * Structure used to query names of interface cloners.
 */
struct IfCloneReq
{
    int         ifcr_total;                 /* total cloners (out) */
    int         ifcr_count;                 /* room for this many in user buffer */
    char        *ifcr_buffer;               /* buffer for cloner names */
};

/*
 * Structure describing information about an interface
 * which may be of interest to management entities.
 */
struct IfData
{
    /* generic interface information */
    uint8_t   ifi_type;                     /* ethernet, tokenring, etc */
    uint8_t   ifi_physical;                 /* e.g., AUI, Thinnet, 10base-T, etc */
    uint8_t   ifi_addrlen;                  /* media address length */
    uint8_t   ifi_hdrlen;                   /* media header length */
    uint8_t   ifi_link_state;               /* current link state */
    uint8_t   ifi_vhid;                     /* carp vhid */
    uint8_t   ifi_baudrate_pf;              /* baudrate power factor */
    uint8_t   ifi_datalen;                  /* length of this data struct */
    uint32_t  ifi_mtu;                      /* maximum transmission unit */
    uint32_t  ifi_metric;                   /* routing metric (external only) */
    uint32_t  ifi_baudrate;                 /* linespeed */
                                            /* volatile statistics */
    uint32_t  ifi_ipackets;                 /* packets received on interface */
    uint32_t  ifi_ierrors;                  /* input errors on interface */
    uint32_t  ifi_opackets;                 /* packets sent on interface */
    uint32_t  ifi_oerrors;                  /* output errors on interface */
    uint32_t  ifi_collisions;               /* collisions on csma interfaces */
    uint32_t  ifi_ibytes;                   /* total number of octets received */
    uint32_t  ifi_obytes;                   /* total number of octets sent */
    uint32_t  ifi_imcasts;                  /* packets received via multicast */
    uint32_t  ifi_omcasts;                  /* packets sent via multicast */
    uint32_t  ifi_iqdrops;                  /* dropped on input, this interface */
    uint32_t  ifi_noproto;                  /* destined for unsupported protocol */
    uint64_t  ifi_hwassist;                 /* HW offload capabilities, see IFCAP */
    time_t    ifi_epoch;                    /* uptime at attach or stat reset */
    nn::socket::TimeVal ifi_lastchange;     /* time of last administrative change */
#ifdef _IFI_OQDROPS
    uint32_t  ifi_oqdrops;                  /* dropped on output */
#endif
};

/*-
 * Interface flags are of two types: network stack owned flags, and driver
 * owned flags.  Historically, these values were stored in the same ifnet
 * flags field, but with the advent of fine-grained locking, they have been
 * broken out such that the network stack is responsible for synchronizing
 * the stack-owned fields, and the device driver the device-owned fields.
 * Both halves can perform lockless reads of the other half's field, subject
 * to accepting the involved races.
 *
 * Both sets of flags come from the same number space, and should not be
 * permitted to conflict, as they are exposed to user space via a single
 * field.
 *
 * The following symbols identify read and write requirements for fields:
 *
 * (i) if_flags field set by device driver before attach, read-only there
 *     after.
 * (n) if_flags field written only by the network stack, read by either the
 *     stack or driver.
 * (d) if_drv_flags field written only by the device driver, read by either
 *     the stack or driver.
 */
enum class IfrFlag : uint16_t
{
    Iff_None            = 0x0,             /* no flags set */
    Iff_Up              = 0x1,             /* (n) interface is up */
    Iff_Broadcast       = 0x2,             /* (i) broadcast address valid */
    Iff_Debug           = 0x4,             /* (n) turn on debugging */
    Iff_Loopback        = 0x8,             /* (i) is a loopback net */
    Iff_PointToPoint    = 0x10,            /* (i) is a point-to-point link */
    Iff_Smart           = 0x20,            /* (i) interface manages own routes */
    Iff_Drv_Running     = 0x40,            /* (d) resources allocated */
    Iff_NoArp           = 0x80,            /* (n) no address resolution protocol */
    Iff_Promisc         = 0x100,           /* (n) receive all packets */
    Iff_AllMulti        = 0x200,           /* (n) receive all multicast packets */
    Iff_Drv_OActive     = 0x400,           /* (d) tx hardware queue is full */
    Iff_Simplex         = 0x800,           /* (i) can't hear own transmissions */
    Iff_Link0           = 0x1000,          /* per link layer defined bit */
    Iff_Link1           = 0x2000,          /* per link layer defined bit */
    Iff_Link2           = 0x4000,          /* per link layer defined bit */
    Iff_AltPhys         = Iff_Link2,       /* use alternate physical connection */
    Iff_Multicast       = 0x8000,          /* (i) supports multicast */
    Iff_Running         = Iff_Drv_Running,
    Iff_OActive         = Iff_Drv_OActive
};

/*
 * Values for if_link_state.
 */
enum class LinkState : uint32_t
{
    Link_State_Unknown  = 0,    /* link invalid/unknown */
    Link_State_Down     = 1,    /* link is down */
    Link_State_Up       = 2     /* link is up */
};

/*
 * Some convenience macros used for setting ifi_baudrate.
 * XXX 1000 vs. 1024? --thorpej@netbsd.org
 */
inline uintmax_t If_Kbps(uintmax_t x)     /* kilobits/sec. */
{
    return x * 1000;
}
inline uintmax_t If_Mbps(uintmax_t x)     /* megabits/sec. */
{
    return If_Kbps(x) * 1000;
}
inline uintmax_t If_Gbps(uintmax_t x)     /* gigabits/sec. */
{
    return If_Mbps(x) * 1000;
}

/*
 * Capabilities that interfaces can advertise.
 *
 * struct ifnet.if_capabilities
 *   contains the optional features & capabilities a particular interface
 *   supports (not only the driver but also the detected hw revision).
 *   Capabilities are defined by IfCap_* below.
 * struct ifnet.if_capenable
 *   contains the enabled (either by default or through ifconfig) optional
 *   features & capabilities on this interface.
 *   Capabilities are defined by IfCap_* below.
 * struct IfData.ifi_hwassist in mbuf CSum_ flag form, controlled by above
 *   contains the enabled optional feature & capabilites that can be used
 *   individually per packet and are specified in the mbuf pkthdr.csum_flags
 *   field.  IfCap_* and CSum_* do not match one to one and CSum_* may be
 *   more detailed or differenciated than IfCap_*.
 *   Hwassist features are defined CSum_* in sys/mbuf.h
 *
 * Capabilities that cannot be arbitrarily changed with ifconfig/ioctl
 * are listed in IfCap_CantChange, similar to Iff_CantChange.
 * This is not strictly necessary because the common code never
 * changes capabilities, and it is left to the individual driver
 * to do the right thing. However, having the filter here
 * avoids replication of the same code in all individual drivers.
 */
enum class IfCapability : uint32_t
{
    IfCap_RxCSum            = 0x00001,     /* can offload checksum on RX */
    IfCap_TxCSum            = 0x00002,     /* can offload checksum on TX */
    IfCap_NetCons           = 0x00004,     /* can be a network console */
    IfCap_Vlan_Mtu          = 0x00008,     /* VLAN-compatible MTU */
    IfCap_Vlan_HwTagging    = 0x00010,     /* hardware VLAN tag support */
    IfCap_Jumbo_Mtu         = 0x00020,     /* 9000 byte MTU supported */
    IfCap_Polling           = 0x00040,     /* driver supports polling */
    IfCap_Vlan_HwCSum       = 0x00080,     /* can do IfCap_HwCSum on VLANs */
    IfCap_Tso4              = 0x00100,     /* can do TCP Segmentation Offload */
    IfCap_Tso6              = 0x00200,     /* can do TCP6 Segmentation Offload */
    IfCap_Lro               = 0x00400,     /* can do Large Receive Offload */
    IfCap_Wol_Ucast         = 0x00800,     /* wake on any unicast frame */
    IfCap_Wol_Mcast         = 0x01000,     /* wake on any multicast frame */
    IfCap_Wol_Magic         = 0x02000,     /* wake on any Magic Packet */
    IfCap_Toe4              = 0x04000,     /* interface can offload TCP */
    IfCap_Toe6              = 0x08000,     /* interface can offload TCP6 */
    IfCap_Vlan_HwFilter     = 0x10000,     /* interface hw can filter vlan tag */
    IfCap_Polling_NoCount   = 0x20000,     /* polling ticks cannot be fragmented */
    IfCap_Vlan_HwTso        = 0x40000,     /* can do IfCap_Tso on VLANs */
    IfCap_LinkState         = 0x80000,     /* the runtime link state is dynamic */
    IfCap_NetMap            = 0x100000,    /* netmap mode supported/enabled */
    IfCap_RxCSum_IpV6       = 0x200000,    /* can offload checksum on IPv6 RX */
    IfCap_TxCSum_IpV6       = 0x400000,    /* can offload checksum on IPv6 TX */
    IfCap_HwStats           = 0x800000,    /* manages counters internally */

    IfCap_HwCSum_IpV6       = (IfCap_RxCSum_IpV6 | IfCap_TxCSum_IpV6),

    IfCap_HwCSum            = (IfCap_RxCSum | IfCap_TxCSum),
    IfCap_Tso               = (IfCap_Tso4 | IfCap_Tso6),
    IfCap_Wol               = (IfCap_Wol_Ucast | IfCap_Wol_Mcast | IfCap_Wol_Magic),
    IfCap_Toe               = (IfCap_Toe4 | IfCap_Toe6),

    IfCap_CantChange        = (IfCap_NetMap)
};

const uint32_t IfQ_MaxLen   = 50;
const uint32_t IfNet_SlowHz = 1;             /* granularity is 1 second */

/*
 * Message format for use in obtaining information about interfaces
 * from getkerninfo and the routing socket
 * For the new, extensible interface see struct IfMsgHdrl below.
 */
struct IfMsgHdr
{
    uint16_t    ifm_msglen;     /* to skip over non-understood messages */
    uint8_t     ifm_version;    /* future binary compatibility */
    uint8_t     ifm_type;       /* message type */
    int         ifm_addrs;      /* like rtm_addrs */
    int         ifm_flags;      /* value of if_flags */
    uint16_t    ifm_index;      /* index for associated ifp */
    IfData      ifm_data;       /* statistics and other data about if */
};

/*
 * The 'l' version shall be used by new interfaces, like NET_RT_IFLISTL.  It is
 * extensible after ifm_data_off or within ifm_data.  Both the IfMsgHdr and
 * IfData now have a member field detailing the struct length in addition to
 * the routing message length.  Macros are provided to find the start of
 * ifm_data and the start of the socket address strucutres immediately following
 * struct IfMsgHdrl given a pointer to struct IfMsgHdrl.
 */
struct IfMsgHdrl
{
    uint16_t    ifm_msglen;     /* to skip over non-understood messages */
    uint8_t     ifm_version;    /* future binary compatibility */
    uint8_t     ifm_type;       /* message type */
    int         ifm_addrs;      /* like rtm_addrs */
    int         ifm_flags;      /* value of if_flags */
    uint16_t    ifm_index;      /* index for associated ifp */
    uint16_t   _ifm_spare1;     /* spare space to grow if_index, see if_var.h */
    uint16_t    ifm_len;        /* length of IfMsgHdrl incl. IfData */
    uint16_t    ifm_data_off;   /* offset of IfData from beginning */
    IfData      ifm_data;       /* statistics and other data about if */
#ifdef _IN_NET_RTSOCK_C
    uint32_t    ifi_oqdrops;
#endif
};

inline IfData *IfMsgHdrlIfmData(IfMsgHdrl *ptr)
{
    return reinterpret_cast<IfData *>(reinterpret_cast<char *>(ptr) + ptr->ifm_data_off);
}

inline void *IfMsgHdrlRta(IfMsgHdrl *ptr)
{
    return reinterpret_cast<void *>(reinterpret_cast<uintptr_t *>(ptr) + ptr->ifm_len);
}

/*
 * Message format for use in obtaining information about interface addresses
 * from getkerninfo and the routing socket
 * For the new, extensible interface see struct ifa_msghdrl below.
 */
struct IfaMsgHdr
{
    uint16_t    ifam_msglen;                /* to skip over non-understood messages */
    uint8_t     ifam_version;               /* future binary compatibility */
    uint8_t     ifam_type;                  /* message type */
    int         ifam_addrs;                 /* like rtm_addrs */
    int         ifam_flags;                 /* value of ifa_flags */
    uint16_t    ifam_index;                 /* index for associated ifp */
    int         ifam_metric;                /* value of ifa_metric */
};

/*
 * The 'l' version shall be used by new interfaces, like NET_RT_IFLISTL.  It is
 * extensible after ifam_metric or within ifam_data.  Both the IfaMsgHdrl and
 * IfData now have a member field detailing the struct length in addition to
 * the routing message length.  Macros are provided to find the start of
 * ifm_data and the start of the socket address strucutres immediately following
 * struct IfaMsgHdrl given a pointer to struct IfaMsgHdrl.
 */
struct IfaMsgHdrl
{
    uint16_t    ifam_msglen;    /* to skip over non-understood messages */
    uint8_t     ifam_version;   /* future binary compatibility */
    uint8_t     ifam_type;      /* message type */
    int         ifam_addrs;     /* like rtm_addrs */
    int         ifam_flags;     /* value of ifa_flags */
    uint16_t    ifam_index;     /* index for associated ifp */
    uint16_t   _ifam_spare1;    /* spare space to grow if_index, see if_var.h */
    uint16_t    ifam_len;       /* length of IfaMsgHdrl incl. IfData */
    uint16_t    ifam_data_off;  /* offset of IfData from beginning */
    int         ifam_metric;    /* value of ifa_metric */
    IfData      ifam_data;      /* statistics and other data about if or address */
};

inline IfData *IfaMsgHdrlIfamData(IfaMsgHdrl *ptr)
{
    return reinterpret_cast<IfData *>(reinterpret_cast<char *>(ptr) + ptr->ifam_data_off);
}

inline void *IfaMsgHdrlRta(IfaMsgHdrl *ptr)
{
    return reinterpret_cast<void *>(reinterpret_cast<uintptr_t *>(ptr) + ptr->ifam_len);
}

/*
 * Message format for use in obtaining information about multicast addresses
 * from the routing socket
 */
struct IfmaMsgHdr
{
    uint16_t    ifmam_msglen;               /* to skip over non-understood messages */
    uint8_t     ifmam_version;              /* future binary compatibility */
    uint8_t     ifmam_type;                 /* message type */
    int         ifmam_addrs;                /* like rtm_addrs */
    int         ifmam_flags;                /* value of ifa_flags */
    uint16_t    ifmam_index;                /* index for associated ifp */
};

enum class IfAnnounceType : uint16_t
{
    Ifan_Arrival       =  0,    /* interface arrival */
    Ifan_Departure     =  1     /* interface departure */
};

/*
 * Message format announcing the arrival or departure of a network interface.
 */
struct IfAnnounceMsgHdr
{
    uint16_t        ifan_msglen;            /* to skip over non-understood messages */
    uint8_t         ifan_version;           /* future binary compatibility */
    uint8_t         ifan_type;              /* message type */
    uint16_t        ifan_index;             /* index for associated ifp */
    char            ifan_name[IfNamSiz];    /* if name, e.g. "en0" */
    IfAnnounceType  ifan_what;              /* what type of announcement */
};

/*
 * Buffer with length to be used in SiocGIfDescr/SiocSIfDescr requests
 */
struct IfReqBuffer
{
    size_t      length;
    void       *buffer;
};

/*
 * Interface request structure used for socket
 * ioctl's.  All interface ioctl's must have parameter
 * definitions which begin with ifr_name.  The
 * remainder may be interface specific.
 */
struct IfReq
{
    char            ifr_name[IfNamSiz];     /* if name, e.g. "en0" */
    union
    {
        SockAddr    ifru_addr;
        SockAddr    ifru_dstaddr;
        SockAddr    ifru_broadaddr;
        IfReqBuffer ifru_buffer;
        IfrFlag     ifru_flags[2];
        short       ifru_index;
        int         ifru_jid;
        int         ifru_metric;
        int         ifru_mtu;
        int         ifru_phys;
        int         ifru_media;
        char*       ifru_data;
        int         ifru_cap[2];
        uint32_t    ifru_fib;
    } ifr_ifru;
#define ifr_addr        ifr_ifru.ifru_addr      /* address */
#define ifr_dstaddr     ifr_ifru.ifru_dstaddr   /* other end of p-to-p link */
#define ifr_broadaddr   ifr_ifru.ifru_broadaddr /* broadcast address */
#define ifr_buffer      ifr_ifru.ifru_buffer    /* user supplied buffer with its length */
#define ifr_flags       ifr_ifru.ifru_flags[0]  /* flags (low 16 bits) */
#define ifr_flagshigh   ifr_ifru.ifru_flags[1]  /* flags (high 16 bits) */
#define ifr_jid         ifr_ifru.ifru_jid       /* jail/vnet */
#define ifr_metric      ifr_ifru.ifru_metric    /* metric */
#define ifr_mtu         ifr_ifru.ifru_mtu       /* mtu */
#define ifr_phys        ifr_ifru.ifru_phys      /* physical wire */
#define ifr_media       ifr_ifru.ifru_media     /* physical media */
#define ifr_data        ifr_ifru.ifru_data      /* for use by interface */
#define ifr_reqcap      ifr_ifru.ifru_cap[0]    /* requested capabilities */
#define ifr_curcap      ifr_ifru.ifru_cap[1]    /* current capabilities */
#define ifr_index       ifr_ifru.ifru_index     /* interface index */
#define ifr_fib         ifr_ifru.ifru_fib       /* interface fib */
};

inline uint32_t SizeofAddrIfReq(const IfReq &ifr)
{
    return (ifr.ifr_addr.sa_len > sizeof(SockAddr) ?
           sizeof(IfReq) - sizeof(SockAddr) + ifr.ifr_addr.sa_len :
           sizeof(IfReq));
}

struct IfAliasReq
{
    char        ifra_name[IfNamSiz];    /* if name, e.g. "en0" */
    SockAddr    ifra_addr;
    SockAddr    ifra_broadaddr;
    SockAddr    ifra_mask;
    int         ifra_vhid;
};

struct OIfAliasReq
{
    char        ifra_name[IfNamSiz];
    SockAddr    ifra_addr;
    SockAddr    ifra_broadaddr;
    SockAddr    ifra_mask;
};

struct IfMediaReq
{
    char        ifm_name[IfNamSiz]; /* if name, e.g. "en0" */
    int         ifm_current;        /* current media options */
    int         ifm_mask;           /* don't care mask */
    int         ifm_status;         /* media status */
    int         ifm_active;         /* active options */
    int         ifm_count;          /* # entries in ifm_ulist array */
    int        *ifm_ulist;          /* media words */
};

const uint32_t If_MacLen = 6;

struct IfDuplicateIpInfo
{
    char        ifdup_name[IfNamSiz];   /* if name, e.g. "en0" */
    uint32_t    ifdup_addr;             /* last duplicate ip address. 0.0.0.0 if no duplicate detected */
    uint8_t     ifdup_mac[If_MacLen];    /* mac address of last duplicator */
    uint32_t    ifdup_maclen;
};

struct IfDrv
{
    char            ifd_name[IfNamSiz]; /* if name, e.g. "en0" */
    unsigned long   ifd_cmd;
    size_t          ifd_len;
    void            *ifd_data;
};

/*
 * Structure used to retrieve aux status data from interfaces.
 * Kernel suppliers to this interface should respect the formatting
 * needed by ifconfig(8): each line starts with a TAB and ends with
 * a newline.  The canonical example to copy and paste is in if_tun.c.
 */

const uint32_t IfStatMax = 800; /* 10 lines of text */

struct IfStat
{
    char    ifs_name[IfNamSiz];     /* if name, e.g. "en0" */
    char    ascii[IfStatMax + 1];
};

/*
 * Structure used in SiocGIfConf request.
 * Used to retrieve interface configuration
 * for machine (useful for programs which
 * must know all networks accessible).
 */
struct IfConf
{
    int         ifc_len;            /* size of associated buffer */
    union
    {
        char*   ifcu_buf;
        IfReq   *ifcu_req;
    } ifc_ifcu;
#define ifc_buf ifc_ifcu.ifcu_buf   /* buffer address */
#define ifc_req ifc_ifcu.ifcu_req   /* array of structures returned */
};

/*
 * interface groups
 */

const char Ifg_All[]    = "all";     /* group contains all interfaces */
const char Ifg_Egress[] = "egress";  /* if(s) default route(s) point to */

struct IfGReq
{
    union
    {
        char    ifgrqu_group[IfNamSiz];
        char    ifgrqu_member[IfNamSiz];
    } ifgrq_ifgrqu;
#define ifgrq_group     ifgrq_ifgrqu.ifgrqu_group
#define ifgrq_member    ifgrq_ifgrqu.ifgrqu_member
};

/*
 * Used to lookup groups for an interface
 */
struct IfGroupReq
{
    char        ifgr_name[IfNamSiz];
    uint32_t    ifgr_len;
    union
    {
        char    ifgru_group[IfNamSiz];
        IfGReq *ifgru_groups;
    } ifgr_ifgru;
#define ifgr_group      ifgr_ifgru.ifgru_group
#define ifgr_groups     ifgr_ifgru.ifgru_groups
};

/*
 * Structure for Sioc[AGD]LIfAddr
 */
const uint16_t Iflr_Prefix = 0x8000;    /* in: prefix given  out: kernel fills id */

struct IfLAddrReq
{
    char        iflr_name[IfNamSiz];
    uint32_t    flags;
    uint32_t    prefixlen;             /* in/out */
    SockAddrStorage addr;              /* in/out */
    SockAddrStorage dstaddr;           /* out */
};

struct IfNameIndex
{
    unsigned int    if_index;   /* 1, 2, ... */
    char           *if_name;    /* null terminated name: "le0", ... */
};

}} /* nn::socket */
