﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/private/net/socket_If.h>

namespace nn {
namespace socket {

const size_t Ether_Addr_Len         = 6;        /* length of an Ethernet address */
const size_t Ether_Type_Len         = 2;        /* length of the Ethernet type field */
const size_t Ether_Crc_Len          = 4;        /* length of the Ethernet CRC */
const size_t Ether_Hdr_Len          = (Ether_Addr_Len * 2 + Ether_Type_Len);
const size_t Ether_Min_Len          = 64;       /* minimum frame len, including CRC */
const size_t Ether_Max_Len          = 1518;     /* maximum frame len, including CRC */
const size_t Ether_Max_Len_Jumbo    = 9018;     /* max jumbo frame len, including CRC */
const size_t Ether_Vlan_Encap_Len   = 4;        /* len of 802.1Q VLAN encapsulation */

/*
 * Mbuf adjust factor to force 32-bit alignment of IP header.
 * Drivers should do m_adj(m, ETHER_ALIGN) when setting up a
 * receive so the upper layers get the IP header properly aligned
 * past the 14-byte Ethernet header.
 */
const uint32_t Ether_Align = 2;         /* driver adjust for IP hdr alignment */

/*
 * Ethernet CRC32 polynomials (big- and little-endian verions).
 */
enum class EtherCrcPoly : uint32_t
{
    Ether_Crc_Poly_Le       = 0xedb88320,
    Ether_Crc_Poly_Be       = 0x04c11db6
};

/*
 * A function to validate a length with
 */
inline bool Ether_Is_Valid_Len(size_t len)
{
    return (len >= Ether_Min_Len && len <= Ether_Max_Len);
}

 /*
 * Structure of a 10Mb/s Ethernet header.
 */
struct EtherHeader
{
    uint8_t     ether_dhost[Ether_Addr_Len];
    uint8_t     ether_shost[Ether_Addr_Len];
    uint16_t    ether_type;
} /*__packed*/;

/*
* Structure of a 48-bit Ethernet address.
*/
struct EtherAddr
{
    uint8_t     octet[Ether_Addr_Len];
} /*__packed*/;

inline bool Ether_Is_Multicast(uint8_t *addr)
{
    return *addr & 0x01; /* is address mcast/bcast? */
}

/*
 *  NOTE: 0x0000-0x05DC (0..1500) are generally IEEE 802.3 length fields.
 *  However, there are some conflicts.
 */
enum class EtherType : uint32_t
{
    EtherType_8023          = 0x0004,       /* IEEE 802.3 packet */
                                            /* 0x0101 .. 0x1FF       Experimental */
    EtherType_Pup           = 0x0200,       /* Xerox PUP protocol - see 0A00 */
    EtherType_PupAt         = 0x0200,       /* PUP Address Translation - see 0A01 */
    EtherType_Sprite        = 0x0500,       /* ??? */
                                            /* 0x0400       Nixdorf */
    EtherType_Ns            = 0x0600,       /* XNS */
    EtherType_NsAt          = 0x0601,       /* XNS Address Translation (3Mb only) */
    EtherType_Dlog1         = 0x0660,       /* DLOG (?) */
    EtherType_Dlog2         = 0x0661,       /* DLOG (?) */
    EtherType_Ip            = 0x0800,       /* IP protocol */
    EtherType_X75           = 0x0801,       /* X.75 Internet */
    EtherType_Nbs           = 0x0802,       /* NBS Internet */
    EtherType_Ecma          = 0x0803,       /* ECMA Internet */
    EtherType_Chaos         = 0x0804,       /* CHAOSnet */
    EtherType_X25           = 0x0805,       /* X.25 Level 3 */
    EtherType_Arp           = 0x0806,       /* Address resolution protocol */
    EtherType_NsCompat      = 0x0807,       /* XNS Compatibility */
    EtherType_FrArp         = 0x0808,       /* Frame Relay ARP (RFC1701) */
                                            /* 0x081C       Symbolics Private */
                                            /* 0x0888 - 0x088A       Xyplex */
    EtherType_UbDebug       = 0x0900,       /* Ungermann-Bass network debugger */
    EtherType_IeeePup       = 0x0A00,       /* Xerox IEEE802.3 PUP */
    EtherType_IeeePupAt     = 0x0A01,       /* Xerox IEEE802.3 PUP Address Translation */
    EtherType_Vines         = 0x0BAD,       /* Banyan VINES */
    EtherType_VinesLoop     = 0x0BAE,       /* Banyan VINES Loopback */
    EtherType_VinesEcho     = 0x0BAF,       /* Banyan VINES Echo */

    /* 0x1000 - 0x100F Berkeley Trailer */
    EtherType_Dca           = 0x1234,       /* DCA - Multicast */
    EtherType_Valid         = 0x1600,       /* VALID system protocol */
    EtherType_DogFight      = 0x1989,       /* Artificial Horizons ("Aviator" dogfight simulator [on Sun]) */
    EtherType_Rcl           = 0x1995,       /* Datapoint Corporation (RCL lan protocol) */

    /* The following 3C0x types are unregistered: */
    EtherType_NbpVcd        = 0x3C00,       /* 3Com NBP virtual circuit datagram (like XNS SPP) not registered */
    EtherType_NbpScd        = 0x3C01,       /* 3Com NBP System control datagram not registered */
    EtherType_NbpCReq       = 0x3C02,       /* 3Com NBP Connect request (virtual cct) not registered */
    EtherType_NbpCRsp       = 0x3C03,       /* 3Com NBP Connect response not registered */
    EtherType_NbpCc         = 0x3C04,       /* 3Com NBP Connect complete not registered */
    EtherType_NbpClReq      = 0x3C05,       /* 3Com NBP Close request (virtual cct) not registered */
    EtherType_NbpClRsp      = 0x3C06,       /* 3Com NBP Close response not registered */
    EtherType_NbpDg         = 0x3C07,       /* 3Com NBP Datagram (like XNS IDP) not registered */
    EtherType_NbpDgb        = 0x3C08,       /* 3Com NBP Datagram broadcast not registered */
    EtherType_NbpClaim      = 0x3C09,       /* 3Com NBP Claim NetBIOS name not registered */
    EtherType_NbpDlte       = 0x3C0A,       /* 3Com NBP Delete NetBIOS name not registered */
    EtherType_NbpRas        = 0x3C0B,       /* 3Com NBP Remote adaptor status request not registered */
    EtherType_NbpRar        = 0x3C0C,       /* 3Com NBP Remote adaptor response not registered */
    EtherType_NbpRst        = 0x3C0D,       /* 3Com NBP Reset not registered */

    EtherType_Pcs           = 0x4242,       /* PCS Basic Block Protocol */
    EtherType_ImlblDiag     = 0x424C,       /* Information Modes Little Big LAN diagnostic */
    EtherType_Diddle        = 0x4321,       /* THD - Diddle */
    EtherType_Imlbl         = 0x4C42,       /* Information Modes Little Big LAN */
    EtherType_SimNet        = 0x5208,       /* BBN Simnet Private */
    EtherType_DecExper      = 0x6000,       /* DEC Unassigned, experimental */
    EtherType_MopDl         = 0x6001,       /* DEC MOP dump/load */
    EtherType_MopRc         = 0x6002,       /* DEC MOP remote console */
    EtherType_DecNet        = 0x6003,       /* DEC DECNET Phase IV route */
    EtherType_Dn            = EtherType_DecNet, /* libpcap, tcpdump */
    EtherType_Lat           = 0x6004,       /* DEC LAT */
    EtherType_DecDiag       = 0x6005,       /* DEC diagnostic protocol (at interface initialization?) */
    EtherType_DecCust       = 0x6006,       /* DEC customer protocol */
    EtherType_Sca           = 0x6007,       /* DEC LAVC, SCA */
    EtherType_Amber         = 0x6008,       /* DEC AMBER */
    EtherType_DecMumps      = 0x6009,       /* DEC MUMPS */
    /* 0x6010 - 0x6014                         3Com Corporation */
    EtherType_TransEther    = 0x6558,       /* Trans Ether Bridging (RFC1701)*/
    EtherType_RawFr         = 0x6559,       /* Raw Frame Relay (RFC1701) */
    EtherType_UbDl          = 0x7000,       /* Ungermann-Bass download */
    EtherType_UbNiu         = 0x7001,       /* Ungermann-Bass NIUs */
    EtherType_UbDiagLoop    = 0x7002,       /* Ungermann-Bass diagnostic/loopback */
    EtherType_UbNmc         = 0x7003,       /* Ungermann-Bass ??? (NMC to/from UB Bridge) */
    EtherType_UbBst         = 0x7005,       /* Ungermann-Bass Bridge Spanning Tree */
    EtherType_Os9           = 0x7007,       /* OS/9 Microware */
    EtherType_Os9Net        = 0x7009,       /* OS/9 Net? */
    /* 0x7020 - 0x7029                         LRT (England) (now Sintrom) */
    EtherType_Racal         = 0x7030,       /* Racal-Interlan */
    EtherType_PrimeNts      = 0x7031,       /* Prime NTS (Network Terminal Service) */
    EtherType_Cabletron     = 0x7034,       /* Cabletron */
    EtherType_CronusVln     = 0x8003,       /* Cronus VLN */
    EtherType_Cronus        = 0x8004,       /* Cronus Direct */
    EtherType_Hp            = 0x8005,       /* HP Probe */
    EtherType_Nestar        = 0x8006,       /* Nestar */
    EtherType_AttStanford   = 0x8008,       /* AT&T/Stanford (local use) */
    EtherType_Excelan       = 0x8010,       /* Excelan */
    EtherType_Sg_Diag       = 0x8013,       /* SGI diagnostic type */
    EtherType_Sg_NetGames   = 0x8014,       /* SGI network games */
    EtherType_Sg_Resv       = 0x8015,       /* SGI reserved type */
    EtherType_Sg_Bounce     = 0x8016,       /* SGI bounce server */
    EtherType_ApolloDomain  = 0x8019,       /* Apollo DOMAIN */
    EtherType_Tymshare      = 0x802E,       /* Tymeshare */
    EtherType_Tigan         = 0x802F,       /* Tigan, Inc. */
    EtherType_RevArp        = 0x8035,       /* Reverse addr resolution protocol */
    EtherType_Aeonic        = 0x8036,       /* Aeonic Systems */
    EtherType_IpxNew        = 0x8037,       /* IPX (Novell Netware?) */
    EtherType_LanBridge     = 0x8038,       /* DEC LANBridge */
    EtherType_Dsmd          = 0x8039,       /* DEC DSM/DDP */
    EtherType_Argonaut      = 0x803A,       /* DEC Argonaut Console */
    EtherType_Vaxeln        = 0x803B,       /* DEC VAXELN */
    EtherType_DecDns        = 0x803C,       /* DEC DNS Naming Service */
    EtherType_Encrypt       = 0x803D,       /* DEC Ethernet Encryption */
    EtherType_DecDts        = 0x803E,       /* DEC Distributed Time Service */
    EtherType_DecLtm        = 0x803F,       /* DEC LAN Traffic Monitor */
    EtherType_DecNetBios    = 0x8040,       /* DEC PATHWORKS DECnet NETBIOS Emulation */
    EtherType_DecLast       = 0x8041,       /* DEC Local Area System Transport */
    /* 0x8042                                  DEC Unassigned */
    EtherType_Planning      = 0x8044,       /* Planning Research Corp. */
    /* 0x8046 - 0x8047                         AT&T */
    EtherType_DecAm         = 0x8048,       /* DEC Availability Manager for Distributed Systems DECamds (but someone at DEC says not) */
    EtherType_ExperData     = 0x8049,       /* ExperData */
    EtherType_Vexp          = 0x805B,       /* Stanford V Kernel exp. */
    EtherType_Vprod         = 0x805C,       /* Stanford V Kernel prod. */
    EtherType_Es            = 0x805D,       /* Evans & Sutherland */
    EtherType_Little        = 0x8060,       /* Little Machines */
    EtherType_Counterpoint  = 0x8062,       /* Counterpoint Computers */
    /* 0x8065 - 0x8066                         Univ. of Mass @ Amherst */
    EtherType_Veeco         = 0x8067,       /* Veeco Integrated Auto. */
    EtherType_Gendyn        = 0x8068,       /* General Dynamics */
    EtherType_Att           = 0x8069,       /* AT&T */
    EtherType_Autophon      = 0x806A,       /* Autophon */
    EtherType_ComDesign     = 0x806C,       /* ComDesign */
    EtherType_Compugraphic  = 0x806D,       /* Compugraphic Corporation */
    /* 0x806E - 0x8077                         Landmark Graphics Corp. */
    EtherType_Matra         = 0x807A,       /* Matra */
    EtherType_Dde           = 0x807B,       /* Dansk Data Elektronik */
    EtherType_Merit         = 0x807C,       /* Merit Internodal (or Univ of Michigan?) */
    /* 0x807D - 0x807F                         Vitalink Communications */
    EtherType_VltlMan       = 0x8080,       /* Vitalink TransLAN III Management */
    /* 0x8081 - 0x8083                         Counterpoint Computers */
    /* 0x8088 - 0x808A                         Xyplex */
    EtherType_Atalk         = 0x809B,       /* AppleTalk */
    EtherType_At            = EtherType_Atalk, /* old NetBSD */
    EtherType_AppleTalk     = EtherType_Atalk, /* HP-UX */
    /* 0x809C - 0x809E                         Datability */
    EtherType_Spider        = 0x809F,       /* Spider Systems Ltd. */
    /* 0x80A3                                  Nixdorf */
    /* 0x80A4 - 0x80B3                         Siemens Gammasonics Inc. */
    /* 0x80C0 - 0x80C3                         DCA (Digital Comm. Assoc.) Data Exchange Cluster */
    /* 0x80C4 - 0x80C5                         Banyan Systems */
    EtherType_Pacer         = 0x80C6,       /* Pacer Software */
    EtherType_Applitek      = 0x80C7,       /* Applitek Corporation */
    /* 0x80C8 - 0x80CC                         Intergraph Corporation */
    /* 0x80CD - 0x80CE                         Harris Corporation */
    /* 0x80CF - 0x80D2                         Taylor Instrument */
    /* 0x80D3 - 0x80D4                         Rosemount Corporation */
    EtherType_Sna           = 0x80D5,       /* IBM SNA Services over Ethernet */
    EtherType_Varian        = 0x80DD,       /* Varian Associates */
    /* 0x80DE - 0x80DF                         TRFS (Integrated Solutions Transparent Remote File System) */
    /* 0x80E0 - 0x80E3                         Allen-Bradley */
    /* 0x80E4 - 0x80F0                         Datability */
    EtherType_Retix         = 0x80F2,       /* Retix */
    EtherType_Aarp          = 0x80F3,       /* AppleTalk AARP */
    /* 0x80F4 - 0x80F5                         Kinetics */
    EtherType_Apollo        = 0x80F7,       /* Apollo Computer */
    EtherType_Vlan          = 0x8100,       /* IEEE 802.1Q VLAN tagging (XXX conflicts) */
    /* 0x80FF - 0x8101                         Wellfleet Communications (XXX conflicts) */
    EtherType_Bofl          = 0x8102,       /* Wellfleet; BOFL (Breath OF Life) pkts [every 5-10 secs.] */
    EtherType_Wellfleet     = 0x8103,       /* Wellfleet Communications */
    /* 0x8107 - 0x8109                         Symbolics Private */
    EtherType_Talaris       = 0x812B,       /* Talaris */
    EtherType_Waterloo      = 0x8130,       /* Waterloo Microsystems Inc. (XXX which?) */
    EtherType_Hayes         = 0x8130,       /* Hayes Microcomputers (XXX which?) */
    EtherType_VgLab         = 0x8131,       /* VG Laboratory Systems */
    /* 0x8132 - 0x8137                         Bridge Communications */
    EtherType_Ipx           = 0x8137,       /* Novell (old) NetWare IPX (ECONFIG E option) */
    EtherType_Novell        = 0x8138,       /* Novell, Inc. */
    /* 0x8139 - 0x813D                         KTI */
    EtherType_Mumps         = 0x813F,       /* M/MUMPS data sharing */
    EtherType_Amoeba        = 0x8145,       /* Vrije Universiteit (NL) Amoeba 4 RPC (obsolete) */
    EtherType_Flip          = 0x8146,       /* Vrije Universiteit (NL) FLIP (Fast Local Internet Protocol) */
    EtherType_VrReserved    = 0x8147,       /* Vrije Universiteit (NL) [reserved] */
    EtherType_Logicraft     = 0x8148,       /* Logicraft */
    EtherType_Ncd           = 0x8149,       /* Network Computing Devices */
    EtherType_Alpha         = 0x814A,       /* Alpha Micro */
    EtherType_Snmp          = 0x814C,       /* SNMP over Ethernet (see RFC1089) */
    /* 0x814D - 0x814E       BIIN */
    EtherType_Tec           = 0x814F,       /* Technically Elite Concepts */
    EtherType_Rational      = 0x8150,       /* Rational Corp */
    /* 0x8151 - 0x8153                         Qualcomm */
    /* 0x815C - 0x815E                         Computer Protocol Pty Ltd */
    /* 0x8164 - 0x8166                         Charles River Data Systems */
    EtherType_Xtp           = 0x817D,       /* Protocol Engines XTP */
    EtherType_SgiTw         = 0x817E,       /* SGI/Time Warner prop. */
    EtherType_Hippi_Fp      = 0x8180,       /* HIPPI-FP encapsulation */
    EtherType_Stp           = 0x8181,       /* Scheduled Transfer STP, HIPPI-ST */
    /* 0x8182 - 0x8183                         Reserved for HIPPI-6400 */
    /* 0x8184 - 0x818C                         SGI prop. */
    EtherType_Motorola      = 0x818D,       /* Motorola */
    EtherType_NetBeui       = 0x8191,       /* PowerLAN NetBIOS/NetBEUI (PC) */
    /* 0x819A - 0x81A3                         RAD Network Devices */
    /* 0x81B7 - 0x81B9                         Xyplex */
    /* 0x81CC - 0x81D5                         Apricot Computers */
    /* 0x81D6 - 0x81DD                         Artisoft Lantastic */
    /* 0x81E6 - 0x81EF                         Polygon */
    /* 0x81F0 - 0x81F2                         Comsat Labs */
    /* 0x81F3 - 0x81F5                         SAIC */
    /* 0x81F6 - 0x81F8                         VG Analytical */
    /* 0x8203 - 0x8205                         QNX Software Systems Ltd. */
    /* 0x8221 - 0x8222                         Ascom Banking Systems */
    /* 0x823E - 0x8240                         Advanced Encryption Systems */
    /* 0x8263 - 0x826A                         Charles River Data Systems */
    /* 0x827F - 0x8282                         Athena Programming */
    /* 0x829A - 0x829B                         Inst Ind Info Tech */
    /* 0x829C - 0x82AB                         Taurus Controls */
    /* 0x82AC - 0x8693                         Walker Richer & Quinn */
    EtherType_Accton        = 0x8390,       /* Accton Technologies (unregistered) */
    EtherType_TalarisMc     = 0x852B,       /* Talaris multicast */
    EtherType_Kalpana       = 0x8582,       /* Kalpana */
    /* 0x8694 - 0x869D                         Idea Courier */
    /* 0x869E - 0x86A1                         Computer Network Tech */
    /* 0x86A3 - 0x86AC                         Gateway Communications */
    EtherType_Sectra        = 0x86DB,       /* SECTRA */
    EtherType_IpV6          = 0x86DD,       /* IP protocol version 6 */
    EtherType_DeltaCon      = 0x86DE,       /* Delta Controls */
    EtherType_Atomic        = 0x86DF,       /* ATOMIC */
    /* 0x86E0 - 0x86EF                         Landis & Gyr Powers */
    /* 0x8700 - 0x8710                         Motorola */
    EtherType_Rdp           = 0x8739,       /* Control Technology Inc. RDP Without IP */
    EtherType_Micp          = 0x873A,       /* Control Technology Inc. Mcast Industrial Ctrl Proto. */
    /* 0x873B - 0x873C                         Control Technology Inc. Proprietary */
    EtherType_TcpComp       = 0x876B,       /* TCP/IP Compression (RFC1701) */
    EtherType_Ipas          = 0x876C,       /* IP Autonomous Systems (RFC1701) */
    EtherType_SecureData    = 0x876D,       /* Secure Data (RFC1701) */
    EtherType_FlowControl   = 0x8808,       /* 802.3x flow control packet */
    EtherType_Slow          = 0x8809,       /* 802.3ad link aggregation (LACP) */
    EtherType_Ppp           = 0x880B,       /* PPP (obsolete by PPPoE) */
    EtherType_Hitachi       = 0x8820,       /* Hitachi Cable (Optoelectronic Systems Laboratory) */
    EtherType_Mpls          = 0x8847,       /* MPLS Unicast */
    EtherType_Mpls_Mcast    = 0x8848,       /* MPLS Multicast */
    EtherType_Axis          = 0x8856,       /* Axis Communications AB proprietary bootstrap/config */
    EtherType_PppoeDisc     = 0x8863,       /* PPP Over Ethernet Discovery Stage */
    EtherType_Pppoe         = 0x8864,       /* PPP Over Ethernet Session Stage */
    EtherType_LanProbe      = 0x8888,       /* HP LanProbe test? */
    EtherType_Pae           = 0x888e,       /* EAPOL PAE/802.1x */
    EtherType_Loopback      = 0x9000,       /* Loopback: used to test interfaces */
    EtherType_Lback         = EtherType_Loopback,    /* DEC MOP loopback */
    EtherType_XnsSm         = 0x9001,       /* 3Com (Formerly Bridge Communications), XNS Systems Management */
    EtherType_TcpSm         = 0x9002,       /* 3Com (Formerly Bridge Communications), TCP/IP Systems Management */
    EtherType_BcLoop        = 0x9003,       /* 3Com (Formerly Bridge Communications), loopback detection */
    EtherType_Debni         = 0xAAAA,       /* DECNET? Used by VAX 6220 DEBNI */
    EtherType_Sonix         = 0xFAF5,       /* Sonix Arpeggio */
    EtherType_Vital         = 0xFF00,       /* BBN VITAL-LanBridge cache wakeups */
    /* 0xFF00 - 0xFFOF                         ISC Bunker Ramo */

    /*
    * The ETHERTYPE_NTRAILER packet types starting at ETHERTYPE_TRAIL have
    * (type-ETHERTYPE_TRAIL)*512 bytes of data followed
    * by an ETHER type (as given above) and then the (variable-length) header.
    */
    EtherType_Trail         = 0x1000,        /* Trailer packet */
    EtherType_NTrailer      = 16,

    EtherType_Max           = 0xFFFF        /* Maximum valid ethernet type, reserved */
};

/*
* Compute the maximum frame size based on ethertype (i.e. possible
* encapsulation) and whether or not an FCS is present.
*/
inline size_t Ether_Max_Frame(IfData *ifp, EtherType etype, bool hasfcs)
{
    return ifp->ifi_mtu + Ether_Hdr_Len + (hasfcs ? Ether_Crc_Len : 0) + ((etype == EtherType::EtherType_Vlan) ? Ether_Vlan_Encap_Len : 0);
}

const uint32_t EtherMtu         = Ether_Max_Len - Ether_Hdr_Len - Ether_Crc_Len;
const uint32_t EtherMin         = Ether_Min_Len - Ether_Hdr_Len - Ether_Crc_Len;
const uint32_t EtherMtu_Jumbo   = Ether_Max_Len_Jumbo - Ether_Hdr_Len - Ether_Crc_Len;

}} /* nn::socket */
