﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief C++ 標準アロケータをサービスフレームワークで使用するためのユーティリティを定義します。
*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_AllocationPolicies.h>

namespace nn { namespace sf {

/**
    @brief 指定した C++ 標準アロケータを用いてサービスオブジェクトを構築するというアロケーションポリシーを表すクラステンプレートです。

    @tparam StdAllocatorT C++ 標準アロケータインターフェイスを持つアロケータクラステンプレートを指定します。多くの場合、std::allocator を指定します。

    @details
     このクラステンプレートのインスタンスクラスは、状態なしのアロケーションポリシーとして使用することができます。
*/
template <template <typename T> class StdAllocatorT>
class StdAllocationPolicy
{
public:

    static const bool HasStatefulAllocator = false;
    typedef detail::StatelessDummyAllocator Allocator;

    template <typename T>
    struct GetAllocator
    {
        struct Allocator
        {
            static void* Allocate(size_t size)
            {
                return StdAllocatorT<T>().allocate(size / sizeof(T));
            }

            static void Deallocate(void* p, size_t size)
            {
                StdAllocatorT<T>().deallocate(static_cast<T*>(p), size / sizeof(T));
            }
        };
        typedef Allocator type;
    };

    template <typename T>
    static void* AllocateAligned(size_t size, size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        return StdAllocatorT<T>().allocate(size / sizeof(T));
    }

    template <typename T>
    static void DeallocateAligned(void* p, size_t size, size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        StdAllocatorT<T>().deallocate(static_cast<T*>(p), size / sizeof(T));
    }

};

}}
