﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークを使用して Shim ライブラリを作成する際に使用できるユーティリティを定義します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/nn_SdkAssert.h>
#include <mutex>
#include <utility>
#include <type_traits>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/detail/sf_StaticMutex.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/nn_Allocator.h>

namespace nn { namespace sf {

namespace detail {

const Bit32 ShimLibraryObjectHolderMagicNumber = 0x18745354;

}

/**
    @brief サービスオブジェクトを安全に保持し Shim ライブラリの作成を支援する型です。

    @tparam Interface ライブラリが最初に取得するオブジェクトのサービスインターフェイス型を指定します。

    @details
     サービスフレームワークで提供されるオブジェクトを使用して、Shim ライブラリを実装するような場合、
     本ユーティリティを使うことを推奨します。

     サービスフレームワークを実装に使用している Shim ライブラリは一般に、
     ライブラリの起点となるサービスオブジェクト参照を一つ持ちます。
     ここではそのサービスオブジェクトのことを、ライブラリサービスオブジェクト参照と呼びます。

     このクラスは、典型的な Shim ライブラリ実装におけるライブラリサービスオブジェクト参照の管理を行い、
     安全かつ、統一性のある実装を支援します。

     このユーティリティは以下の機能を持ちます。

     - 指定されたオブジェクト取得関数を使ってライブラリサービスオブジェクト参照を生成し、ホルダーを初期化する
     - 指定されたオブジェクトを直接使って、ホルダーを初期化する
     - 初期化回数をスレッドセーフに管理し(InitializeHolderDirectly を除く)、多重初期化を防ぐ
     - ライブラリサービスオブジェクト参照を取得する

     この型の変数はグローバルに配置し、必ず NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER マクロを使用して初期化してください。
     ライブラリコードのロード完了時にこの変数の初期化が終わっていることが保証されます。

     このユーティリティは、内部に指定されたインターフェイスを持つライブラリサービスオブジェクト参照を持ち、これの管理を行います。
     このサービスオブジェクトの参照を設定するには InitializeHolder() または InitializeHolderDirectly() を呼んでください。

     また、このユーティリティは、内部に初期化回数カウンタを持ち、その初期値は 0 です。
     InitializeHolder() または InitializeHolderDirectly() を呼ぶたびに 1 だけ増えます。
     FinalizeHolder() を呼ぶたびに 1 だけ減り、その結果、カウンタが 0 になると保持しているライブラリサービスオブジェクト参照は解放されます。

     ライブラリサービスオブジェクト参照を取得するには GetObject() または operator->() を呼んでください。

     このクラステンプレートのメンバ関数は、同じインスタンスに対して同時に呼び出すことができます。

    @see NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER
*/
template <typename Interface>
struct ShimLibraryObjectHolder
{
private:

    void AssertInitializerUsed() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == detail::ShimLibraryObjectHolderMagicNumber, "This needs to be initialized by NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER");
    }

public:

    Interface* _p;
    mutable detail::StaticMutex _mutex;
    int _initializeCount;
    void (*_finalizer)(void* finalizerArgument);
    void* _finalizerArgument;
    Bit32 _magicNumber;

    /**
        @brief nn::sf::ShimLibraryObjectHolder のインスタンスの初期化用マクロです。
    */
    #define NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER \
        { \
            nullptr, \
            NN_SF_DETAIL_STATIC_MUTEX_INITIALIZER(false), \
            0, \
            nullptr, \
            nullptr, \
            ::nn::sf::detail::ShimLibraryObjectHolderMagicNumber, \
        }

    /**
        @brief InitializeHolder() 関数に渡すオブジェクト取得関数の型を表します。

        @pre pOut != nullptr
        @pre (*pOut).Get() = nullptr

        @post (*pOut).Get() != nullptr
    */
    typedef Result (*ObjectGetterFunction)(SharedPointer<Interface>* pOut);

    /**
        @brief 指定されたオブジェクト取得関数オブジェクトを使用して Shim ライブラリオブジェクトを指定します。

        @tparam ObjectGetter オブジェクトを取得するための関数オブジェクトの型を指定します。通常は明示的な指定は不要です。この関数オブジェクトは ObjectGetterFunction と同等の呼び出しができる必要があります。

        @param[in] objectGetter オブジェクトの取得を行う関数オブジェクトを指定します。
        @param[in] finalizer 解放時に呼ばれる関数を指定します。
        @param[in] finalizerArgument 解放時に呼ばれる関数の引数を指定します。

        @pre SharedPointer<Interface> 型の変数 p に対して objectGetter(&p) を呼び出して成功が返る場合には、p は有効なオブジェクトを指している

        @post 初期化回数カウンタ == この関数を呼ぶ前の初期化回数カウンタ + 1
        @post this->GetObject().Get() != nullptr

        @return objectGetter の呼び出し結果をそのまま返します。objectGetter を呼び出さなかった場合には成功を返します。

        @details
         指定された関数オブジェクトを使用してオブジェクトを取得し、そのオブジェクトを使ってライブラリサービスオブジェクト参照を初期化します。

         初期化回数カウンタが 0 で呼んだ場合にのみ objectGetter が呼ばれオブジェクトが取得されます。

         finalizer が指定されている場合には、 FinalizeHolder() で実際にオブジェクトが解放される際に、
         finalizer(finalizerArgument) が呼ばれます。

         この関数は、Shim ライブラリの通常の初期化関数の中で呼ばれることを想定しています、
         objectGetter には多くの場合 HIPC プロキシなどからオブジェクトを取得する関数を指定します。

        @see ObjectGetterFunction
    */
    template <typename ObjectGetter>
    Result InitializeHolder(ObjectGetter objectGetter, void (*finalizer)(void* argument) = nullptr, void* finalizerArgument = nullptr) NN_NOEXCEPT
    {
        AssertInitializerUsed();
        std::lock_guard<decltype(_mutex)> lk(_mutex);
        if (_initializeCount == 0)
        {
            SharedPointer<Interface> p;
            NN_RESULT_DO(objectGetter(&p));
            NN_SDK_REQUIRES(static_cast<bool>(p), "ObjectGetter should not return NULL object.");
            this->_initializeCount = 1;
            this->_p = p.Detach();
            this->_finalizer = finalizer;
            this->_finalizerArgument = finalizerArgument;
        }
        else
        {
            if (this->_finalizer)
            {
                NN_SDK_REQUIRES_EQUAL(this->_finalizer, finalizer);
                NN_SDK_REQUIRES_EQUAL(this->_finalizerArgument, finalizerArgument);
            }
            else
            {
                this->_finalizer = finalizer;
                this->_finalizerArgument = finalizerArgument;
            }
            ++_initializeCount;
        }
        NN_RESULT_SUCCESS;
    }

    /**
        @brief 指定されたオブジェクトを使用して直接ライブラリサービスオブジェクト参照を初期化します。

        @param[in] p 初期化に使用するオブジェクトを指定します。

        @pre static_cast<bool>(p) == true
        @pre 以下のいずれか一方
            - 初期化回数カウンタ == 0
            - 初期化回数カウンタ > 0 かつ this->GetObject().Get() == p.Get()

        @post 初期化回数カウンタ == この関数を呼ぶ前の初期化回数カウンタ + 1
        @post this->GetObject().Get() == p.Get()

        @details
         指定されたオブジェクト参照を使って直接初期化し、初期化回数カウンタをインクリメントします。

         初期化回数カウンタがすでに 1 以上のときには、既に保持していたオブジェクト参照と同じ参照しかこの関数に与えることはできません。

         この関数によって初期化回数カウンタが 0 より大きくなった状態で InitializeHolder() を呼ぶことは可能です。
         この場合には InitializeHolder() に渡されたオブジェクト取得関数は呼ばれません。
    */
    void InitializeHolderDirectly(SharedPointer<Interface> p) NN_NOEXCEPT
    {
        AssertInitializerUsed();
        NN_SDK_REQUIRES(static_cast<bool>(p));
        std::lock_guard<decltype(_mutex)> lk(_mutex);
        if (_initializeCount == 0)
        {
            this->_p = p.Detach();
        }
        else
        {
            NN_SDK_REQUIRES(this->GetObject().Get() == p.Get(), "Must give the same object.");
        }
        ++this->_initializeCount;
    }

    /**
        @brief 終了処理を行います。

        @pre 初期化回数カウンタ > 0

        @details
         初期化回数カウンタ == 1 のとき内部オブジェクト参照を解放します。
         初期化回数カウンタ > 1 より大きいとき、初期化回数カウンタを 1 だけ減らします。
    */
    void FinalizeHolder() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_initializeCount > 0, "Must be initialized.");
        std::lock_guard<decltype(_mutex)> lk(_mutex);
        --_initializeCount;
        if (_initializeCount == 0)
        {
            ReleaseSharedObject(_p);
            this->_p = nullptr;
            if (_finalizer)
            {
                _finalizer(_finalizerArgument);
            }
            this->_finalizer = nullptr;
            this->_finalizerArgument = nullptr;
        }
    }

    /**
        @brief ライブラリサービスオブジェクト参照を取得します。

        @pre 初期化回数カウンタ > 0

        @return 内部に保持しているライブラリサービスオブジェクト参照を返します。
    */
    SharedPointer<Interface> GetObject() const NN_NOEXCEPT
    {
        return SharedPointer<Interface>(this->operator->(), true);
    }

    /**
        @brief ライブラリが初期化されているときに、ライブラリサービスオブジェクト参照への実ポインタを取得します。

        @pre 初期化回数カウンタ > 0

        @return 内部オブジェクトへのポインタを返します。
    */
    Interface* operator->() const NN_NOEXCEPT
    {
        static_assert(std::is_trivial<ShimLibraryObjectHolder>::value, "[SF-Internal]");
        AssertInitializerUsed();
        NN_SDK_REQUIRES(_initializeCount > 0, "Must be initialized.");
        NN_SDK_ASSERT_NOT_NULL(_p);
        return _p;
    }
};

namespace detail {

const Bit32 SimpleAllInOneHipcClientManagerMagicNumber = 0x11758165;
const Bit32 SimpleAllInOneHipcSubDomainClientManagerMagicNumber = 0x14514768;

}

/**
    @brief HIPC プロキシを使用してサービスオブジェクトを取得し ShimLibraryObjectHolder に設定する統合されたユーティリティです。

    @tparam ObjectCountMax オブジェクトの最大数を指定します。

    @details
     この型は、内部に ObjectCountMax 個のプロキシオブジェクトを保持できるだけのメモリ領域を持ち、
     このメモリ領域上にプロキシオブジェクトを構築します。

     初期化したい ShimLibraryObjectHolder を引数にとって InitializeShimLibraryHolder() を呼ぶことで、
     内部で対応した ShimLibraryObjectHolder::InitializeHolder() を呼び出して ShimLibraryObjectHolder を初期化します。

     この型の変数はグローバルに配置し、必ず NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER マクロを使用して初期化してください。
     ライブラリコードのロード完了時にこの変数の初期化が終わっていることが保証されます。

    @see ShimLibraryObjectHolder, NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER
*/
template <size_t ObjectCountMax>
struct SimpleAllInOneHipcClientManager
{
private:

    void AssertInitializerUsed() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == detail::SimpleAllInOneHipcClientManagerMagicNumber, "This needs to be initialized by NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER");
    }

public:

    sf::ProxyObjectAllocator<ObjectCountMax> _allocator;
    Bit32 _magicNumber;

    /**
        @brief nn::sf::SimpleAllInOneHipcClientManager のインスタンスの初期化用マクロです。
    */
    #define NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER \
        { \
            NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER, \
            ::nn::sf::detail::SimpleAllInOneHipcClientManagerMagicNumber, \
        }

    /**
        @brief ShimLibraryObjectHolder を HIPC プロキシで得られたサービスオブジェクトで初期化します。

        @tparam Interface 対応する ShimLibraryObjectHolder の引数を指定します。通常、この型引数は省略可能です。

        @param[in] pHolder 対象となる ShimLibraryObjectHolder を指定します。
        @param[in] serviceName サービス名を指定します。
        @param[in] pMemoryResource (省略可能)メモリリソースを明示的に指定します。

        @details
         SimpleAllInOneHipcClientManager が内部に持つアロケータを使用し、
         CreateHipcProxyByName() に serviceName を渡してえられたオブジェクトを使用して ShimLibraryObjectHolder を初期化します。

         なお pMemoryResource を指定した場合には、インスタンス内部に持つアロケータは使用せずに、
         pMemoryResource を使用して構築されます。
         こちらの使用を想定する場合には ObjectCountMax を 0 に指定してもかまいません。

        @see CreateHipcProxyByName()
    */
    template <typename Interface>
    Result InitializeShimLibraryHolder(sf::ShimLibraryObjectHolder<Interface>* pHolder, const char* serviceName, MemoryResource* pMemoryResource = nullptr) NN_NOEXCEPT
    {
        AssertInitializerUsed();
        return pHolder->InitializeHolder([this, serviceName, pMemoryResource](sf::SharedPointer<Interface>* pOut) mutable -> Result
        {
            auto success = false;

            this->_allocator.Initialize();
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    this->_allocator.Finalize();
                }
            };
            if (!pMemoryResource)
            {
                pMemoryResource = _allocator.GetMemoryResource();
            }

            NN_RESULT_DO((sf::CreateHipcProxyByName<Interface, sf::MemoryResourceAllocationPolicy>(pOut, pMemoryResource, serviceName)));

            success = true;
            NN_RESULT_SUCCESS;
        },
        [](void* p)
        {
            auto this_ = static_cast<SimpleAllInOneHipcClientManager*>(p);
            this_->_allocator.Finalize();
        }, this);
    }
};

/**
    @brief HIPC-SUB プロキシを使用してサービスオブジェクトを取得し ShimLibraryObjectHolder に設定する統合されたユーティリティです。

    @tparam ObjectCountMax オブジェクトの最大数を指定します。

    @details
     この型は、内部に ObjectCountMax 個のプロキシオブジェクトを保持できるだけのメモリ領域を持ち、
     このメモリ領域上にプロキシオブジェクトを構築します。
     また、内部に HipcSimpleClientSessionManager も保持し、サブドメインを管理します。
     HipcSimpleClientSessionManager に直接アクセスしセッション数を参照・変更するような場合には、
     GetClientSessionManager() を使用してください。

     初期化したい ShimLibraryObjectHolder を引数にとって InitializeShimLibraryHolder() を呼ぶことで、
     内部で対応した ShimLibraryObjectHolder::InitializeHolder() を呼び出して ShimLibraryObjectHolder を初期化します。

     この型の変数はグローバルに配置し、必ず NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER マクロを使用して初期化してください。
     ライブラリコードのロード完了時にこの変数の初期化が終わっていることが保証されます。

    @see ShimLibraryObjectHolder, NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER
*/
template <size_t ObjectCountMax>
struct SimpleAllInOneHipcSubDomainClientManager
{
private:

    void AssertInitializerUsed() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == detail::SimpleAllInOneHipcSubDomainClientManagerMagicNumber, "This needs to be initialized by NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER");
    }

public:

    sf::ProxyObjectAllocator<ObjectCountMax> _allocator;
    std::aligned_storage<sizeof(sf::HipcSimpleClientSessionManager), NN_ALIGNOF(sf::HipcSimpleClientSessionManager)>::type _sessionManager;
    Bit32 _magicNumber;

    /**
        @brief nn::sf::SimpleAllInOneHipcSubDomainClientManager のインスタンスの初期化用マクロです。
    */
    #define NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER \
        { \
            NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER, \
            {}, \
            ::nn::sf::detail::SimpleAllInOneHipcSubDomainClientManagerMagicNumber, \
        }

    /**
        @brief ShimLibraryObjectHolder を HIPC-SUB プロキシで得られたサービスオブジェクトで初期化します。

        @tparam Interface 対応する ShimLibraryObjectHolder の引数を指定します。通常、この型引数は省略可能です。

        @param[in] pHolder 対象となる ShimLibraryObjectHolder を指定します。
        @param[in] serviceName サービス名を指定します。
        @param[in] pMemoryResource (省略可能)メモリリソースを明示的に指定します。

        @details
         SimpleAllInOneHipcClientManager が内部に持つアロケータを使用し、
         内部に保持している HipcSimpleClientSessionManager の InitializeByName() に serviceName を渡してえられたオブジェクトを使用して ShimLibraryObjectHolder を初期化します。

         なお pMemoryResource を指定した場合には、インスタンス内部に持つアロケータは使用せずに、
         pMemoryResource を使用して構築されます。
         こちらの使用を想定する場合には ObjectCountMax を 0 に指定してもかまいません。

        @see SimpleAllInOneHipcClientManager::InitializeByName()
    */
    template <typename Interface>
    Result InitializeShimLibraryHolder(sf::ShimLibraryObjectHolder<Interface>* pHolder, const char* serviceName, MemoryResource* pMemoryResource = nullptr) NN_NOEXCEPT
    {
        AssertInitializerUsed();
        return pHolder->InitializeHolder([this, serviceName, pMemoryResource](sf::SharedPointer<Interface>* pOut) mutable -> Result
        {
            auto success = false;

            this->_allocator.Initialize();
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    this->_allocator.Finalize();
                }
            };
            if (!pMemoryResource)
            {
                pMemoryResource = _allocator.GetMemoryResource();
            }

            auto pSessionManager = new (&this->_sessionManager) sf::HipcSimpleClientSessionManager;
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    pSessionManager->~HipcSimpleClientSessionManager();
                }
            };

            sf::SharedPointer<Interface> ret;
            NN_RESULT_DO((pSessionManager->InitializeByName<Interface, sf::MemoryResourceAllocationPolicy>(&ret, pMemoryResource, serviceName)));

            success = true;
            *pOut = std::move(ret);
            NN_RESULT_SUCCESS;
        },
        [](void* p)
        {
            auto this_ = static_cast<SimpleAllInOneHipcSubDomainClientManager*>(p);
            this_->GetClientSessionManager().~HipcSimpleClientSessionManager();
            this_->_allocator.Finalize();
        }, this);
    }

    /**
        @brief 内部に保持される HipcSimpleClientSessionManager の参照を取得します。

        @pre 初期化されている。
    */
    sf::HipcSimpleClientSessionManager& GetClientSessionManager() NN_NOEXCEPT
    {
        AssertInitializerUsed();
        return reinterpret_cast<sf::HipcSimpleClientSessionManager&>(this->_sessionManager);
    }

    /**
        @brief 内部に保持される HipcSimpleClientSessionManager の参照を取得します。

        @pre 初期化されている。
    */
    const sf::HipcSimpleClientSessionManager& GetClientSessionManager() const NN_NOEXCEPT
    {
        AssertInitializerUsed();
        return reinterpret_cast<const sf::HipcSimpleClientSessionManager&>(this->_sessionManager);
    }
};

}}
