﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief mem ライブラリをサービスフレームワーク上で使用するためのユーティリティです。

    @details
     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/mem/mem_StandardAllocator.h>
     - <nn/sf/sf_MemoryResource.h>

    @see nn::lmem
*/

#include <nn/nn_Common.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/sf/sf_MemoryResource.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace sf {

/**
    @brief mem::StandardAllocator を MemoryResource として扱うためのアダプタです。

    @see mem::StandardAllocator, MemoryResourceAllocator, MemoryResourceStaticAllocator, DefaultAllocationPolicy
*/
class StandardAllocatorMemoryResource
    : public MemoryResource
{
public:

    /**
        @brief 指定した mem::StandardAllocator を用いて初期化するコンストラクタです。

        @param[in] pStandardAllocator mem::StandardAllocator を指定します。

        @pre pStandardAllocator が初期化された mem::StandardAllocator である

        @post this->GetAllocator() == pStandardAllocator
    */
    explicit StandardAllocatorMemoryResource(mem::StandardAllocator* pStandardAllocator) NN_NOEXCEPT
        : m_pStandardAllocator(pStandardAllocator)
    {
    }

    /**
        @brief 関連付けられている mem::StandardAllocator を取得します。

        @return 関連付けられている mem::StandardAllocator を返します。
    */
    mem::StandardAllocator* GetAllocator() const NN_NOEXCEPT
    {
        return m_pStandardAllocator;
    }

private:

    virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStandardAllocator->Allocate(bytes, alignment);
    }

    virtual void do_deallocate(void* p, std::size_t, std::size_t) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pStandardAllocator->Free(p);
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
    {
        return this == &other;
    }

    mem::StandardAllocator* m_pStandardAllocator;

};

}}
