﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで使用されるサービスインターフェイスの共通の型 IServiceObject を定義します。
    @details
    このファイルを直接インクルードする必要はありません。 <nn/sf/sf_Types.h> をインクルードしてください。
*/

#include <nn/nn_Common.h>
#include <type_traits>
#include <utility>
#include <nn/sf/sf_Out.h>
#include <nn/sf/sf_ISharedObject.h>

/**
    @namespace nn::sf
    @brief サービスフレームワークの名前空間です。
*/
namespace nn { namespace sf {

namespace detail {

    class CmifProxyInfo;
    class CmifProxyInfoAccessor;

    class InterfaceTypeInfo;

}

/**
    @brief すべてのサービスインターフェイスの基底となるインターフェイスクラスです。

    @details
     ISharedObject を基底クラスに持ち、寿命管理が行われます。
     デストラクタを直接呼び出すことはできません。

     寿命管理についての詳細は ISharedObject を参照してください。
*/
class IServiceObject : public ISharedObject
{
protected:

    /**
        @brief protected デストラクタ

        @details
         デストラクタを直接呼ぶことはできません。
         インスタンスの破棄は Release() を経由して行ってください。
    */
    ~IServiceObject() NN_NOEXCEPT {}

private:

    friend class sf::detail::CmifProxyInfoAccessor;
    virtual detail::CmifProxyInfo* GetProxyInfo() NN_NOEXCEPT
    {
        return nullptr;
    }

    virtual detail::InterfaceTypeInfo* GetInterfaceTypeInfo() NN_NOEXCEPT
    {
        return nullptr;
    }

};

/**
    @brief サービスインターフェイスの共有ポインタに対する Out の特殊化です。

    @tparam Interface サービスインターフェイスを指定します。
*/
template <typename Interface>
class Out<SharedPointer<Interface>>
{
    template <typename, typename>
    friend class nn::sf::Out;
private:

    detail::SharedPointerBase* m_Out;

    NN_IMPLICIT Out(detail::SharedPointerBase* p) NN_NOEXCEPT
        : m_Out(p)
    {
    }

public:

    /**
        @brief コピーコンストラクタ

        @param[in] x コピー元オブジェクト
    */
    Out(const Out& x) NN_NOEXCEPT
        : m_Out(x.m_Out)
    {
    }

    /**
        @brief コンストラクタ: 与えられたポインタで初期化します。

        @param[in] pOut 変換元の SharedPointer へのポインタ
    */
    NN_IMPLICIT Out(SharedPointer<Interface>* pOut) NN_NOEXCEPT
        : m_Out(&pOut->m_Base)
    {
    }

    /**
        @brief コンストラクタ: 与えられた Out オブジェクトで初期化します。

        @tparam U 変換元の Out オブジェクトのインターフェイス型を指定します。std::is_base_of<Interface, U>::value == true が必要です。
        @param[in] pOut 変換元の Out オブジェクト
    */
    template <typename U>
    NN_IMPLICIT Out(Out<SharedPointer<U>> pOut) NN_NOEXCEPT
        : m_Out(pOut.m_Out)
    {
        static_assert(std::is_base_of<U, Interface>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<U, Interface>::value");
    }

    /**
        @brief コンストラクタ: 与えられたポインタで初期化します。

        @param[in] pOut 変換元の SharedPointer へのポインタ
    */
    template <typename U>
    NN_IMPLICIT Out(SharedPointer<U>* pOut) NN_NOEXCEPT
        : m_Out(&pOut->m_Base)
    {
        static_assert(std::is_base_of<U, Interface>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<U, Interface>::value");
    }

    /**
        @brief バッファに値をセットします。

        @param[in] p セットする値
    */
    void Set(SharedPointer<Interface> p) const NN_NOEXCEPT
    {
        *m_Out = std::move(p.m_Base);
    }

    /**
        @brief バッファに値をセットします。

        @tparam U セットするインターフェイスの型を指定します。std::is_base_of<Interface, U>::value == true が必要です。
        @param[in] p セットする値
    */
    template <typename U>
    void Set(SharedPointer<U> p) const NN_NOEXCEPT
    {
        static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
        *m_Out = std::move(p.m_Base);
    }

    /**
        @brief Out::operator*() の返り値への代入をサポートするためのヘルパです。
    */
    class DerefProxy
    {
        template <typename, typename>
        friend class nn::sf::Out;
    private:

        const Out* m_pTarget;

        explicit DerefProxy(const Out* pTarget) NN_NOEXCEPT
            : m_pTarget(pTarget)
        {
        }

    public:

        /**
            @brief 代入演算子: 関連付けられた Out オブジェクトへの設定を行います。

            @param[in] p 代入元の共有ポインタを指定します。

            @return *this を返します。
        */
        DerefProxy& operator=(SharedPointer<Interface> p) NN_NOEXCEPT
        {
            m_pTarget->Set(std::move(p));
            return *this;
        }

        /**
            @brief 代入演算子: 関連付けられた Out オブジェクトへの設定を行います。

            @tparam U 代入するインターフェイスの型を指定します。std::is_base_of<Interface, U>::value == true が必要です。
            @param[in] p 代入元の共有ポインタを指定します。

            @return *this を返します。
        */
        template <typename U>
        DerefProxy& operator=(SharedPointer<U> p) NN_NOEXCEPT
        {
            static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
            m_pTarget->Set(std::move(p));
            return *this;
        }

    };

    /**
        @brief 逆参照演算子: Out::operator*() の返り値への代入をサポートするためのヘルパを返します。

        @return Out::operator*() の返り値への代入をサポートするためのヘルパを返します。
    */
    DerefProxy operator*() const NN_NOEXCEPT
    {
        return DerefProxy(this);
    }

    template <typename U>
    Out<SharedPointer<U>> DownCast() NN_NOEXCEPT
    {
        return Out<SharedPointer<U>>(this->m_Out);
    }

};

// インターフェイス直の Out はサポートされない
template <typename I>
class Out<I, typename std::enable_if<std::is_base_of<IServiceObject, I>::value>::type>
{
    static_assert(!std::is_same<I, I>::value, "Out<I : nn::sf::IServiceObject> is not supported. Use Out<SharedPointer<I>>, instead.");
};

}}
