﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC を使用する際に、セッションハンドルによるプロキシオブジェクト生成を提供します。
    @details
     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/sf/sf_HipcPortCommon.h>
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_StandardAllocationPolicy.h>

#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/cmif/client/sf_CmifProxyFactory.h>
#include <nn/sf/hipc/client/sf_HipcClientMessage.h>
#include <nn/sf/hipc/client/sf_HipcManagerAccessor.h>
#include <nn/sf/detail/sf_CmifProxyInfoAccessor.h>

namespace nn { namespace sf {

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
struct UnspecifiedForHipcClientSessionHandle;
/**
    @brief クライアントセッションハンドルを表す型です。

    @details
     この型は内部型のエイリアスです。内部型を直接使わず、この型を使用するようにしてください。
*/
typedef UnspecifiedForHipcClientSessionHandle HipcClientSessionHandle;
#else
typedef nn::sf::hipc::HipcClientSessionHandle HipcClientSessionHandle;
#endif

namespace detail {

    struct HipcRefAccessor;

}

/**
    @brief CreateHipcProxy 系の関数から得られるサービスオブジェクト専用の共有ポインタを表します。

    @tparam Interface インターフェイスを指定します。

    @details
     このクラスは SharedPointer<Interface> を継承しており SharedPointer<Interface> として振る舞うことができます。
     また、 SharedPointer<Interface> への変換が可能です。
*/
template <typename Interface>
class HipcRef
    : public SharedPointer<Interface>
{
    friend struct detail::HipcRefAccessor;
public:

    /**
        @brief デフォルトコンストラクタ: 何も指さない HipcRef として初期化します。

        @post static_cast<bool>(*this) == false
    */
    HipcRef() NN_NOEXCEPT
    {
    }

    /**
        @brief (内部実装用) クライアントセッションハンドルを取得します。

        @return クライアントセッションハンドルを返します。

        @pre static_cast<bool>(*this) == true

        @details
         内部実装用です。通常は直接使用する必要はありません。
    */
    HipcClientSessionHandle GetClientSessionHandle() const NN_NOEXCEPT
    {
        return GetProxyBaseObject()->GetClientHandle();
    }

    template <typename AllocationPolicy>
    Result CloneSession(HipcRef<Interface>* pOut, int32_t tag, typename AllocationPolicy::Allocator* pAllocator = nullptr) NN_NOEXCEPT;

private:

    HipcRef(Interface* p, bool addReference) NN_NOEXCEPT
        : SharedPointer<Interface>(p, addReference)
    {
    }

    typedef hipc::client::Hipc2ProxyKind ProxyKind;

    typename ProxyKind::ProxyBaseObject* GetProxyBaseObject() const NN_NOEXCEPT
    {
        return static_cast<typename ProxyKind::ProxyBaseObject*>(detail::CmifProxyInfoAccessor::GetCmifBaseObject(this->Get()));
    }

};

namespace detail {

    struct HipcRefAccessor
    {
        template <typename Interface>
        static HipcRef<Interface> MakeHipcRef(Interface* p, bool addReference)
        {
            return HipcRef<Interface>(p, addReference);
        }

        template <typename Interface, typename AllocationPolicy>
        static HipcRef<Interface> CreateHipcProxyByHandle(typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle)
        {
            typedef nn::sf::cmif::client::CmifProxyFactory<
                Interface,
                typename HipcRef<Interface>::ProxyKind,
                AllocationPolicy
            > ProxyFactory;
            return MakeHipcRef<Interface>(ProxyFactory::CreateProxyObject(pAllocator, handle), false);
        }
    };

    template <typename Interface, typename AllocationPolicy>
    inline Result CloneHipcRefImpl(HipcRef<Interface>* pOut, HipcRef<Interface> source, int32_t tag, typename AllocationPolicy::Allocator* pAllocator) NN_NOEXCEPT
    {
        HipcClientSessionHandle handle;
        NN_RESULT_DO(sf::hipc::client::DuplicateSessionEx(&handle, source.GetClientSessionHandle(), tag));
        *pOut = HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(pAllocator, handle);
        NN_RESULT_SUCCESS;
    }

}

template <typename Interface>
template <typename AllocationPolicy>
inline Result HipcRef<Interface>::CloneSession(HipcRef<Interface>* pOut, int32_t tag, typename AllocationPolicy::Allocator* pAllocator) NN_NOEXCEPT
{
    return detail::CloneHipcRefImpl<Interface, AllocationPolicy>(pOut, *this, tag, pAllocator);
}

}}
