﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief 拡張ヒープをサービスフレームワークで使用するためのユーティリティを定義します。

    @details
     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/lmem/lmem_ExpHeap.h>

    @see nn::lmem
*/

#include <nn/nn_Common.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <type_traits>

namespace nn { namespace sf {
/**
    @brief 拡張ヒープを SF のアロケータとして使用するアダプタです。

    @details
     使用前に Attach() を呼び出し、初期化済みの拡張ヒープハンドルをアタッチする必要があります。

     Policy メンバは、状態ありのアロケーションポリシーとして使用できます。
*/
struct ExpHeapAllocator
{
    /**
        @brief 拡張ヒープハンドルをアタッチします。

        @param[in] handle 拡張ヒープハンドル

        @pre アタッチされていない
        @pre handle が正しく初期化された拡張ヒープハンドルである(nn::lmem::CreateExpHeap の返り値である)

        @post アロケーション可能状態である

        @see nn::lmem::CreateExpHeap
    */
    void Attach(nn::lmem::HeapHandle handle) NN_NOEXCEPT
    {
        this->_handle = handle;
        os::InitializeMutex(&_mutex, false, 0);
    }

    /**
        @brief 拡張ヒープハンドルをデタッチします。

        @pre アロケーション可能な状態である(アタッチされている)

        @post 再アタッチ可能な状態である

        @details
         拡張ヒープハンドルをデタッチし、アロケーションができない状態にします。
    */
    void Detach() NN_NOEXCEPT
    {
        os::FinalizeMutex(&_mutex);
        this->_handle = 0;
    }

    /**
        @brief サービスフレームワークの状態ありのアロケーションポリシーとして使用できる型です。
    */
    typedef StatefulAllocationPolicy<ExpHeapAllocator> Policy;

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        os::LockMutex(&_mutex);
        auto p = nn::lmem::AllocateFromExpHeap(_handle, size);
        os::UnlockMutex(&_mutex);
        return p;
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        os::LockMutex(&_mutex);
        nn::lmem::FreeToExpHeap(_handle, p);
        os::UnlockMutex(&_mutex);
    }

    nn::lmem::HeapHandle _handle;
    nn::os::MutexType _mutex;
};
static_assert(std::is_pod<ExpHeapAllocator>::value, "[SF-Internal]");

/**
    @brief SF に適合したアロケータで、内部に拡張ヒープとそのためのバッファを持つ構造体テンプレートです。

    @details
     使用前に Initialize() で初期化をする必要があります。

     Policy メンバは、状態なしのアロケーションポリシーとして使用できます。
*/
template <size_t Size, typename Tag = void>
struct ExpHeapStaticAllocator
{

    /**
        @brief 内部拡張ヒープを初期化します

        @param[in] option 内部拡張ヒープ作成時のオプションパラメータ(nn::lmem::CreateExpHeap() の第 3 引数に渡されます)

        @post アロケーション可能状態である

        @see nn::lmem::CreateExpHeap()
    */
    static void Initialize(int option) NN_NOEXCEPT
    {
        static_assert(std::is_pod<ExpHeapStaticAllocator>::value, "[SF-Internal]");
        _globals.allocator.Attach(nn::lmem::CreateExpHeap(&_globals.buffer, Size, option));
    }

    /**
        @brief 拡張ヒープハンドルを直接指定して内部拡張ヒープを初期化します

        @param[in] handle 初期化された拡張ヒープハンドルを指定します

        @pre handle が初期化された拡張ヒープハンドルである
        @post アロケーション可能状態である

        @details
         この関数を使用して初期化した場合には、型引数 Size で指定したサイズの内部バッファは使用されません。
         この関数を使用して初期化する際には、Size には 0 を指定してください。

        @see nn::lmem::CreateExpHeap()
    */
    static void Initialize(lmem::HeapHandle handle) NN_NOEXCEPT
    {
        static_assert(std::is_pod<ExpHeapStaticAllocator>::value, "[SF-Internal]");
        _globals.allocator.Attach(handle);
    }

    /**
        @brief サービスフレームワークの状態なしのアロケーションポリシーとして使用できる型です。
    */
    typedef StatelessAllocationPolicy<ExpHeapStaticAllocator<Size, Tag>> Policy;

    static void* Allocate(size_t size) NN_NOEXCEPT
    {
        return _globals.allocator.Allocate(size);
    }

    static void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        _globals.allocator.Deallocate(p, size);
    }

    struct Globals
    {
        ExpHeapAllocator allocator;
        typename std::aligned_storage<Size == 0 ? 1 : Size>::type buffer;
    };

    static Globals _globals;

};

template <size_t Size, typename Tag>
typename ExpHeapStaticAllocator<Size, Tag>::Globals ExpHeapStaticAllocator<Size, Tag>::_globals;

}}
