﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>
#include <nn/nn_Abort.h>
#include <atomic>
#include <type_traits>

namespace nn { namespace sf { namespace impl {

/**
    @brief 固定サイズの一つの領域だけをアロケートすることができるアロケータです。

    @tprama T アロケートする型です。

    @detail
    この型は StatelessTypedAllocationPolicy テンプレートの引数に取ることができます。
    通常は、この型を直接使うのではなく、typedef の StaticOneAllocationPolicy を使用すれば十分です。
*/
template <typename T>
class StaticOneAllocator
{
private:

    typedef typename std::aligned_storage<sizeof(T), NN_ALIGNOF(T)>::type Storage;

    struct Holder
    {
        Storage storage;
        std::atomic_flag allocated;
    };
    // static_assert(std::is_literal_type<Holder>::value, ""); // VS2013 で引っかかるため暫定的にコメントアウト

    static Holder s_Holder;

public:

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(size == sizeof(s_Holder.storage));
        return s_Holder.allocated.test_and_set() ? nullptr : &s_Holder.storage;
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(size == sizeof(s_Holder.storage));
        NN_ABORT_UNLESS(p == &s_Holder.storage);
        s_Holder.allocated.clear();
    }

};

template <typename T>
typename StaticOneAllocator<T>::Holder StaticOneAllocator<T>::s_Holder = { {}, ATOMIC_FLAG_INIT };

/**
    @brief 固定サイズの一つの領域だけをアロケートするアロケーションポリシーです。
*/
typedef StatelessTypedAllocationPolicy<StaticOneAllocator> StaticOneAllocationPolicy;

}}}
