﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/detail/sf_BinaryVersion.h>
#if NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE == NN_SF_LATEST_BINARY_VERSION

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/sf/hipc/server/sf_HipcServerSessionManagerWithDomain.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_Event.h>

#include <mutex>
#include <cstring>
#include <utility>
#include <array>
#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>
#include <nn/util/util_BitUtil.h>
#include <nn/sf/hipc/sf_HipcTlsBuffer.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/lmem/lmem_UnitHeap.h>

namespace nn { namespace sf { namespace hipc { namespace server { inline namespace v1 {

namespace detail {

class HipcAllInOneServerManagerBase
    : public HipcServerSessionManagerWithDomain
{
protected:

    using HipcServerSessionManagerWithDomain::DomainStorage;
    using HipcServerSessionManagerWithDomain::DomainEntryStorage;

    struct ListTagForAll
    {
    };

    HipcAllInOneServerManagerBase(DomainEntryStorage* entryBuffer, int entryBufferCount) NN_NOEXCEPT;
    ~HipcAllInOneServerManagerBase() NN_NOEXCEPT;

    class SessionForAllInOne
        : public HipcServerSession
    {
        friend class HipcAllInOneServerManagerBase;
    private:
        bool m_Received;
        void* m_SavedMessageBuffer;
        size_t m_SavedMessageBufferSize;
    };

    class PortForAllInOne
        : public os::MultiWaitHolderType
        , private util::IntrusiveListBaseNode<PortForAllInOne, ListTagForAll>
    {
        friend class HipcAllInOneServerManagerBase;
        friend class util::IntrusiveListBaseNodeTraits<PortForAllInOne, ListTagForAll>;
    private:
        int m_Index;
        HipcServerPortHandle m_PortHandle;
        char m_ServiceName[hipc::HipcServiceNameLengthMax + 1];
        bool m_ServiceNameManaged;
        cmif::server::CmifServerObjectInfo m_Object;
    };

    // 上位でオーバライドが必要な関数
    virtual Result OnNeedsToAccept(int portIndex, PortForAllInOne* pPort) NN_NOEXCEPT = 0;
    virtual PortForAllInOne* AllocatePort() NN_NOEXCEPT = 0;
    virtual HipcServerSession* AllocateServerSession() NN_NOEXCEPT NN_OVERRIDE = 0;
    virtual void DeallocateServerSession(HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE = 0;

    // AllocateSavedMessageBuffer は GetSavedMessageBuffer として扱う。
    // DeallocateSavedMessageBuffer は nop 関数とする。
    // バイナリ互換維持のため改名は行わない。
    virtual void* AllocateSavedMessageBuffer(size_t size, HipcServerSession* pSession) NN_NOEXCEPT = 0;
    virtual void DeallocateSavedMessageBuffer(void* p, HipcServerSession* pSession) NN_NOEXCEPT = 0;
    NN_FORCEINLINE void* GetSavedMessageBuffer(size_t size, HipcServerSession* pSession) NN_NOEXCEPT
    {
        return AllocateSavedMessageBuffer(size, pSession);
    }

    // OnNeedsToAccept 実装用関数
    template <typename Interface>
    Result AcceptImpl(PortForAllInOne* pPort, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        return HipcServerSessionManager::Accept(pPort->m_PortHandle, std::move(p));
    }

public:

    // ポート初期化
    void InitializePort(int portIndex, HipcServerPortHandle portHandle) NN_NOEXCEPT;
    Result InitializePort(int portIndex, int32_t maxSession, const char* serviceName) NN_NOEXCEPT;

    template <typename Interface>
    void RegisterObjectForPort(SharedPointer<Interface> p, HipcServerPortHandle portHandle) NN_NOEXCEPT
    {
        return InitializePortImpl(0, cmif::server::CmifServerObjectInfo(std::move(p)), portHandle);
    }

    template <typename Interface>
    Result RegisterObjectForPort(SharedPointer<Interface> p, int32_t maxSession, const char* serviceName) NN_NOEXCEPT
    {
        return InitializePortImpl(0, cmif::server::CmifServerObjectInfo(std::move(p)), maxSession, serviceName);
    }

    // 開始・終了
    void Start() NN_NOEXCEPT;
    void RequestStop() NN_NOEXCEPT;

    // 待機
    os::MultiWaitHolderType* WaitSignaled() NN_NOEXCEPT;

    // ポート
    static const uintptr_t PortTag = 2;
    Result ProcessAsPort(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    // セッション
    static const uintptr_t SessionTag = 1;
    Result ReceiveAsSession(os::MultiWaitHolderType* p) NN_NOEXCEPT;
    Result ReceiveAsSession(os::MultiWaitHolderType* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;
    Result ProcessAsSession(os::MultiWaitHolderType* p) NN_NOEXCEPT;
    Result ProcessAsSession(os::MultiWaitHolderType* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;
    Result ProcessInvokeRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT;
    Result ProcessInvokeRequestWithDefer(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    // ユーティリティ
    Result ProcessAuto(os::MultiWaitHolderType* p) NN_NOEXCEPT;
    void WaitAndProcessAuto() NN_NOEXCEPT;
    void LoopAuto() NN_NOEXCEPT;

    // ユーザシグナル
    void AddUserWaitHolder(os::MultiWaitHolderType* p) NN_NOEXCEPT;

private:

    class SafeMutex
        : public os::Mutex
    {
    public:

        SafeMutex() :
            Mutex(false)
        {
        }

    };

    os::MultiWaitType m_MultiWait;

    os::Event m_StopRequestEvent;
    os::MultiWaitHolderType m_StopRequestEventHolder;
    os::Event m_NotificationEvent;
    os::MultiWaitHolderType m_NotificationEventHolder;

    SafeMutex m_SelectMutex;

    util::IntrusiveList<PortForAllInOne, util::IntrusiveListBaseNodeTraits<PortForAllInOne, ListTagForAll>> m_AllPortList;

    SafeMutex m_DeferredListMutex;
    os::MultiWaitType m_DeferredList;

    void RegisterPortImpl(PortForAllInOne* pPort, HipcServerPortHandle portHandle) NN_NOEXCEPT;
    virtual void RegisterServerSessionToWait(HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE final;
    void LinkToDeferredList(os::MultiWaitHolderType* p) NN_NOEXCEPT;
    void LinkDeferred() NN_NOEXCEPT;
    void InitializePortImpl(int portIndex, cmif::server::CmifServerObjectInfo&& x, HipcServerPortHandle portHandle) NN_NOEXCEPT;
    Result InitializePortImpl(int portIndex, cmif::server::CmifServerObjectInfo&& x, int32_t maxSessions, const char* serviceName) NN_NOEXCEPT;

    bool WaitAndProcessAutoImpl() NN_NOEXCEPT;
};

} // detail

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
class HipcSimpleAllInOneServerManager
    : public detail::HipcAllInOneServerManagerBase
{
private:

    virtual Result OnNeedsToAccept(int portIndex, PortForAllInOne* pPort) NN_NOEXCEPT NN_OVERRIDE = 0;

private:

    os::Mutex m_Mutex;
    PortForAllInOne m_Ports[PortCountMax];
    bool m_PortAllocated[PortCountMax];
    SessionForAllInOne m_Sessions[SessionCountMax];
    bool m_SessionAllocated[SessionCountMax];
    char m_PointerBufferStorage[Option::PointerTransferBufferSize * SessionCountMax + 16];
    char m_SavedMessageBufferStorage[(Option::CanDeferInvokeRequest ? hipc::MessageBufferSizeOnTls : 0) * SessionCountMax + 16];
    uintptr_t m_PointerBufferHead;
    uintptr_t m_SavedMessageBufferHead;

    std::array<DomainStorage, Option::SubDomainCountMax> m_DomainStorage;
    std::array<bool, Option::SubDomainCountMax> m_DomainStorageAllocated;
    std::array<DomainEntryStorage, Option::ObjectInSubDomainCountMax> m_DomainEntryStorage;

    virtual PortForAllInOne* AllocatePort() NN_NOEXCEPT NN_OVERRIDE final
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        for (size_t i = 0; i < PortCountMax; ++i)
        {
            if (!m_PortAllocated[i])
            {
                m_PortAllocated[i] = true;
                return &m_Ports[i];
            }
        }
        return nullptr;
    }

    virtual HipcServerSession* AllocateServerSession() NN_NOEXCEPT NN_OVERRIDE final
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        for (size_t i = 0; i < SessionCountMax; ++i)
        {
            if (!m_SessionAllocated[i])
            {
                m_SessionAllocated[i] = true;
                return &m_Sessions[i];
            }
        }
        return nullptr;
    }

    virtual void DeallocateServerSession(HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        auto p = static_cast<SessionForAllInOne*>(pSession);
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        auto i = p - m_Sessions;
        m_SessionAllocated[i] = false;
    }

    virtual void* AllocatePointerTransferBuffer(size_t* pOut, HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        if (NN_STATIC_CONDITION(Option::PointerTransferBufferSize == 0))
        {
            *pOut = 0;
            return nullptr;
        }
        auto p = static_cast<SessionForAllInOne*>(pSession);
        auto i = p - m_Sessions;
        *pOut = Option::PointerTransferBufferSize;
        return reinterpret_cast<void*>(m_PointerBufferHead + i * Option::PointerTransferBufferSize);
    }

    virtual void DeallocatePointerTransferBuffer(HipcServerSession* pSession, void* p, size_t size) NN_NOEXCEPT final
    {
        NN_UNUSED(pSession);
        NN_UNUSED(p);
        NN_UNUSED(size);
    }

    virtual void* AllocateSavedMessageBuffer(size_t size, HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        if (!NN_STATIC_CONDITION(Option::CanDeferInvokeRequest))
        {
            NN_UNUSED(size);
            NN_UNUSED(pSession);
            return nullptr;
        }
        auto p = static_cast<SessionForAllInOne*>(pSession);
        auto i = p - m_Sessions;
        return reinterpret_cast<void*>(m_SavedMessageBufferHead + i * hipc::MessageBufferSizeOnTls);
    }

    virtual void DeallocateSavedMessageBuffer(void* p, HipcServerSession* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        NN_UNUSED(p);
        NN_UNUSED(pSession);
    }

    virtual void* AllocateDomainStorage() NN_NOEXCEPT NN_OVERRIDE final
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        for (size_t i = 0; i < Option::SubDomainCountMax; ++i)
        {
            if (!m_DomainStorageAllocated[i])
            {
                m_DomainStorageAllocated[i] = true;
                return &m_DomainStorage[i];
            }
        }
        return nullptr;
    }

    virtual void DeallocateDomainStorage(void* pDomain) NN_NOEXCEPT NN_OVERRIDE final
    {
        auto p = static_cast<DomainStorage*>(pDomain);
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        auto i = p - m_DomainStorage.data();
        m_DomainStorageAllocated[i] = false;
    }

protected:

    HipcSimpleAllInOneServerManager() NN_NOEXCEPT
        : detail::HipcAllInOneServerManagerBase(reinterpret_cast<DomainEntryStorage*>(&m_DomainEntryStorage), Option::ObjectInSubDomainCountMax)
        , m_Mutex(false)
    {
        std::memset(m_PortAllocated, 0, sizeof(m_PortAllocated));
        std::memset(m_SessionAllocated, 0, sizeof(m_SessionAllocated));
        m_DomainStorageAllocated.fill(false);
        this->m_PointerBufferHead = util::align_up(reinterpret_cast<uintptr_t>(m_PointerBufferStorage), 16);
        this->m_SavedMessageBufferHead = util::align_up(reinterpret_cast<uintptr_t>(m_SavedMessageBufferStorage), 16);
    }

};

}}}}}

#elif NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE == 0
    #include <nn/sf/hipc/server/v0/sf_HipcAllInOneServerManager-v0.h>
#else
    #error "invalid NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE"
#endif
