﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/sf/hipc/client/sf_HipcClientSessionManager.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/sf_HipcPortCommon.h>
#include <nn/sf/sf_HipcClientProxyByHandle.h>
#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <atomic>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Result.h>

namespace nn { namespace sf { namespace hipc { namespace client {

template <typename SessionResourceManager>
class HipcSimpleClientSessionManagerBase
{
protected:

    struct Entry
    {
        HipcClientSessionHandle handle;
        Entry* pNext;
    };

private:

    class HandleAllocator
    {
    private:

        Entry* m_Entries;

        mutable os::Mutex m_Mutex;
        mutable os::ConditionVariable m_ConditionVariable;
        Entry* m_pAllocatable;
        Entry* m_pEmpty;

        SessionResourceManager* m_pSessionResourceManager;

        HipcClientSessionHandle AllocateUnsafe() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread(), "[SF-Internal]");
            NN_SDK_ASSERT_NOT_NULL(m_pAllocatable);
            auto allocated = m_pAllocatable;
            this->m_pAllocatable = allocated->pNext;
            allocated->pNext = m_pEmpty;
            this->m_pEmpty = allocated;
            return allocated->handle;
        }

        HipcClientSessionHandle AllocateImpl() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            for (;;)
            {
                if (m_pSessionResourceManager->OnAcquire())
                {
                    if (m_pAllocatable)
                    {
                        return AllocateUnsafe();
                    }
                    m_pSessionResourceManager->OnRelease();
                }
                m_ConditionVariable.Wait(m_Mutex);
            }
        }

        void DeallocateUnsafe(HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread(), "[SF-Internal]");
            NN_SDK_ASSERT_NOT_NULL(m_pEmpty);
            auto deallocated = m_pEmpty;
            this->m_pEmpty = deallocated->pNext;
            deallocated->pNext = m_pAllocatable;
            this->m_pAllocatable = deallocated;
            deallocated->handle = handle;
        }

        void DeallocateImpl(HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            auto needsSignal = m_pAllocatable == nullptr;
            DeallocateUnsafe(handle);
            auto needsBroadcast = m_pSessionResourceManager->OnRelease();
            if (needsBroadcast)
            {
                m_ConditionVariable.Broadcast();
            }
            else if (needsSignal)
            {
                m_ConditionVariable.Signal();
            }
        }

    public:

        HandleAllocator(Entry* entries, int entryCount, SessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
            : m_Entries(entries)
            , m_Mutex(false)
            , m_pAllocatable(nullptr)
            , m_pSessionResourceManager(pSessionResourceManager)
        {
            this->m_pEmpty = m_Entries + 0;
            for (auto i = 1; i < entryCount; ++i)
            {
                m_Entries[i - 1].pNext = &m_Entries[i];
            }
            m_Entries[entryCount - 1].pNext = nullptr;
        }

        void AddHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            DeallocateImpl(handle);
        }

        HipcClientSessionHandle RemoveHandle() NN_NOEXCEPT
        {
            return AllocateImpl();
        }

        HipcClientSessionHandle Allocate() NN_NOEXCEPT
        {
            return AllocateImpl();
        }

        bool TryAllocate(HipcClientSessionHandle* pOut) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            if (!m_pAllocatable)
            {
                return false;
            }
            *pOut = AllocateUnsafe();
            return true;
        }

        void Deallocate(HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            DeallocateImpl(handle);
        }

    };

    class Impl
        : private hipc::client::HipcClientSessionManager
    {
    private:

        typedef hipc::client::HipcClientSessionManager Base;

        HandleAllocator m_HandleAllocator;

        virtual HipcClientSessionHandle Allocate() NN_NOEXCEPT
        {
            return m_HandleAllocator.Allocate();
        }

        virtual void Deallocate(HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            m_HandleAllocator.Deallocate(handle);
        }

        int m_EntryCount;
        int m_SessionCount;
        mutable os::Mutex m_Mutex;

        Result AddHandleImpl() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            HipcClientSessionHandle handle;
            NN_RESULT_DO(Base::CreateNewSession(&handle));
            m_HandleAllocator.AddHandle(handle);
            ++m_SessionCount;
            NN_RESULT_SUCCESS;
        }

        void RemoveHandleImpl() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            nn::sf::hipc::client::CloseClientSessionHandleSafely(m_HandleAllocator.RemoveHandle());
            --m_SessionCount;
        }

        Result SetSessionCountImpl(int sessionCount) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            auto n = sessionCount - m_SessionCount;
            if (n > 0)
            {
                auto countToRemove = 0;
                NN_UTIL_SCOPE_EXIT
                {
                    for (auto i = 0; i < countToRemove; ++i)
                    {
                        RemoveHandleImpl();
                    }
                };
                for (auto i = 0; i < n; ++i)
                {
                    NN_RESULT_DO(AddHandleImpl());
                    ++countToRemove;
                }
                countToRemove = 0;
                NN_RESULT_SUCCESS;
            }
            else
            {
                for (auto i = 0; i < -n; ++i)
                {
                    RemoveHandleImpl();
                }
                NN_RESULT_SUCCESS;
            }
        }

    public:

        Impl(Entry* entries, int entryCount, SessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
            : m_HandleAllocator(entries, entryCount, pSessionResourceManager)
            , m_EntryCount(entryCount)
            , m_SessionCount(0)
            , m_Mutex(false)
        {
        }

        ~Impl() NN_NOEXCEPT
        {
            Release();
        }

        void Release() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            static_cast<void>(SetSessionCountImpl(0));
        }

        template <typename Interface, typename AllocationPolicy>
        Result InitializeByHandle(SharedPointer<Interface>* pOut, HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            NN_RESULT_DO((Base::InitializeByHandle<Interface, AllocationPolicy>(pOut, handle)));
            m_HandleAllocator.AddHandle(handle);
            ++m_SessionCount;
            NN_RESULT_SUCCESS;
        }

        template <typename Interface, typename AllocationPolicy>
        Result InitializeByHandle(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT
        {
            NN_RESULT_DO((Base::InitializeByHandle<Interface, AllocationPolicy>(pOut, pAllocator, handle)));
            m_HandleAllocator.AddHandle(handle);
            ++m_SessionCount;
            NN_RESULT_SUCCESS;
        }

        Result IncreaseSession() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_RESULT_DO(AddHandleImpl());
            NN_RESULT_SUCCESS;
        }

        Result SetSessionCount(int sessionCount) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_SDK_REQUIRES(0 < sessionCount);
            NN_SDK_REQUIRES(sessionCount <= m_EntryCount);
            NN_ABORT_UNLESS(sessionCount <= m_EntryCount, "[SF-USER-PreConditionViolation]");
            return SetSessionCountImpl(sessionCount);
        }

        int GetSessionCount() const NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            return m_SessionCount;
        }

    };

    Impl m_Impl;

protected:

    HipcSimpleClientSessionManagerBase(Entry* entries, int entryCount, SessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
        : m_Impl(entries, entryCount, pSessionResourceManager)
    {
    }

public:

    template <typename Interface, typename AllocationPolicy>
    Result InitializeByHandle(SharedPointer<Interface>* pOut, HipcClientSessionHandle handle) NN_NOEXCEPT
    {
        return m_Impl.template InitializeByHandle<Interface, AllocationPolicy>(pOut, handle);
    }

    template <typename Interface, typename AllocationPolicy>
    Result InitializeByHandle(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT
    {
        return m_Impl.template InitializeByHandle<Interface, AllocationPolicy>(pOut, pAllocator, handle);
    }

    template <typename Interface, typename AllocationPolicy>
    Result InitializeByName(SharedPointer<Interface>* pOut, const char* serviceName) NN_NOEXCEPT
    {
        hipc::InitializeHipcServiceResolution();
        NN_UTIL_SCOPE_EXIT
        {
            hipc::FinalizeHipcServiceResolution();
        };
        HipcClientSessionHandle clientSessionHandle;
        NN_RESULT_DO(hipc::ConnectToHipcService(&clientSessionHandle, serviceName));
        return m_Impl.template InitializeByHandle<Interface, AllocationPolicy>(pOut, clientSessionHandle);
    }

    template <typename Interface, typename AllocationPolicy>
    Result InitializeByName(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT
    {
        hipc::InitializeHipcServiceResolution();
        NN_UTIL_SCOPE_EXIT
        {
            hipc::FinalizeHipcServiceResolution();
        };
        HipcClientSessionHandle clientSessionHandle;
        NN_RESULT_DO(hipc::ConnectToHipcService(&clientSessionHandle, serviceName));
        return m_Impl.template InitializeByHandle<Interface, AllocationPolicy>(pOut, pAllocator, clientSessionHandle);
    }

    Result IncreaseSession() NN_NOEXCEPT
    {
        return m_Impl.IncreaseSession();
    }

    Result SetSessionCount(int sessionCount) NN_NOEXCEPT
    {
        return m_Impl.SetSessionCount(sessionCount);
    }

    int GetSessionCount() const NN_NOEXCEPT
    {
        return m_Impl.GetSessionCount();
    }

    void Release() NN_NOEXCEPT
    {
        m_Impl.Release();
    }

};

template <int N, typename SessionResourceManager>
class HipcSimpleClientSessionManagerBaseN
    : public HipcSimpleClientSessionManagerBase<SessionResourceManager>
{
private:

    typename HipcSimpleClientSessionManagerBase<SessionResourceManager>::Entry m_Entries[N];

public:

    NN_IMPLICIT HipcSimpleClientSessionManagerBaseN(SessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
        : HipcSimpleClientSessionManagerBase<SessionResourceManager>(m_Entries, N, pSessionResourceManager)
    {
    }

};

class DefaultSessionResourceManager
{
public:
    bool OnAcquire() NN_NOEXCEPT
    {
        return true;
    }
    bool OnRelease() NN_NOEXCEPT
    {
        return false;
    }
};

}}}}
