﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/cmif/client/sf_CmifClientMessage.h>

#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/cmif/sf_InlineContext.h>
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h>
#include <nn/sf/hipc/detail/sf_HipcMessageFormat.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor2.h>

#include <utility>
#include <limits>
#include <memory>
#include <array>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/sf/detail/sf_Config.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <nn/sf/hipc/client/sf_HipcCloseClientSessionHandleSafely.h>
#include <nn/sf/detail/sf_CommonUtil.h>
#include <algorithm>
#include <nn/sf/hipc/client/sf_HipcManagerAccessor.h>
#include <atomic>
#include <nn/os/os_Mutex.h>

namespace nn { namespace sf { namespace hipc { namespace client {

class Hipc2ClientProxyBaseObject
    : public nn::sf::cmif::client::CmifBaseObject
{
private:

    bool m_Attached;
    mutable std::atomic<bool> m_PointerBufferSizeCached;
    mutable std::atomic<uint16_t> m_PointerBufferSize;
    HipcClientSessionHandle m_ClientHandle;

    void CachePointerBufferSize() const NN_NOEXCEPT
    {
        if (!m_PointerBufferSizeCached)
        {
            uint16_t pointerBufferSize;
            // TODO: ハンドリング
            NN_ABORT_UNLESS_RESULT_SUCCESS(QueryPointerBufferSize(&pointerBufferSize, m_ClientHandle));
            this->m_PointerBufferSize = pointerBufferSize;
            this->m_PointerBufferSizeCached = true;
        }
    }

public:
    Hipc2ClientProxyBaseObject() NN_NOEXCEPT
        : m_Attached(false)
        , m_PointerBufferSizeCached(false)
    {
    }

    explicit Hipc2ClientProxyBaseObject(HipcClientSessionHandle clientHandle) NN_NOEXCEPT
        : m_Attached(true)
        , m_PointerBufferSizeCached(false)
        , m_ClientHandle(clientHandle)
    {
    }

    ~Hipc2ClientProxyBaseObject() NN_NOEXCEPT
    {
        if (m_Attached)
        {
            CloseClientSessionHandleSafely(m_ClientHandle);
        }
    }

    void Attach(HipcClientSessionHandle clientHandle, uint16_t pointerBufferSize) NN_NOEXCEPT
    {
        this->m_Attached = true;
        this->m_PointerBufferSize = pointerBufferSize;
        this->m_PointerBufferSizeCached = true;
        this->m_ClientHandle = clientHandle;
    }

    void Attach(HipcClientSessionHandle clientHandle) NN_NOEXCEPT
    {
        this->m_Attached = true;
        this->m_PointerBufferSizeCached = false;
        this->m_ClientHandle = clientHandle;
    }

    HipcClientSessionHandle GetClientHandle() const NN_NOEXCEPT
    {
        return m_ClientHandle;
    }

    uint16_t GetPointerBufferSize() const NN_NOEXCEPT
    {
        CachePointerBufferSize();
        return m_PointerBufferSize;
    }

    void Detach() NN_NOEXCEPT
    {
        this->m_Attached = false;
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return this->m_Attached;
    }
};

/**
    @brief HIPC のための CmifClientMessage 実装
 */
class Hipc2ClientMessageBase
{
protected:

    NN_IMPLICIT Hipc2ClientMessageBase(Bit16 invokeTag, HipcClientSessionHandle clientHandle) NN_NOEXCEPT
        : m_InvokeTag(invokeTag)
        , m_PointerBufferSize(0)
        , m_ClientHandle(clientHandle)
    {
    }

    explicit Hipc2ClientMessageBase(Bit16 invokeTag, Hipc2ClientProxyBaseObject *pProxy, bool usePointerTransfer) NN_NOEXCEPT
        : m_InvokeTag(invokeTag)
        , m_PointerBufferSize(usePointerTransfer ? pProxy->GetPointerBufferSize() : 0)
        , m_ClientHandle(pProxy->GetClientHandle())
    {
    }

public:

    void SetPointerBufferSize(uint16_t pointerBufferSize) NN_NOEXCEPT
    {
        this->m_PointerBufferSize = pointerBufferSize;
    }

    HipcClientSessionHandle GetClientHandle() const NN_NOEXCEPT
    {
        return m_ClientHandle;
    }

    struct SpecificMethodMetaInfo
    {
        bool valid;
        size_t unfixedSizedPointerOutOffset;
        int unfixedSizedPointerOutCount;
        detail::HipcMessageDataInfo messageDataInfo;
        cmif::CmifMessageMetaInfo metaInfo;
    };

    NN_SF_DETAIL_CONSTEXPR static SpecificMethodMetaInfo MakeSpecificMethodMetaInfo(const cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        SpecificMethodMetaInfo ret = {};
        ret.valid = false;

        auto sendCount = 0;
        auto receiveCount = 0;
        auto pointerInCount = 0;
        auto pointerOutCount = 0;
        auto unfixedSizedPointerOutCount = 0;
        for (int i = 0; i < metaInfo.bufferCount; ++i)
        {
            auto attribute = metaInfo.bufferAttributes[i];
            if (attribute & cmif::BufferAttribute_HipcMapAlias)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++sendCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++receiveCount;
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++pointerInCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++pointerInCount;
                    ++sendCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutCount;
                    ++receiveCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected]");
            }
        }

        auto copyCount = 0;
        auto moveCount = 0;
        for (int i = 0; i < metaInfo.inNativeHandleCount; ++i)
        {
            auto attribute = metaInfo.inNativeHandleAttributes[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                ++copyCount;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                ++moveCount;
            }
            else
            {
                return ret;
            }
        }

        size_t inRawSize = 0;

        inRawSize += metaInfo.inRawDataSize + 16;

        inRawSize = sf::detail::AlignUp(inRawSize, sizeof(uint16_t));
        auto unfixedSizedPointerOutOffset = inRawSize;
        inRawSize += unfixedSizedPointerOutCount * sizeof(uint16_t);
        inRawSize = sf::detail::AlignUp(inRawSize, sizeof(uint32_t));

        detail::HipcMessageHeaderInfo headerInfo = {
            metaInfo.inProcessIdEnable, // bool hasPid;
            copyCount, // int copyHandleCount;
            moveCount, // int moveHandleCount;
            pointerInCount, // int pointerCount;
            sendCount, // int sendCount;
            receiveCount, // int receiveCount;
            0, // int exchangeCount;
            inRawSize, // int rawDataByteSize;
            pointerOutCount, // int receiveListCount;
            detail::HipcMessageReceiveBufferMode_Multi, // receiveBufferMode
        };
        auto messageDataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        if (!messageDataInfo.valid)
        {
            return ret;
        }

        ret.metaInfo = metaInfo;
        ret.messageDataInfo = messageDataInfo;
        ret.unfixedSizedPointerOutCount = unfixedSizedPointerOutCount;
        ret.unfixedSizedPointerOutOffset = unfixedSizedPointerOutOffset;
        ret.valid = true;
        return ret;
    } // NOLINT[impl/function_size]

private:

    nn::Result WriteHandleDataImpl(detail::HipcMessageWriter* p, const cmif::client::CmifClientMessageInfo& info, const cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        auto copyIndex = 0;
        auto moveIndex = 0;
        for (int i = 0; i < metaInfo.inNativeHandleCount; ++i)
        {
            auto attribute = metaInfo.inNativeHandleAttributes[i];
            auto&& handle = info.inNativeHandles[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                auto handleValue = hipc::detail::RegisterOsHandle(handle.GetOsHandle());
                p->SetCopyHandle(copyIndex, handleValue);
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                NN_ABORT_UNLESS(handle.IsManaged(), "[SF-HIPC-InvalidHandle:NotManaged]");
                auto handleValue = hipc::detail::RegisterOsHandle(handle.GetOsHandle());
                p->SetMoveHandle(moveIndex, handleValue);
                handle.Detach();
                ++moveIndex;
            }
            else
            {
                NN_RESULT_THROW(hipc::ResultNotSupported());
            }
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result WriteBufferDataImpl(detail::HipcMessageWriter* p, const cmif::client::CmifClientMessageInfo& info, const SpecificMethodMetaInfo& hipcMethodMetaInfo) NN_NOEXCEPT
    {
        auto unfixedSizedPointerOutSizeInfos = reinterpret_cast<uint16_t*>(reinterpret_cast<uintptr_t>(m_InRawPointer) + hipcMethodMetaInfo.unfixedSizedPointerOutOffset);
        const auto& metaInfo = hipcMethodMetaInfo.metaInfo;
        size_t restPointerBufferSize = m_PointerBufferSize;
        struct NN_ALIGNAS(4) AutoBufferInfo
        {
            uint8_t index;
            uint8_t mapIndex;
            uint8_t pointerIndex;
            uint8_t unfixedSizedPointerOutIndex;
        };
        AutoBufferInfo autoBufferInfos[sf::detail::ArgumentCountMax];
        auto autoBufferCount = 0;
        auto sendIndex = 0;
        auto receiveIndex = 0;
        auto pointerInIndex = 0;
        auto pointerOutIndex = 0;
        auto unfixedSizedPointerOutIndex = 0;
        for (int i = 0; i < metaInfo.bufferCount; ++i)
        {
            auto&& e = info.buffers[i];
            auto attribute = metaInfo.bufferAttributes[i];
            if (attribute & cmif::BufferAttribute_HipcMapAlias)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetSend(sendIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                    ++sendIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    p->SetReceive(receiveIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                    ++receiveIndex;
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                auto occupiedSize = sf::detail::AlignUp(e.size, 16);
                NN_RESULT_THROW_UNLESS(occupiedSize <= restPointerBufferSize, sf::hipc::ResultInsufficientPointerTransferBuffer());
                restPointerBufferSize -= occupiedSize;
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetPointer(pointerInIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    ++pointerInIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    p->SetReceiveList(pointerOutIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    ++pointerOutIndex;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        NN_RESULT_THROW_UNLESS(e.size <= (std::numeric_limits<uint16_t>::max)(), hipc::ResultNotSupported());
                        auto unfixedSizedPointerOutSize = static_cast<uint16_t>(e.size);
                        std::memcpy(&unfixedSizedPointerOutSizeInfos[unfixedSizedPointerOutIndex], &unfixedSizedPointerOutSize, sizeof(unfixedSizedPointerOutSize));
                        ++unfixedSizedPointerOutIndex;
                    }
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    if (e.size <= (std::numeric_limits<uint16_t>::max)())
                    {
                        auto& autoBufferInfo = autoBufferInfos[autoBufferCount];
                        autoBufferInfo.index = static_cast<decltype(autoBufferInfo.index)>(i);
                        autoBufferInfo.mapIndex = static_cast<decltype(autoBufferInfo.mapIndex)>(sendIndex);
                        autoBufferInfo.pointerIndex = static_cast<decltype(autoBufferInfo.pointerIndex)>(pointerInIndex);
                        ++autoBufferCount;
                    }
                    else
                    {
                        p->SetSend(sendIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetPointer(pointerInIndex, 0, 0);
                    }
                    ++sendIndex;
                    ++pointerInIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    if (e.size <= (std::numeric_limits<uint16_t>::max)())
                    {
                        auto& autoBufferInfo = autoBufferInfos[autoBufferCount];
                        autoBufferInfo.index = static_cast<decltype(autoBufferInfo.index)>(i);
                        autoBufferInfo.mapIndex = static_cast<decltype(autoBufferInfo.mapIndex)>(receiveIndex);
                        autoBufferInfo.pointerIndex = static_cast<decltype(autoBufferInfo.pointerIndex)>(pointerOutIndex);
                        autoBufferInfo.unfixedSizedPointerOutIndex = static_cast<decltype(autoBufferInfo.unfixedSizedPointerOutIndex)>(unfixedSizedPointerOutIndex);
                        ++autoBufferCount;
                    }
                    else
                    {
                        p->SetReceive(receiveIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetReceiveList(pointerOutIndex, 0, 0);
                    }
                    ++receiveIndex;
                    ++pointerOutIndex;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutIndex;
                    }
                    continue;
                }
            }
            NN_ABORT("[SF-CMIF-Unexpected]");
        }

        while (autoBufferCount > 0)
        {
            const auto& minAutoBufferInfo = *std::min_element(autoBufferInfos + 0, autoBufferInfos + autoBufferCount, [&info] (const AutoBufferInfo& x, const AutoBufferInfo& y)
            {
                return info.buffers[x.index].size - info.buffers[y.index].size;
            });
            auto minIndex = &minAutoBufferInfo - autoBufferInfos;
            auto minSize = info.buffers[minAutoBufferInfo.index].size;
            auto occupiedSize = sf::detail::AlignUp(minSize, 16);
            if (occupiedSize <= restPointerBufferSize)
            {
                const auto& autoBufferInfo = minAutoBufferInfo;
                restPointerBufferSize -= occupiedSize;
                auto&& e = info.buffers[autoBufferInfo.index];
                auto attribute = metaInfo.bufferAttributes[autoBufferInfo.index];
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetSend(autoBufferInfo.mapIndex, 0, 0, detail::GetMapTransferAttribute(attribute));
                    p->SetPointer(autoBufferInfo.pointerIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                }
                else
                {
                    NN_SDK_ASSERT(attribute & cmif::BufferAttribute_Out, "[SF-Internal]");
                    p->SetReceive(autoBufferInfo.mapIndex, 0, 0, detail::GetMapTransferAttribute(attribute));
                    p->SetReceiveList(autoBufferInfo.pointerIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        auto unfixedSizedPointerOutSize = static_cast<uint16_t>(e.size);
                        std::memcpy(&unfixedSizedPointerOutSizeInfos[autoBufferInfo.unfixedSizedPointerOutIndex], &unfixedSizedPointerOutSize, sizeof(unfixedSizedPointerOutSize));
                    }
                }
                --autoBufferCount;
                if (minIndex != autoBufferCount)
                {
                    autoBufferInfos[minIndex] = autoBufferInfos[autoBufferCount];
                }
            }
            else
            {
                for (int j = 0; j < autoBufferCount; ++j)
                {
                    const auto& autoBufferInfo = autoBufferInfos[j];
                    auto&& e = info.buffers[autoBufferInfo.index];
                    auto attribute = metaInfo.bufferAttributes[autoBufferInfo.index];
                    if (attribute & cmif::BufferAttribute_In)
                    {
                        p->SetSend(autoBufferInfo.mapIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetPointer(autoBufferInfo.pointerIndex, 0, 0);
                    }
                    else
                    {
                        NN_SDK_ASSERT(attribute & cmif::BufferAttribute_Out, "[SF-Internal]");
                        p->SetReceive(autoBufferInfo.mapIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetReceiveList(autoBufferInfo.pointerIndex, 0, 0);
                    }
                }
                autoBufferCount = 0;
            }
        }
        NN_RESULT_SUCCESS;
    } // NOLINT[impl/function_size]

public:

    NN_FORCEINLINE nn::Result PrepareForSyncRequest(cmif::client::CmifClientMessageInfo&& info, const SpecificMethodMetaInfo& hipcMethodMetaInfo) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(hipcMethodMetaInfo.valid, hipc::ResultNotSupported());
        const auto& metaInfo = hipcMethodMetaInfo.metaInfo;

        NN_SDK_REQUIRES(metaInfo.inObjectCount == 0); // HIPC は InObject には対応しません。
        NN_SDK_REQUIRES(metaInfo.outObjectCount <= MaxOutObjects);

        this->m_MessageBuffer = hipc::GetMessageBufferOnTls();
        auto messageBuffer = m_MessageBuffer;
        auto messageBufferSize = hipc::MessageBufferSizeOnTls;

        detail::HipcMessageWriter writer(hipcMethodMetaInfo.messageDataInfo, messageBuffer, messageBufferSize);

        this->m_InRawPointer = writer.GetRawPointer();
        writer.SetHeader(m_InvokeTag);

        if (metaInfo.inNativeHandleCount > 0)
        {
            NN_RESULT_DO(WriteHandleDataImpl(&writer, info, metaInfo));
        }
        if (metaInfo.bufferCount > 0)
        {
            NN_RESULT_DO(WriteBufferDataImpl(&writer, info, hipcMethodMetaInfo));
        }

        this->m_OutObjectCount = metaInfo.outObjectCount;
        this->m_OutNativeHandleCount = metaInfo.outNativeHandleCount;
        std::copy(metaInfo.outNativeHandleAttributes, metaInfo.outNativeHandleAttributes + metaInfo.outNativeHandleCount, this->m_OutNativeHandleAttributes);
        NN_RESULT_SUCCESS;
    } // NOLINT(readability/fn_size)

    cmif::client::CmifClientMessageInInfo GetCmifClientMessageInInfo() const NN_NOEXCEPT
    {
        cmif::client::CmifClientMessageInInfo inInfo = { nn::util::align_up(reinterpret_cast<uintptr_t>(m_InRawPointer), 16) };
        return inInfo;
    }

    nn::Result SendSyncRequest() NN_NOEXCEPT
    {
        {
            // context
            auto inlineContext = cmif::GetInlineContext();
            auto contextPointer = nn::util::align_up(reinterpret_cast<uintptr_t>(this->m_InRawPointer), 16) + 12;
            std::memcpy(reinterpret_cast<char*>(contextPointer), &inlineContext, sizeof(inlineContext));
        }

        auto messageBuffer = m_MessageBuffer;
        auto messageBufferSize = hipc::MessageBufferSizeOnTls;
        NN_RESULT_DO(hipc::SendSyncRequest(m_ClientHandle, messageBuffer, messageBufferSize));

        m_Reader.Initialize(messageBuffer);
        NN_ABORT_UNLESS(m_Reader.GetMessageByteSize() <= messageBufferSize, "[SF-Internal]");

        const detail::HipcMessageHeaderInfo& headerInfo = m_Reader.GetHeaderInfo().baseInfo;
        for (auto i = 0; i < headerInfo.pointerCount; ++i)
        {
            hipc::detail::UnregisterAddress(m_Reader.GetPointer(i).pointer);
        }

        NN_RESULT_SUCCESS;
    }

    cmif::client::CmifClientMessageOutInfo GetCmifClientMessageOutInfo() const NN_NOEXCEPT
    {
        cmif::client::CmifClientMessageOutInfo outInfo = { nn::util::align_up(m_Reader.GetRawPointer(), 16) };
        return outInfo;
    }

    void AttachOutObjects(cmif::client::CmifBaseObject* outObjects[]) const NN_NOEXCEPT
    {
        for (int i = 0; i < m_OutObjectCount; i++)
        {
            auto p = static_cast<Hipc2ClientProxyBaseObject*>(outObjects[i]);
            auto handleValue = m_Reader.GetMoveHandle(i);
            if (handleValue == detail::InvalidInternalHandleValue)
            {
                p->Detach();
                continue;
            }
            auto handle = hipc::detail::UnregisterSessionHandle(handleValue);
            p->Attach(handle, m_PointerBufferSize);
        }
    }

    void GetOutNativeHandles(NativeHandle outNativeHandles[]) const NN_NOEXCEPT
    {
        auto copyIndex = 0;
        auto moveIndex = 0;
        for (int i = 0; i < m_OutNativeHandleCount; ++i)
        {
            auto attribute = m_OutNativeHandleAttributes[i];
            auto&& handle = outNativeHandles[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                auto handleValue = m_Reader.GetCopyHandle(copyIndex);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(handleValue), true);
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                auto handleValue = m_Reader.GetMoveHandle(moveIndex + m_OutObjectCount);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(handleValue), true);
                ++moveIndex;
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected:InvalidResponse]");
            }
        }
    }

private:
    static const int MaxOutObjects = 8;

    Bit16 m_InvokeTag;
    uint16_t m_PointerBufferSize;
    const HipcClientSessionHandle m_ClientHandle;
    void* m_MessageBuffer;

    // request
    uint32_t* m_InRawPointer;

    // reply
    detail::HipcMessageReader m_Reader;

    // metaInfo
    int m_OutObjectCount;
    int m_OutNativeHandleCount;
    int m_OutNativeHandleAttributes[nn::sf::detail::ArgumentCountMax];
};

template <nn::Bit16 InvokeTag, bool UsePointerTransfer = true>
class Hipc2ClientMessage
    : public Hipc2ClientMessageBase
{
public:

    NN_IMPLICIT Hipc2ClientMessage(HipcClientSessionHandle clientHandle) NN_NOEXCEPT
        : Hipc2ClientMessageBase(InvokeTag, clientHandle)
    {
    }

    explicit Hipc2ClientMessage(Hipc2ClientProxyBaseObject *pProxy) NN_NOEXCEPT
        : Hipc2ClientMessageBase(InvokeTag, pProxy, UsePointerTransfer)
    {
    }

};

struct Hipc2ProcessorArgument
{
    nn::sf::hipc::HipcClientSessionHandle handle;
    uint16_t pointerTransferBufferSize;
};

/**
    @brief HIPC プロキシの構成の定義
    @detail
        nn::sf::cmif::client::CmifProxyFactory が要求する typename ProxyKind の実装。
        CmifProxyFactory 内部に注入され、HIPC 用プロキシが生成される。
        この構造体が満たすべき仕様は CmifProxyFactory のリファレンスを参照。
 */
template <sf::hipc::detail::MessageType MessageType>
struct Hipc2ProxyKindBase
{
    typedef nn::sf::hipc::HipcClientSessionHandle ObjectInfo;
    typedef nn::sf::hipc::client::Hipc2ClientProxyBaseObject ProxyBaseObject;
    typedef nn::sf::hipc::client::Hipc2ClientMessage<MessageType> CmifClientMessage;
    typedef Hipc2ProcessorArgument ProcessorArgument;
};

typedef Hipc2ProxyKindBase<sf::hipc::detail::MessageType_Invoke2Method> Hipc2ProxyKind;

struct Hipc2ClientCoreProcessorImpl
{

    template <typename... Args>
    static void Discard(Args...) NN_NOEXCEPT
    {
    }

    template <typename ArgumentInfo>
    struct GetAttribute
    {
        using type = typename ArgumentInfo::Attribute;
    };

    template <typename Attribute>
    class AttributeSetter
    {
    private:

        Attribute* m_Attributes;

    public:

        NN_SF_DETAIL_CONSTEXPR AttributeSetter(Attribute attributes[]) NN_NOEXCEPT
            : m_Attributes(attributes)
        {
        }

        template <typename ArgumentInfo>
        NN_SF_DETAIL_CONSTEXPR void operator()(sf::detail::TypeInvokeTag<ArgumentInfo>) NN_NOEXCEPT
        {
            m_Attributes[ArgumentInfo::Index::value] = ArgumentInfo::Attribute::value;
        }
    };

    template <int N>
    struct AttributeArray
    {
        int data[N > 0 ? N : 1];
    };

    struct HipcMethodMetaInfo
    {
        bool valid;
        bool needsPointerBufferSize;
        size_t unfixedSizedPointerOutOffset;
        int unfixedSizedPointerOutCount;
        sf::hipc:: detail::HipcMessageDataInfo messageDataInfo;
    };

    NN_SF_DETAIL_CONSTEXPR static HipcMethodMetaInfo MakeHipcMethodMetaInfo(const cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        HipcMethodMetaInfo ret = {};
        ret.valid = false;

        auto sendCount = 0;
        auto receiveCount = 0;
        auto pointerInCount = 0;
        auto pointerOutCount = 0;
        auto unfixedSizedPointerOutCount = 0;
        auto needsPointerBufferSize = false;
        for (int i = 0; i < metaInfo.bufferCount; ++i)
        {
            auto attribute = metaInfo.bufferAttributes[i];
            if (attribute & cmif::BufferAttribute_HipcMapAlias)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++sendCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++receiveCount;
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                needsPointerBufferSize = true;
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++pointerInCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                needsPointerBufferSize = true;
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++pointerInCount;
                    ++sendCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutCount;
                    ++receiveCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
                else
                {
                    return ret;
                }
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected]");
            }
        }

        auto copyCount = 0;
        auto moveCount = 0;
        for (int i = 0; i < metaInfo.inNativeHandleCount; ++i)
        {
            auto attribute = metaInfo.inNativeHandleAttributes[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                ++copyCount;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                ++moveCount;
            }
            else
            {
                return ret;
            }
        }

        size_t inRawSize = 0;

        inRawSize += metaInfo.inRawDataSize + 16;

        inRawSize = sf::detail::AlignUp(inRawSize, sizeof(uint16_t));
        auto unfixedSizedPointerOutOffset = inRawSize;
        inRawSize += unfixedSizedPointerOutCount * sizeof(uint16_t);
        inRawSize = sf::detail::AlignUp(inRawSize, sizeof(uint32_t));

        detail::HipcMessageHeaderInfo headerInfo = {
            metaInfo.inProcessIdEnable, // bool hasPid;
            copyCount, // int copyHandleCount;
            moveCount, // int moveHandleCount;
            pointerInCount, // int pointerCount;
            sendCount, // int sendCount;
            receiveCount, // int receiveCount;
            0, // int exchangeCount;
            inRawSize, // int rawDataByteSize;
            pointerOutCount, // int receiveListCount;
            detail::HipcMessageReceiveBufferMode_Multi, // receiveBufferMode
        };
        auto messageDataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        if (!messageDataInfo.valid)
        {
            return ret;
        }

        ret.messageDataInfo = messageDataInfo;
        ret.unfixedSizedPointerOutCount = unfixedSizedPointerOutCount;
        ret.unfixedSizedPointerOutOffset = unfixedSizedPointerOutOffset;
        ret.needsPointerBufferSize = needsPointerBufferSize;
        ret.valid = true;
        return ret;
    } // NOLINT[impl/function_size]

    static Result WriteHandleDataImpl(detail::HipcMessageWriter* p, int inHandleCount, sf::NativeHandle inHandles[], const int* inHandleAttributes) NN_NOEXCEPT
    {
        auto copyIndex = 0;
        auto moveIndex = 0;
        for (int i = 0; i < inHandleCount; ++i)
        {
            auto attribute = inHandleAttributes[i];
            auto&& handle = inHandles[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                auto handleValue = hipc::detail::RegisterOsHandle(handle.GetOsHandle());
                p->SetCopyHandle(copyIndex, handleValue);
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                NN_ABORT_UNLESS(handle.IsManaged(), "[SF-HIPC-InvalidHandle:NotManaged]");
                auto handleValue = hipc::detail::RegisterOsHandle(handle.GetOsHandle());
                p->SetMoveHandle(moveIndex, handleValue);
                handle.Detach();
                ++moveIndex;
            }
            else
            {
                NN_RESULT_THROW(hipc::ResultNotSupported());
            }
        }
        NN_RESULT_SUCCESS;
    }

    static Result WriteBufferDataImpl(detail::HipcMessageWriter* p, int bufferCount, const cmif::PointerAndSize* buffers, const int* bufferAttributes, size_t pointerBufferSize, size_t unfixedSizedPointerOutOffset) NN_NOEXCEPT
    {
        auto unfixedSizedPointerOutSizeInfos = reinterpret_cast<uint16_t*>(reinterpret_cast<uintptr_t>(p->GetRawPointer()) + unfixedSizedPointerOutOffset);
        size_t restPointerBufferSize = pointerBufferSize;
        struct NN_ALIGNAS(4) AutoBufferInfo
        {
            uint8_t index;
            uint8_t mapIndex;
            uint8_t pointerIndex;
            uint8_t unfixedSizedPointerOutIndex;
        };
        AutoBufferInfo autoBufferInfos[sf::detail::ArgumentCountMax];
        auto autoBufferCount = 0;
        auto sendIndex = 0;
        auto receiveIndex = 0;
        auto pointerInIndex = 0;
        auto pointerOutIndex = 0;
        auto unfixedSizedPointerOutIndex = 0;
        for (int i = 0; i < bufferCount; ++i)
        {
            auto&& e = buffers[i];
            auto attribute = bufferAttributes[i];
            if (attribute & cmif::BufferAttribute_HipcMapAlias)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetSend(sendIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                    ++sendIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    p->SetReceive(receiveIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                    ++receiveIndex;
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                auto occupiedSize = sf::detail::AlignUp(e.size, 16);
                NN_RESULT_THROW_UNLESS(occupiedSize <= restPointerBufferSize, sf::hipc::ResultInsufficientPointerTransferBuffer());
                restPointerBufferSize -= occupiedSize;
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetPointer(pointerInIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    ++pointerInIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    p->SetReceiveList(pointerOutIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    ++pointerOutIndex;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        NN_RESULT_THROW_UNLESS(e.size <= (std::numeric_limits<uint16_t>::max)(), hipc::ResultNotSupported());
                        auto unfixedSizedPointerOutSize = static_cast<uint16_t>(e.size);
                        std::memcpy(&unfixedSizedPointerOutSizeInfos[unfixedSizedPointerOutIndex], &unfixedSizedPointerOutSize, sizeof(unfixedSizedPointerOutSize));
                        ++unfixedSizedPointerOutIndex;
                    }
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    if (e.size <= (std::numeric_limits<uint16_t>::max)())
                    {
                        auto& autoBufferInfo = autoBufferInfos[autoBufferCount];
                        autoBufferInfo.index = static_cast<decltype(autoBufferInfo.index)>(i);
                        autoBufferInfo.mapIndex = static_cast<decltype(autoBufferInfo.mapIndex)>(sendIndex);
                        autoBufferInfo.pointerIndex = static_cast<decltype(autoBufferInfo.pointerIndex)>(pointerInIndex);
                        ++autoBufferCount;
                    }
                    else
                    {
                        p->SetSend(sendIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetPointer(pointerInIndex, 0, 0);
                    }
                    ++sendIndex;
                    ++pointerInIndex;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    if (e.size <= (std::numeric_limits<uint16_t>::max)())
                    {
                        auto& autoBufferInfo = autoBufferInfos[autoBufferCount];
                        autoBufferInfo.index = static_cast<decltype(autoBufferInfo.index)>(i);
                        autoBufferInfo.mapIndex = static_cast<decltype(autoBufferInfo.mapIndex)>(receiveIndex);
                        autoBufferInfo.pointerIndex = static_cast<decltype(autoBufferInfo.pointerIndex)>(pointerOutIndex);
                        autoBufferInfo.unfixedSizedPointerOutIndex = static_cast<decltype(autoBufferInfo.unfixedSizedPointerOutIndex)>(unfixedSizedPointerOutIndex);
                        ++autoBufferCount;
                    }
                    else
                    {
                        p->SetReceive(receiveIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetReceiveList(pointerOutIndex, 0, 0);
                    }
                    ++receiveIndex;
                    ++pointerOutIndex;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutIndex;
                    }
                    continue;
                }
            }
            NN_ABORT("[SF-CMIF-Unexpected]");
        }

        while (autoBufferCount > 0)
        {
            const auto& minAutoBufferInfo = *std::min_element(autoBufferInfos + 0, autoBufferInfos + autoBufferCount, [&buffers] (const AutoBufferInfo& x, const AutoBufferInfo& y)
            {
                return buffers[x.index].size - buffers[y.index].size;
            });
            auto minIndex = &minAutoBufferInfo - autoBufferInfos;
            auto minSize = buffers[minAutoBufferInfo.index].size;
            auto occupiedSize = sf::detail::AlignUp(minSize, 16);
            if (occupiedSize <= restPointerBufferSize)
            {
                const auto& autoBufferInfo = minAutoBufferInfo;
                restPointerBufferSize -= occupiedSize;
                auto&& e = buffers[autoBufferInfo.index];
                auto attribute = bufferAttributes[autoBufferInfo.index];
                if (attribute & cmif::BufferAttribute_In)
                {
                    p->SetSend(autoBufferInfo.mapIndex, 0, 0, detail::GetMapTransferAttribute(attribute));
                    p->SetPointer(autoBufferInfo.pointerIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                }
                else
                {
                    NN_SDK_ASSERT(attribute & cmif::BufferAttribute_Out, "[SF-Internal]");
                    p->SetReceive(autoBufferInfo.mapIndex, 0, 0, detail::GetMapTransferAttribute(attribute));
                    p->SetReceiveList(autoBufferInfo.pointerIndex, hipc::detail::RegisterAddress(e.pointer), e.size);
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        auto unfixedSizedPointerOutSize = static_cast<uint16_t>(e.size);
                        std::memcpy(&unfixedSizedPointerOutSizeInfos[autoBufferInfo.unfixedSizedPointerOutIndex], &unfixedSizedPointerOutSize, sizeof(unfixedSizedPointerOutSize));
                    }
                }
                --autoBufferCount;
                if (minIndex != autoBufferCount)
                {
                    autoBufferInfos[minIndex] = autoBufferInfos[autoBufferCount];
                }
            }
            else
            {
                for (int j = 0; j < autoBufferCount; ++j)
                {
                    const auto& autoBufferInfo = autoBufferInfos[j];
                    auto&& e = buffers[autoBufferInfo.index];
                    auto attribute = bufferAttributes[autoBufferInfo.index];
                    if (attribute & cmif::BufferAttribute_In)
                    {
                        p->SetSend(autoBufferInfo.mapIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetPointer(autoBufferInfo.pointerIndex, 0, 0);
                    }
                    else
                    {
                        NN_SDK_ASSERT(attribute & cmif::BufferAttribute_Out, "[SF-Internal]");
                        p->SetReceive(autoBufferInfo.mapIndex, hipc::detail::RegisterAddress(e.pointer), e.size, detail::GetMapTransferAttribute(attribute));
                        p->SetReceiveList(autoBufferInfo.pointerIndex, 0, 0);
                    }
                }
                autoBufferCount = 0;
            }
        }
        NN_RESULT_SUCCESS;
    } // NOLINT[impl/function_size]

    static void GetOutNativeHandles(detail::HipcMessageReader* pReader, int outHandleCount, int outObjectCount, NativeHandle outNativeHandles[], const int outHandleAttributes[]) NN_NOEXCEPT
    {
        auto copyIndex = 0;
        auto moveIndex = 0;
        for (int i = 0; i < outHandleCount; ++i)
        {
            auto attribute = outHandleAttributes[i];
            auto&& handle = outNativeHandles[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                auto handleValue = pReader->GetCopyHandle(copyIndex);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(handleValue), true);
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                auto handleValue = pReader->GetMoveHandle(moveIndex + outObjectCount);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(handleValue), true);
                ++moveIndex;
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected:InvalidResponse]");
            }
        }
    }

    static void AttachOutObjects(detail::HipcMessageReader* pReader, int outObjectCount, cmif::client::CmifBaseObject* outObjects[]) NN_NOEXCEPT
    {
        for (int i = 0; i < outObjectCount; i++)
        {
            auto p = static_cast<Hipc2ClientProxyBaseObject*>(outObjects[i]);
            auto handleValue = pReader->GetMoveHandle(i);
            if (handleValue == detail::InvalidInternalHandleValue)
            {
                p->Detach();
                continue;
            }
            auto handle = hipc::detail::UnregisterSessionHandle(handleValue);
            p->Attach(handle);
        }
    }

};

/*
    Hipc2ClientCoreProcessor: HIPC プロキシの ClientCoreProcessor 実装

    ClientMessage なしで最小限のコードとしている。
    汎用実装と ClientMessage の実装とを手動でインラインしたものとほぼ同等である。
    特に、同じ名前の関数に関しては、同じ実装を流用している。
*/
template <sf::hipc::detail::MessageType MessageType, typename CoreMethodInfo, typename ArgumentInfos>
struct Hipc2ClientCoreProcessor;

template <sf::hipc::detail::MessageType MessageType, typename CoreMethodInfo, typename... ArgumentInfos_>
struct Hipc2ClientCoreProcessor<MessageType, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
    : public Hipc2ClientCoreProcessorImpl
{
private:

    using ProxyBaseObject = Hipc2ClientProxyBaseObject;
    using ProcessorArgument = Hipc2ProcessorArgument;

    NN_SF_DETAIL_CONSTEXPR static cmif::CmifMessageMetaInfo MakeCmifMessageMetaInfo() NN_NOEXCEPT
    {
        cmif::CmifMessageMetaInfo ret = {};
        ret.inProcessIdEnable = CoreMethodInfo::InProcessIdEnable::value;
        ret.inRawDataSize = sizeof(cmif::CmifInHeader) + CoreMethodInfo::InRawSize::value;
        ret.outRawDataSize = sizeof(cmif::CmifOutHeader) + CoreMethodInfo::OutRawSize::value;
        ret.bufferCount = CoreMethodInfo::BufferCount::value;
        ret.inObjectCount = CoreMethodInfo::InObjectCount::value;
        ret.outObjectCount = CoreMethodInfo::OutObjectCount::value;
        ret.inNativeHandleCount = CoreMethodInfo::InHandleCount::value;
        ret.outNativeHandleCount = CoreMethodInfo::OutHandleCount::value;
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::Buffers>(AttributeSetter<int>(ret.bufferAttributes));
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::InHandles>(AttributeSetter<int>(ret.inNativeHandleAttributes));
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::OutHandles>(AttributeSetter<int>(ret.outNativeHandleAttributes));
        return ret;
    }

    NN_SF_DETAIL_CONSTEXPR static AttributeArray<CoreMethodInfo::BufferCount::value> MakeBufferAttributes() NN_NOEXCEPT
    {
        AttributeArray<CoreMethodInfo::BufferCount::value> ret = {};
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::Buffers>(AttributeSetter<int>(ret.data));
        return ret;
    }

    NN_SF_DETAIL_CONSTEXPR static AttributeArray<CoreMethodInfo::InHandleCount::value> MakeInHandleAttributes() NN_NOEXCEPT
    {
        AttributeArray<CoreMethodInfo::InHandleCount::value> ret = {};
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::InHandles>(AttributeSetter<int>(ret.data));
        return ret;
    }

    NN_SF_DETAIL_CONSTEXPR static AttributeArray<CoreMethodInfo::OutHandleCount::value> MakeOutHandleAttributes() NN_NOEXCEPT
    {
        AttributeArray<CoreMethodInfo::OutHandleCount::value> ret = {};
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::OutHandles>(AttributeSetter<int>(ret.data));
        return ret;
    }

    NN_FORCEINLINE static Result ProcessImpl(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        NN_SF_DETAIL_CONSTEXPR auto metaInfo = MakeCmifMessageMetaInfo();
        NN_SF_DETAIL_CONSTEXPR auto hipcMethodMetaInfo = MakeHipcMethodMetaInfo(metaInfo);

        NN_RESULT_THROW_UNLESS(hipcMethodMetaInfo.valid, hipc::ResultNotSupported());
        NN_SDK_REQUIRES(metaInfo.inObjectCount == 0); // HIPC は InObject には対応しません。

        uint16_t pointerBufferSize = 0;
        if (NN_STATIC_CONDITION(hipcMethodMetaInfo.needsPointerBufferSize))
        {
            pointerBufferSize = p->GetPointerBufferSize();
        }
        auto handle = p->GetClientHandle();
        return ProcessImpl({handle, pointerBufferSize}, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects, 0, 0);
    }

public:

    NN_FORCEINLINE static Result ProcessImpl(const ProcessorArgument& pa, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[], size_t inRawPadding, size_t outRawPadding) NN_NOEXCEPT
    {
        NN_SF_DETAIL_CONSTEXPR auto metaInfo = MakeCmifMessageMetaInfo();
        NN_SF_DETAIL_CONSTEXPR auto hipcMethodMetaInfo = MakeHipcMethodMetaInfo(metaInfo);

        NN_RESULT_THROW_UNLESS(hipcMethodMetaInfo.valid, hipc::ResultNotSupported());
        NN_SDK_REQUIRES(metaInfo.inObjectCount == 0); // HIPC は InObject には対応しません。

        auto handle = pa.handle;
        auto pointerBufferSize = pa.pointerTransferBufferSize;
        auto messageBuffer = hipc::GetMessageBufferOnTls();
        auto messageBufferSize = hipc::MessageBufferSizeOnTls;

        std::array<NativeHandle, CoreMethodInfo::InHandleCount::value> inHandles;
        {
            NN_SF_DETAIL_CONSTEXPR auto messageDataInfo = hipcMethodMetaInfo.messageDataInfo;
            sf::hipc::detail::HipcMessageWriter writer(messageDataInfo, messageBuffer, messageBufferSize);
            writer.SetHeader(MessageType);

            // in buffers
            if (NN_STATIC_CONDITION(CoreMethodInfo::BufferCount::value > 0))
            {
                std::array<cmif::PointerAndSize, CoreMethodInfo::BufferCount::value> buffers;
                Discard(args.template PrepareBuffer<ArgumentInfos_>(buffers.data())...);
                NN_RESULT_DO(WriteBufferDataImpl(&writer, CoreMethodInfo::BufferCount::value, buffers.data(), MakeBufferAttributes().data, pointerBufferSize, hipcMethodMetaInfo.unfixedSizedPointerOutOffset));
            }

            // in handles
            if (NN_STATIC_CONDITION(CoreMethodInfo::InHandleCount::value > 0))
            {
                Discard(args.template PrepareInHandle<ArgumentInfos_>(inHandles.data())...);
                NN_RESULT_DO(WriteHandleDataImpl(&writer, CoreMethodInfo::InHandleCount::value, inHandles.data(), MakeInHandleAttributes().data));
            }

            // in raw
            auto inRawPointer = writer.GetRawPointer();
            auto inRawDataPointer = nn::util::align_up(reinterpret_cast<uintptr_t>(inRawPointer), 16) + inRawPadding;
            auto inHeader = cmif::CmifInHeader::Create(methodId);
            std::memcpy(reinterpret_cast<char*>(inRawDataPointer), &inHeader, sizeof(inHeader));
            if (NN_STATIC_CONDITION(CoreMethodInfo::InRawSize::value > 0))
            {
                auto pInRaw = reinterpret_cast<char*>(inRawDataPointer) + sizeof(inHeader);
                std::memset(pInRaw, 0, CoreMethodInfo::InRawSize::value - inRawPadding);
                Discard(args.template PrepareInRaw<ArgumentInfos_>(pInRaw)...);
            }

            // context
            auto inlineContext = cmif::GetInlineContext();
            auto contextPointer = nn::util::align_up(reinterpret_cast<uintptr_t>(inRawPointer), 16) + 12;
            std::memcpy(reinterpret_cast<char*>(contextPointer), &inlineContext, sizeof(inlineContext));
        }

        NN_RESULT_DO(hipc::SendSyncRequest(handle, messageBuffer, messageBufferSize));

        detail::HipcMessageReader reader;
        reader.Initialize(messageBuffer);
        NN_ABORT_UNLESS(reader.GetMessageByteSize() <= messageBufferSize, "[SF-Internal]");

        {
            const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;
            for (auto i = 0; i < headerInfo.pointerCount; ++i)
            {
                hipc::detail::UnregisterAddress(reader.GetPointer(i).pointer);
            }
        }

        auto outRawDataPointer = util::align_up(reader.GetRawPointer(), 16) + outRawPadding;
        {
            cmif::CmifOutHeader outHeader;
            std::memcpy(&outHeader, reinterpret_cast<void*>(outRawDataPointer), sizeof(outHeader));
            NN_SF_RESULT_THROW_UNLESS(outHeader.Check(), cmif::ResultInvalidCmifOutHeader());
            NN_SF_RESULT_DO(outHeader.result);
        }

        // out raw
        if (NN_STATIC_CONDITION(CoreMethodInfo::OutRawSize::value > 0))
        {
            auto pOutRaw = reinterpret_cast<const char*>(outRawDataPointer) + sizeof(cmif::CmifOutHeader);
            NN_UNUSED(pOutRaw);
            Discard(args.template GetOutRaw<ArgumentInfos_>(pOutRaw)...);
        }

        // out handles
        if (NN_STATIC_CONDITION(CoreMethodInfo::OutHandleCount::value > 0))
        {
            std::array<sf::NativeHandle, CoreMethodInfo::OutHandleCount::value> outHandles;
            GetOutNativeHandles(&reader, CoreMethodInfo::OutHandleCount::value, CoreMethodInfo::OutObjectCount::value, outHandles.data(), MakeOutHandleAttributes().data);
            Discard(args.template GetOutHandle<ArgumentInfos_>(outHandles.data())...);
        }

        // out objects
        if (NN_STATIC_CONDITION(CoreMethodInfo::OutObjectCount::value > 0))
        {
            std::array<sf::cmif::client::CmifBaseObject*, CoreMethodInfo::OutObjectCount::value> pBaseObjects;
            for (auto i = 0u; i < CoreMethodInfo::OutObjectCount::value; ++i)
            {
                pBaseObjects[i] = sf::detail::CmifProxyInfoAccessor::GetCmifBaseObject(pOutObjects[i].Get());
            }
            AttachOutObjects(&reader, CoreMethodInfo::OutObjectCount::value, pBaseObjects.data());
            for (auto i = 0u; i < CoreMethodInfo::OutObjectCount::value; ++i)
            {
                if (!static_cast<ProxyBaseObject*>(pBaseObjects[i])->IsValid())
                {
                    pOutObjects[i].Reset();
                }
            }
            Discard(args.template GetOutObject<ArgumentInfos_>(pOutObjects)...);
        }

        NN_RESULT_SUCCESS;
    } // NOLINT[impl/function_size]

    NN_NOINLINE static Result Process(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, nullptr);
    }

    NN_NOINLINE static Result ProcessWithOutObjects(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects);
    }

};

}}}} // ~ nn::sf::hipc::client

namespace nn { namespace sf { namespace cmif { namespace client {

/*
    HIPC プロキシに対応する ClientCoreProcessor の特殊化

    実装は Hipc2ClientCoreProcessor に移譲する。
*/
template <sf::hipc::detail::MessageType MessageType, typename CoreMethodInfo, typename... ArgumentInfos_>
struct ClientCoreProcessor<sf::hipc::client::Hipc2ProxyKindBase<MessageType>, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
    : public sf::hipc::client::Hipc2ClientCoreProcessor<MessageType, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
};

}}}}
