﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/cmif/sf_CmifDomainCommon.h>
#include <nn/sf/cmif/sf_ICmifDomain.h>
#include <nn/sf/cmif/server/sf_CmifServerDomain.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/cmif/server/detail/sf_CmifProcessFunctionTable.h>
#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/cmif/detail/sf_CmifDomainMessageFormat.h>
#include <nn/sf/cmif/sf_CmifResult.h>

namespace nn { namespace sf { namespace cmif { namespace server {

class CmifDomainServerObject
    : public ICmifDomain
    , public CmifServerDomain
{
private:

    virtual CmifServerDomain* GetDomain() NN_NOEXCEPT = 0;

    class CmifDomainServerMessage
        : public CmifServerMessage
    {
    private:

        cmif::server::CmifServerMessage* m_pBaseMessage;
        CmifServerDomain* m_pDomain;
        CmifDomainObjectId* m_InObjectIds;
        CmifDomainObjectId m_OutObjectIds[8];
        void* m_OutObjectIdPtrInHeader;
        int m_InObjectCount;
        int m_OutObjectCount;
        size_t m_OutRawSize;

    public:

        CmifDomainServerMessage(cmif::server::CmifServerMessage* pBaseMessage, CmifServerDomain* pDomain, CmifDomainObjectId inObjectIds[], int inObjectCount) NN_NOEXCEPT
            : m_pBaseMessage(pBaseMessage)
            , m_pDomain(pDomain)
            , m_InObjectIds(inObjectIds)
            , m_InObjectCount(inObjectCount)
        {
        }

        virtual nn::Result PrepareForProcess(const CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_THROW_UNLESS(metaInfo.outObjectCount <= static_cast<int>(sizeof(m_OutObjectIds) / sizeof(*m_OutObjectIds)), sf::cmif::ResultInvalidOutObjectCount()); // out オブジェクト過多
            NN_RESULT_THROW_UNLESS(metaInfo.inObjectCount == m_InObjectCount, sf::cmif::ResultInvalidInObjectCount()); // in object count の不一致
            NN_RESULT_DO(m_pBaseMessage->PrepareForProcess(cmif::detail::MakeBaseMetaInfo(metaInfo)));
            NN_RESULT_DO(m_pDomain->ReserveEntry(m_OutObjectIds, metaInfo.outObjectCount));
            this->m_OutObjectCount = metaInfo.outObjectCount;
            this->m_OutRawSize = metaInfo.outRawDataSize;
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result OverwriteClientProcessId(nn::Bit64 *pInOutPid) const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->OverwriteClientProcessId(pInOutPid);
        }

        virtual nn::Result GetBuffers(PointerAndSize* buffers) const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->GetBuffers(buffers);
        }

        virtual nn::Result GetInNativeHandles(nn::sf::NativeHandle handles[]) const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->GetInNativeHandles(handles);
        }

        virtual nn::Result GetInObjects(CmifServerObjectInfo* inObjects) const NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < m_InObjectCount; ++i)
            {
                inObjects[i] = m_pDomain->GetObject(m_InObjectIds[i]);
            }
            NN_RESULT_SUCCESS;
        }

        virtual void BeginPreparingForReply(PointerAndSize* pOutRawData) NN_NOEXCEPT NN_OVERRIDE
        {
            PointerAndSize rawData;
            m_pBaseMessage->BeginPreparingForReply(&rawData);
            cmif::detail::CmifDomainMessageOutHeader outHeader = {};
            NN_SDK_ASSERT(sizeof(outHeader) + m_OutRawSize + sizeof(*m_OutObjectIds) * m_OutObjectCount <= rawData.size, "[SF-Internal]");
            auto pCurrent = reinterpret_cast<char*>(rawData.pointer);
            outHeader.outObjectCount = m_OutObjectCount;
            std::memcpy(pCurrent, &outHeader, sizeof(outHeader));
            pCurrent += sizeof(outHeader);
            pOutRawData->pointer = reinterpret_cast<decltype(pOutRawData->pointer)>(pCurrent);
            pOutRawData->size = m_OutRawSize;
            pCurrent += m_OutRawSize;
            m_OutObjectIdPtrInHeader = pCurrent;
        }

        virtual void BeginPreparingForErrorReply(PointerAndSize* pOutRawData, size_t outRawSize) NN_NOEXCEPT NN_OVERRIDE
        {
            PointerAndSize rawData;
            cmif::detail::CmifDomainMessageOutHeader outHeader = {};
            m_pBaseMessage->BeginPreparingForErrorReply(&rawData, sizeof(outHeader) + outRawSize);
            NN_SDK_ASSERT(sizeof(outHeader) + outRawSize <= rawData.size, "[SF-Internal]");
            auto pCurrent = reinterpret_cast<char*>(rawData.pointer);
            outHeader.outObjectCount = 0;
            std::memcpy(pCurrent, &outHeader, sizeof(outHeader));
            pCurrent += sizeof(outHeader);
            pOutRawData->pointer = reinterpret_cast<decltype(pOutRawData->pointer)>(pCurrent);
            pOutRawData->size = outRawSize;
            m_pDomain->UnReserveEntry(m_OutObjectIds, m_OutObjectCount);
        }

        virtual void SetBuffers(PointerAndSize buffers[]) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->SetBuffers(buffers);
        }

        virtual void SetOutObjects(CmifServerObjectInfo* outObjects) NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < m_OutObjectCount; ++i)
            {
                if (!outObjects[i])
                {
                    m_pDomain->UnReserveEntry(m_OutObjectIds + i, 1);
                    m_OutObjectIds[i] = InvalidCmifDomainObjectId;
                    continue;
                }
                m_pDomain->RegisterObject(m_OutObjectIds[i], std::move(outObjects[i]));
            }
            std::memcpy(m_OutObjectIdPtrInHeader, m_OutObjectIds, sizeof(*m_OutObjectIds) * m_OutObjectCount);
        }

        virtual void SetOutNativeHandles(nn::sf::NativeHandle outNativeHandles[]) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->SetOutNativeHandles(outNativeHandles);
        }

        virtual void EndPreparingForReply() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseMessage->EndPreparingForReply();
        }

    };

    static Result ProcessMessageImpl(CmifServerMessage* pBaseMessage, CmifServerDomain* pDomain, cmif::PointerAndSize inRawData) NN_NOEXCEPT
    {
        cmif::detail::CmifDomainMessageInHeader inHeader;
        NN_RESULT_THROW_UNLESS(sizeof(inHeader) <= inRawData.size, sf::cmif::ResultInvalidCmifHeaderSize()); // in raw のサイズが足りない
        std::memcpy(&inHeader, reinterpret_cast<void*>(inRawData.pointer), sizeof(inHeader));
        cmif::PointerAndSize inDomainRawData;
        inDomainRawData.pointer = inRawData.pointer + sizeof(inHeader);
        inDomainRawData.size = inRawData.size - sizeof(inHeader);
        switch (inHeader.requestKind)
        {
            case cmif::detail::GroupedRequestKind_InvokeMethod:
            {
                auto target = pDomain->GetObject(inHeader.targetObjectId);
                NN_RESULT_THROW_UNLESS(static_cast<bool>(target), sf::cmif::ResultTargetObjectNotFound()); // オブジェクトが見つからなかった
                NN_RESULT_THROW_UNLESS(inHeader.inRawSize + inHeader.inObjectCount * sizeof(CmifDomainObjectId) <= inDomainRawData.size, sf::cmif::ResultInvalidCmifHeaderSize()); // in raw のサイズが足りない
                cmif::PointerAndSize cmifInRawData;
                cmifInRawData.pointer = inDomainRawData.pointer;
                cmifInRawData.size = inHeader.inRawSize;
                CmifDomainObjectId inObjectIds[8];
                NN_RESULT_THROW_UNLESS(inHeader.inObjectCount <= sizeof(inObjectIds) / sizeof(*inObjectIds), sf::cmif::ResultInvalidInObjectCount()); // in オブジェクト数過多
                std::memcpy(inObjectIds, reinterpret_cast<void*>(inDomainRawData.pointer + inHeader.inRawSize), sizeof(*inObjectIds) * inHeader.inObjectCount);
                CmifDomainServerMessage serverMessage(pBaseMessage, pDomain, inObjectIds, inHeader.inObjectCount);
                return target.ProcessServerMessage(&serverMessage, cmifInRawData);
            }
            case cmif::detail::GroupedRequestKind_Release:
            {
                pDomain->UnregisterObject(inHeader.targetObjectId);
                NN_RESULT_SUCCESS;
            }
            default:
            {
                NN_RESULT_THROW(sf::cmif::ResultInvalidCmifInHeader()); // 不正な CmifDomainMessageInHeader の requestKind
            }
        }
    }

public:

    static Result ProcessMessage(IServiceObject* p, CmifServerMessage* pBaseMessage, const PointerAndSize& rawData) NN_NOEXCEPT
    {
        return ProcessMessageImpl(pBaseMessage, static_cast<CmifDomainServerObject*>(p)->GetDomain(), rawData);
    }

protected:

    ~CmifDomainServerObject() NN_NOEXCEPT
    {
    }

};

namespace detail {

    template <typename Tag>
    struct CmifProcessFunctionTableGetter<CmifDomainServerObject, Tag>
    {
        static const CmifProcessFunctionTable s_Table;
    };

    template <typename Tag>
    const CmifProcessFunctionTable CmifProcessFunctionTableGetter<CmifDomainServerObject, Tag>::s_Table =
    {
        CmifDomainServerObject::ProcessMessage,
    };

}

}}}}
