﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/cmif/client/sf_CmifClientMessage.h>
#include <nn/sf/cmif/sf_CmifDomainCommon.h>
#include <nn/sf/cmif/detail/sf_CmifDomainMessageFormat.h>
#include <utility>
#include <nn/util/util_Exchange.h>
#include <cstring>
#include <array>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/detail/sf_CommonUtil.h>

namespace nn { namespace sf { namespace cmif { namespace client {

template <typename BaseProxyKind>
struct CmifDomainProcessorArgument
{
    typename BaseProxyKind::ProcessorArgument baseArgument;
    CmifDomainObjectId id;
};

template <typename BaseProxyKind>
class CmifDomainClientMessage
{
private:

    typedef typename BaseProxyKind::CmifClientMessage BaseClientMessage;
    typedef typename BaseProxyKind::ObjectInfo BaseObjectInfo;
    typedef typename BaseProxyKind::ProxyBaseObject BaseProxyBaseObject;
    typedef CmifDomainProcessorArgument<BaseProxyKind> ProcessorArgument;

public:

    struct ObjectInfo
    {
        BaseObjectInfo baseInfo;
        CmifDomainObjectId id;
    };

    class ProxyBaseObject
        : public CmifBaseObject
    {
    private:

        bool m_Attached;
        ObjectInfo m_ObjectInfo;

        NN_SF_DETAIL_CONSTEXPR static CmifMessageMetaInfo MakeMetaInfo() NN_NOEXCEPT
        {
            CmifMessageMetaInfo ret = {};
            ret.inRawDataSize = sizeof(cmif::detail::CmifDomainMessageInHeader);
            return ret;
        }

    public:

        ProxyBaseObject() NN_NOEXCEPT
            : m_Attached(false)
        {
        }

        explicit ProxyBaseObject(const ObjectInfo& info) NN_NOEXCEPT
            : m_Attached(true)
            , m_ObjectInfo(info)
        {
        }

        ~ProxyBaseObject() NN_NOEXCEPT
        {
            if (m_Attached)
            {
                cmif::detail::CmifDomainMessageInHeader inHeader = {};
                inHeader.requestKind = cmif::detail::GroupedRequestKind_Release;
                inHeader.targetObjectId = m_ObjectInfo.id;
                CmifClientMessageInfo info = {};
                BaseClientMessage baseMessage(m_ObjectInfo.baseInfo);
                NN_SF_DETAIL_CONSTEXPR auto baseMataInfo = BaseClientMessage::MakeSpecificMethodMetaInfo(MakeMetaInfo());
                NN_ABORT_UNLESS_RESULT_SUCCESS(baseMessage.PrepareForSyncRequest(std::move(info), baseMataInfo));
                auto inInfo = baseMessage.GetCmifClientMessageInInfo();
                std::memcpy(reinterpret_cast<void*>(inInfo.inRawDataPointer), &inHeader, sizeof(inHeader));
                NN_ABORT_UNLESS_RESULT_SUCCESS(baseMessage.SendSyncRequest());
            }
        }

        void Attach(const ObjectInfo& info) NN_NOEXCEPT
        {
            this->m_Attached = true;
            this->m_ObjectInfo = info;
        }

        const ObjectInfo& GetObjectInfo() const NN_NOEXCEPT
        {
            return m_ObjectInfo;
        }

        void Attach(const ProcessorArgument& pa) NN_NOEXCEPT
        {
            this->m_Attached = true;
            this->m_ObjectInfo.baseInfo = pa.baseArgument;
            this->m_ObjectInfo.id = pa.id;
        }

        ProcessorArgument GetProcessorArgument() const NN_NOEXCEPT
        {
            return ProcessorArgument{m_ObjectInfo.baseInfo, m_ObjectInfo.id};
        }

        void Detach() NN_NOEXCEPT
        {
            this->m_Attached = false;
        }

        bool IsValid() const NN_NOEXCEPT
        {
            return this->m_Attached;
        }
    };

private:

    BaseClientMessage m_BaseMessage;
    ObjectInfo m_ObjectInfo;

    CmifClientMessageInInfo m_InInfo;
    int m_OutObjectCount;
    size_t m_OutRawDataSize;

    CmifClientMessageOutInfo m_OutInfo;
    cmif::detail::CmifDomainMessageOutHeader m_DomainOutHeader;

public:

    CmifDomainClientMessage(const BaseObjectInfo& baseInfo, CmifDomainObjectId id) NN_NOEXCEPT
        : m_BaseMessage(baseInfo)
    {
        this->m_ObjectInfo.baseInfo = baseInfo;
        this->m_ObjectInfo.id = id;
    }

    explicit CmifDomainClientMessage(ProxyBaseObject* pBaseObject) NN_NOEXCEPT
        : m_BaseMessage(pBaseObject->GetObjectInfo().baseInfo)
        , m_ObjectInfo(pBaseObject->GetObjectInfo())
    {
    }

    struct SpecificMethodMetaInfo
    {
        typename BaseClientMessage::SpecificMethodMetaInfo baseInfo;
        size_t inRawDataSize;
        int inObjectCount;
        int outObjectCount;
        size_t outRawDataSize;
    };

    NN_SF_DETAIL_CONSTEXPR static SpecificMethodMetaInfo MakeSpecificMethodMetaInfo(const CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        SpecificMethodMetaInfo ret = {};
        auto baseMetaInfo = cmif::detail::MakeBaseMetaInfo(metaInfo);
        ret.baseInfo = BaseClientMessage::MakeSpecificMethodMetaInfo(baseMetaInfo);
        ret.inRawDataSize = metaInfo.inRawDataSize;
        ret.inObjectCount = metaInfo.inObjectCount;
        ret.outObjectCount = metaInfo.outObjectCount;
        ret.outRawDataSize = metaInfo.outRawDataSize;
        return ret;
    }

    NN_FORCEINLINE Result PrepareForSyncRequest(CmifClientMessageInfo&& info, const SpecificMethodMetaInfo& metaInfo) NN_NOEXCEPT
    {
        auto inRawSize = metaInfo.inRawDataSize;
        auto inObjects = util::Exchange(&info.inObjects, nullptr);
        auto inObjectCount = metaInfo.inObjectCount;
        this->m_OutObjectCount = metaInfo.outObjectCount;
        this->m_OutRawDataSize = metaInfo.outRawDataSize;
        NN_RESULT_DO(m_BaseMessage.PrepareForSyncRequest(std::move(info), metaInfo.baseInfo));
        auto baseInInfo = m_BaseMessage.GetCmifClientMessageInInfo();
        auto p = reinterpret_cast<char*>(baseInInfo.inRawDataPointer);
        {
            cmif::detail::CmifDomainMessageInHeader inHeader = {};
            inHeader.requestKind = cmif::detail::GroupedRequestKind_InvokeMethod;
            inHeader.targetObjectId = m_ObjectInfo.id;
            inHeader.inRawSize = static_cast<decltype(inHeader.inRawSize)>(inRawSize);
            inHeader.inObjectCount = static_cast<decltype(inHeader.inObjectCount)>(inObjectCount);
            std::memcpy(p, &inHeader, sizeof(inHeader));
            p += sizeof(inHeader);
        }
        {
            m_InInfo.inRawDataPointer = reinterpret_cast<uintptr_t>(p);
            p += inRawSize;
        }
        {
            for (int i = 0; i < inObjectCount; ++i)
            {
                auto id = static_cast<const ProxyBaseObject*>(inObjects[i])->GetObjectInfo().id;
                std::memcpy(p, &id, sizeof(id));
                p += sizeof(id);
            }
        }
        NN_RESULT_SUCCESS;
    }

    CmifClientMessageInInfo GetCmifClientMessageInInfo() const NN_NOEXCEPT
    {
        return m_InInfo;
    }

    Result SendSyncRequest() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_BaseMessage.SendSyncRequest());
        auto outInfo = m_BaseMessage.GetCmifClientMessageOutInfo();
        std::memcpy(&m_DomainOutHeader, reinterpret_cast<const void*>(outInfo.outRawDataPointer), sizeof(m_DomainOutHeader));
        outInfo.outRawDataPointer += sizeof(cmif::detail::CmifDomainMessageOutHeader);
        this->m_OutInfo = outInfo;
        NN_RESULT_SUCCESS;
    }

    CmifClientMessageOutInfo GetCmifClientMessageOutInfo() const NN_NOEXCEPT
    {
        return m_OutInfo;
    }

    void AttachOutObjects(CmifBaseObject* outObjects[]) const NN_NOEXCEPT
    {
        auto outObjectIdsBuffer = reinterpret_cast<const char*>(m_OutInfo.outRawDataPointer + m_OutRawDataSize);
        for (int i = 0; i < m_OutObjectCount; ++i)
        {
            auto info = m_ObjectInfo;
            std::memcpy(&info.id, outObjectIdsBuffer, sizeof(info.id));
            outObjectIdsBuffer += sizeof(info.id);
            auto p = static_cast<ProxyBaseObject*>(outObjects[i]);
            if (info.id.value == InvalidCmifDomainObjectId.value)
            {
                p->Detach();
                continue;
            }
            p->Attach(info);
        }
    }

    void GetOutNativeHandles(NativeHandle outNativeHandles[]) const NN_NOEXCEPT
    {
        m_BaseMessage.GetOutNativeHandles(outNativeHandles);
    }

};

template <typename BaseProxyKind>
struct CmifDomainProxy
{
    typedef CmifDomainClientMessage<BaseProxyKind> CmifClientMessage;
    typedef typename CmifClientMessage::ObjectInfo ObjectInfo;
    typedef typename CmifClientMessage::ProxyBaseObject ProxyBaseObject;
    typedef CmifDomainProcessorArgument<BaseProxyKind> ProcessorArgument;
};

template <typename ArgumentInfos, typename = std::tuple<>>
struct CmifDomainProxyClientCoreProcessorCaller;

template <typename ArgumentInfos, typename... Args>
struct CmifDomainProxyClientCoreProcessorCaller<ArgumentInfos, std::tuple<Args...>>
{
    template <typename F>
    NN_FORCEINLINE static Result Call(F&& f, Args&&... args) NN_NOEXCEPT
    {
        static_assert(sf::detail::TypeSizeOf<ArgumentInfos>::value == 0, "");
        return (std::forward<F>(f))(std::forward<Args>(args)...);
    }

    template <typename F, typename... Rests>
    NN_FORCEINLINE static Result Call(F&& f, Args&&... args, InObjectClientArgumentType x, Rests&&... rests) NN_NOEXCEPT
    {
        using ArgumentInfo = typename sf::detail::TypeElementAt<0, ArgumentInfos>::type;
        f.template SetInObject<ArgumentInfo>(x);
        using NextArgumentInfos = typename sf::detail::TypeRemoveHead<ArgumentInfos>::type;
        return CmifDomainProxyClientCoreProcessorCaller<NextArgumentInfos, std::tuple<Args...>>::Call(
            std::forward<F>(f),
            std::forward<Args>(args)...,
            std::forward<Rests>(rests)...
        );
    }

    template <typename F, typename... Rests>
    NN_FORCEINLINE static Result Call(F&& f, Args&&... args, OutObjectClientArgumentType x, Rests&&... rests) NN_NOEXCEPT
    {
        using ArgumentInfo = typename sf::detail::TypeElementAt<0, ArgumentInfos>::type;
        using NextArgumentInfos = typename sf::detail::TypeRemoveHead<ArgumentInfos>::type;
        auto result = CmifDomainProxyClientCoreProcessorCaller<NextArgumentInfos, std::tuple<Args...>>::Call(
            std::forward<F>(f),
            std::forward<Args>(args)...,
            std::forward<Rests>(rests)...
        );
        NN_SF_RESULT_DO(result);
        f.template GetOutObject<ArgumentInfo>(x);
        NN_RESULT_SUCCESS;
    }

    template <typename F, typename Rest, typename... Rests>
    NN_FORCEINLINE static Result Call(F&& f, Args&&... args, Rest&& rest, Rests&&... rests) NN_NOEXCEPT
    {
        using NextArgumentInfos = typename sf::detail::TypeRemoveHead<ArgumentInfos>::type;
        return CmifDomainProxyClientCoreProcessorCaller<NextArgumentInfos, std::tuple<Args..., Rest>>::Call(
            std::forward<F>(f),
            std::forward<Args>(args)...,
            std::forward<Rest>(rest),
            std::forward<Rests>(rests)...
        );
    }
};

template <typename CoreMethodInfo, typename ArgumenInfos = typename CoreMethodInfo::ArgumentInfos>
struct MakeDomainProxyCoreMethodInfo;

template <typename CoreMethodInfo, typename... ArgumenInfos>
struct MakeDomainProxyCoreMethodInfo<CoreMethodInfo, std::tuple<ArgumenInfos...>>
{
    struct IsNotObjectArgumentInfo
    {
        template <typename T>
        using predicate = NN_SF_DETAIL_INTEGRAL_CONSTANT(!(false
            || T::Kind::value == sf::cmif::detail::ArgumentKind::InObject
            || T::Kind::value == sf::cmif::detail::ArgumentKind::OutObject
        ));
    };
    using DomainProxyArgumentInfos = typename sf::detail::TypeFilter<IsNotObjectArgumentInfo, std::tuple<ArgumenInfos...>>::type;
    using InHeaderArgumentInfo = sf::cmif::detail::InRawArgumentInfo<sizeof(cmif::detail::CmifDomainMessageInHeader), NN_ALIGNOF(cmif::detail::CmifDomainMessageInHeader), -static_cast<ptrdiff_t>(sizeof(cmif::detail::CmifDomainMessageInHeader) + sizeof(cmif::CmifInHeader))>;
    using OutHeaderArgumentInfo = sf::cmif::detail::OutRawArgumentInfo<sizeof(cmif::detail::CmifDomainMessageOutHeader), NN_ALIGNOF(cmif::detail::CmifDomainMessageOutHeader), -static_cast<ptrdiff_t>(sizeof(cmif::detail::CmifDomainMessageOutHeader) + sizeof(cmif::CmifOutHeader))>;
    using InObjectsIdsArgumentInfo = sf::cmif::detail::InRawArgumentInfo<CoreMethodInfo::InObjectCount::value * sizeof(CmifDomainObjectId), 1, CoreMethodInfo::InRawSize::value>;
    using OutObjectsIdsArgumentInfo = sf::cmif::detail::OutRawArgumentInfo<CoreMethodInfo::OutObjectCount::value * sizeof(CmifDomainObjectId), 1, CoreMethodInfo::OutRawSize::value>;
    using HeaderArgumentInfos = std::tuple<InHeaderArgumentInfo, OutHeaderArgumentInfo, InObjectsIdsArgumentInfo, OutObjectsIdsArgumentInfo>;
    using BaseArgumenInfos = typename sf::detail::TypeFlatten<std::tuple<DomainProxyArgumentInfos, HeaderArgumentInfos>>::type;
    using BaseInRawSize = NN_SF_DETAIL_INTEGRAL_CONSTANT(sizeof(cmif::detail::CmifDomainMessageInHeader) + CoreMethodInfo::InRawSize::value + CoreMethodInfo::InObjectCount::value * sizeof(CmifDomainObjectId));
    using BaseOutRawSize = NN_SF_DETAIL_INTEGRAL_CONSTANT(sizeof(cmif::detail::CmifDomainMessageOutHeader) + CoreMethodInfo::OutRawSize::value + CoreMethodInfo::OutObjectCount::value * sizeof(CmifDomainObjectId));
    using type = cmif::detail::CoreMethodInfo<BaseArgumenInfos, BaseInRawSize::value, BaseOutRawSize::value, CoreMethodInfo::InProcessIdEnable::value>;
};

/*
    CmifDomainProxyClientCoreProcessor: BaseProxyKind をベースとする SubDomain プロキシの ClientCoreProcessor 実装

    CmifDomainProxyClientCoreProcessorCaller<std::tuple<ArgumentInfos...>>::Call() の中で、
    以下の変換ルールに従って ClientCoreProcessor<BaseProxyKind, ...> に移譲を行う。

    - ArgumentParameter
      - pa.baseParameter を渡す
    - ユーザ引数 (DomainProxyArgumentInfos)
      - InObject: 対応する id の値で、後にまとめて渡す
        - InObject としては渡さない
      - OutObject: 対応する id への OutRaw として、後にまとめて渡す
        - OutObject としては渡さない
      - それ以外はそのままの値を、そのままの順番で渡す
    - 追加引数 (HeaderArgumentInfos)
      - CmifDomainMessageInHeader を InRaw として渡す (InHeaderArgumentInfo)
        - RawOffset = -(sizeof(CmifDomainMessageInHeader) + sizeof(CmifInHeader))
      - CmifDomainMessageOutHeader を OutRaw として渡す (OutHeaderArgumentInfo)
        - RawOffset = -(sizeof(CmifDomainMessageOutHeader) + sizeof(CmifOutHeader))
      - InObject に対応する id 配列を InRaw(id * InObjectCount) として渡す (InObjectsIdsArgumentInfo)
        - rawOffset = CoreMethodInfo::InRawSize::value (ユーザデータのあと)
      - OutObject に対応する id 配列を OutRaw(id * OutObjectCount) として渡す (OutObjectsIdsArgumentInfo)
        - rawOffset = CoreMethodInfo::OutRawSize::value (ユーザデータのあと)

    また、この際、CallObject を渡し、InObject および OutObject に対して特殊処理を行っている。
*/
template <typename BaseProxyKind, typename CoreMethodInfo, typename ArgumentInfos_>
class CmifDomainProxyClientCoreProcessor;

template <typename BaseProxyKind, typename CoreMethodInfo, typename... ArgumentInfos_>
class CmifDomainProxyClientCoreProcessor<BaseProxyKind, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
private:

    using ProxyKind = CmifDomainProxy<BaseProxyKind>;
    using ProcessorArgument = typename ProxyKind::ProcessorArgument;

    using ProxyBaseObject = typename CmifDomainClientMessage<BaseProxyKind>::ProxyBaseObject;
    using BaseCoreMethodInfo = typename MakeDomainProxyCoreMethodInfo<CoreMethodInfo>::type;
    using BaseClientCoreProcessor = ClientCoreProcessor<BaseProxyKind, BaseCoreMethodInfo>;

    struct CallObject
    {
        const ProcessorArgument* pProcessorArgument;
        MethodId methodId;
        SharedPointer<IServiceObject>* pOutObjects;

        size_t inRawPadding;
        size_t outRawPadding;
        cmif::detail::CmifDomainMessageInHeader inHeader;
        cmif::detail::CmifDomainMessageOutHeader outHeader;
        std::array<CmifDomainObjectId, CoreMethodInfo::InObjectCount::value> inIds;
        std::array<CmifDomainObjectId, CoreMethodInfo::OutObjectCount::value> outIds;

        template <typename ArgumentInfo>
        void SetInObject(InObjectClientArgumentType x) NN_NOEXCEPT
        {
            auto p = sf::detail::CmifProxyInfoAccessor::GetCmifBaseObject(x.pInObject);
            inIds[ArgumentInfo::Index::value] = static_cast<ProxyBaseObject*>(p)->GetObjectInfo().id;
        }

        template <typename... Args>
        NN_FORCEINLINE Result operator()(Args&&... args) NN_NOEXCEPT
        {
            NN_SF_RESULT_DO(BaseClientCoreProcessor::ProcessImpl(
                pProcessorArgument->baseArgument,
                std::forward<Args>(args)...,
                inHeader,
                RawClientArgumentInternalValue{&outHeader},
                RawClientArgumentInternalValue{inIds.data()},
                RawClientArgumentInternalValue{outIds.data()},
                methodId,
                nullptr, // pOutObjects
                inRawPadding + sizeof(inHeader),
                outRawPadding + sizeof(inHeader)
            ));

            for (auto i = 0u; i < CoreMethodInfo::OutObjectCount::value; ++i)
            {
                auto id = outIds[i];
                if (id.value == InvalidCmifDomainObjectId.value)
                {
                    pOutObjects[i].Reset();
                    continue;
                }
                auto p = static_cast<ProxyBaseObject*>(sf::detail::CmifProxyInfoAccessor::GetCmifBaseObject(pOutObjects[i].Get()));
                p->Attach(ProcessorArgument{pProcessorArgument->baseArgument, id});
            }
            NN_SF_RESULT_SUCCESS;
        }

        template <typename ArgumentInfo>
        void GetOutObject(OutObjectClientArgumentType x) NN_NOEXCEPT
        {
            x.GetOutObject<ArgumentInfo>(pOutObjects);
        }
    };

    NN_FORCEINLINE static Result ProcessImpl(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p->GetProcessorArgument(), std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects, 0, 0);
    }

public:

    NN_FORCEINLINE static Result ProcessImpl(const ProcessorArgument& pa, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[], size_t inRawPadding, size_t outRawPadding) NN_NOEXCEPT
    {
        CallObject callObject;
        callObject.pProcessorArgument = &pa;
        callObject.methodId = methodId;
        callObject.pOutObjects = pOutObjects;
        callObject.inRawPadding = inRawPadding;
        callObject.outRawPadding = outRawPadding;
        callObject.inHeader = {};
        callObject.inHeader.requestKind = cmif::detail::GroupedRequestKind_InvokeMethod;
        callObject.inHeader.targetObjectId = pa.id;
        callObject.inHeader.inRawSize = static_cast<decltype(callObject.inHeader.inRawSize)>(sizeof(cmif::CmifInHeader) + CoreMethodInfo::InRawSize::value);
        callObject.inHeader.inObjectCount = static_cast<decltype(callObject.inHeader.inObjectCount)>(CoreMethodInfo::InObjectCount::value);

        using Caller = CmifDomainProxyClientCoreProcessorCaller<std::tuple<ArgumentInfos_...>>;
        return Caller::Call(callObject, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)...);
    }

    NN_NOINLINE static Result Process(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, nullptr);
    }

    NN_NOINLINE static Result ProcessWithOutObjects(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects);
    }

};

/*
    SubDomain プロキシに対する ClientCoreProcessor 特殊化

    実装は CmifDomainProxyClientCoreProcessor に移譲している。
*/
template <typename BaseProxyKind, typename CoreMethodInfo, typename... ArgumentInfos_>
struct ClientCoreProcessor<CmifDomainProxy<BaseProxyKind>, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
    : public CmifDomainProxyClientCoreProcessor<BaseProxyKind, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
};

}}}}
