﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Pinmux ライブラリの設定変更機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>

#include "pinmux_Type.h"

namespace nn {
namespace pinmux {

/**
* @brief Pinmux の ピングループごとのセッション用パラメータです。
*
* @details Pinmux の ピングループごとのセッション用パラメータです。内部の変数に直接アクセスしないでください。
*/
struct PinmuxSession
{
    void* _sessionHandle;
};


/**
 * @brief 指定したピングループとのセッションをオープンします。
 *
 * @param[out]  pOutSession           ピングループごとのセッション用パラメータ
 * @param[in]   name                  アクセス対象のピングループの名前
 * @pre         ライブラリは初期化済みである。
 * @post        pOutSession に必要な情報が入力され、指定されたピングループがコントロール可能な状態になります。
 * @details     指定したピングループとのセッションをオープンします。アクセスするピングループごとにセッションをオープンをしてください。
 */
void OpenSession(PinmuxSession* pOutSession, AssignablePinGroupName name) NN_NOEXCEPT;

/**
 * @brief 指定したピングループとのセッションをクローズします。
 *
 * @param[in]   pSession       ピングループごとのセッション用パラメータ
 * @pre         pSession がオープンされている。
 * @post        pSession で指定されたピングループとの通信を終了します。
 * @details     pSession で指定されたピングループとの通信を終了します。
 */
void CloseSession(PinmuxSession* pSession) NN_NOEXCEPT;

/**
 * @brief ピングループの機能を設定します。
 *
 * @param[in]   pSession       ピングループごとのセッション用パラメータ
 * @param[in]   assignment     設定したいピングループの機能
 * @pre         pSession がオープンされている。
 * @post        pSession で指定されたピングループが assingnment で指定された機能に設定されます。
 * @details     pSession で指定されたピングループが assingnment で指定された機能に設定されます。
 */
void SetPinAssignment(PinmuxSession* pSession, PinAssignment assignment) NN_NOEXCEPT;


/**
 * @brief ピングループの機能を取得します。
 *
 * @param[in]   pSession       ピングループごとのセッション用パラメータ
 * @pre         pSession がオープンされている。
 * @post        pSession で指定されたピングループが assingnment で指定された機能に設定されます。
 * @details     pSession で指定されたピングループが assingnment で指定された機能に設定されます。
 */
PinAssignment GetPinAssignment(PinmuxSession* pSession) NN_NOEXCEPT;

} // pinmux
} // nn


