﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief オーバーレイ通知表示用の通知を送信するための機能を提供します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_Result.h>
#include <type_traits>

namespace nn { namespace ovln {

//! @name 送信(オーバレイ通知表示用)
//! @{

/**
    @brief オーバレイ通知送信ライブラリを初期化します。

    @retresult
    @endretresult

    @details
     オーバレイ通知送信ライブラリを初期化します。
     この関数は複数回呼ぶことができ、
     この関数と FinalizeSenderLibraryForOverlay() 関数とが同じ回数だけ呼ばれている状態でこの関数を呼んだときに限り、
     実際の初期化処理が行われます。
*/
Result InitializeSenderLibraryForOverlay() NN_NOEXCEPT;

/**
    @brief オーバレイ通知送信ライブラリを終了します。

    @details
     オーバレイ送信受信ライブラリを終了します。
     InitializeSenderLibraryForOverlay() と同じ回数だけ呼んだ場合に実際の終了処理が行われます。
*/
void FinalizeSenderLibraryForOverlay() NN_NOEXCEPT;

/**
    @brief オーバレイ通知送信に使用するための型です。

    @details
     以下の状態を持ちます。

     - 未初期化状態 (デフォルト)
     - 初期化済み状態

    @see InitializeSenderForValue(SenderForOverlayType*), InitializeSenderWithQueue(SenderForOverlayType*, int)
*/
struct SenderForOverlayType
{
    std::aligned_storage<32>::type _storage;
};

/**
    @brief SenderForOverlayType を値送信向けに初期化します。

    @param[out] pSender 初期化する SenderForOverlayType の変数へのポインタを指定します。

    @retresult
    @endretresult

    @pre オーバレイ通知送信ライブラリが初期化されている
    @pre pSender が未初期化状態である
    @post pSender が初期化済み状態である

    @details
     pSender を、一つの値の変化を通知するために使用するように初期化します。

     この関数で初期化された pSender は、メッセージを一つだけ保持できるキューを持ちます。
     キューにメッセージがないときに送信を行うと、このキューに保持され、
     受信されるとキューは空になります。
     受信されていないメッセージがキューにたまっているときに送信を行うと、
     既に存在したメッセージは破棄され、新しいメッセージがキューに保持されます。

     この関数で初期化された pSender に対して Send(SenderForOverlayType*, const Message&) を呼んだ場合には、
     必ず true が返ります。
*/
Result InitializeSenderForValue(SenderForOverlayType* pSender) NN_NOEXCEPT;

/**
    @brief SenderForOverlayType を値送信向けに初期化します。

    @param[out] pSender 初期化する SenderForOverlayType の変数へのポインタを指定します。
    @param[in] queueLength キューの長さを指定します。

    @retresult
    @endretresult

    @pre オーバレイ通知送信ライブラリが初期化されている
    @pre pSender が未初期化状態である
    @pre queueLength > 0
    @post pSender が初期化済み状態である

    @details
     pSender を、指定した長さのキューを使って通知を行うように初期化します。

     この関数で初期化された pSender は、 queueLength 個のメッセージを保持できるキューを持ちます。
     キューは FIFO (first-in-first-out) としてふるまいます。

     この関数で初期化された pSender に対して Send(SenderForOverlayType*, const Message&) を呼んだ場合には、
     キューに空きがあればエンキューした後に true を、キューに空きがなければ false を返します。
*/
Result InitializeSenderWithQueue(SenderForOverlayType* pSender, int queueLength) NN_NOEXCEPT;

/**
    @brief SenderForOverlayType を終了します。

    @param[in] pSender 終了する SenderForOverlayType の変数へのポインタを指定します。

    @pre オーバレイ通知送信ライブラリが初期化されている。
    @pre pSender が初期化済み状態である

    @details
     指定された SenderForOverlayType を破棄します。

     キューに未受信のメッセージが存在した場合には、全て破棄され、以後受信はできなくなります。
     メッセージを確実に受信させるため、一般的な用途では、プログラムの開始時に初期化したのち、
     本関数は呼ばずにプログラムを終了することを推奨します。

     テスト・デバッグ用途で、繰り返し実行するために本関数を呼ぶ必要があり、
     送信したメッセージを確実に受信したいような場合には、
     GetUnreceivedMessageCount(const SenderForOverlayType*) の返り値が 0 になることを
     適度なスリープを挟んでポーリングしたのちに本関数を呼ぶようにしてください。
*/
void FinalizeSender(SenderForOverlayType* pSender) NN_NOEXCEPT;

/**
    @brief メッセージを送信します。

    @param[in] pSender 送信対象の SenderForOverlayType の変数へのポインタを指定します。
    @param[in] message 送信するメッセージを指定します。

    @pre オーバレイ通知送信ライブラリが初期化されている
    @pre pSender が初期化済み状態である
    @pre message にデータが正しく設定されている

    @return 正しく送信できた場合には true を返し、そうでない場合には false を返します。

    @details
     pSender に message を送信します。
     送信に成功した場合には true を返し、失敗した場合には false を返します。
*/
bool Send(SenderForOverlayType* pSender, const Message& message) NN_NOEXCEPT;

/**
    @brief 未受信のメッセージ数を取得します。

    @param[in] pSender 対象の SenderForOverlayType の変数へのポインタを指定します。

    @pre オーバレイ通知送信ライブラリが初期化されている
    @pre pSender が初期化済み状態である

    @return  未受信のメッセージ数を返します。

    @details
     pSender に送信したメッセージのうち、未受信のメッセージ数を取得します。
     この関数から返る値は、受信されることによって、非同期に変わりえます。
*/
uint32_t GetUnreceivedMessageCount(const SenderForOverlayType* pSender) NN_NOEXCEPT;

//! @}

}}
