﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace ovln { namespace format {

//! @brief ペアレンタルコントロールの設定が変更されたことを示すメッセージです。
const uint32_t PctlSettingChangedMessage = 4000;
//! @brief プレイタイマーの残り時間・超過時間を通知するメッセージです。
const uint32_t PctlRemainingTimeMessage = 4001;
//! @brief サーバー連携の解除が行われたことを通知するメッセージです。
const uint32_t PctlUnlinkMessage = 4002;
//! @brief プレイタイマー開始時に時間が超過していることを通知するメッセージです。
const uint32_t PctlTimeExceededOnStartMessage = 4003;
//! @brief アラーム通知設定が変化したことを通知するメッセージです。
const uint32_t PctlAlarmSettingChangedMessage = 4004;

//! @brief 変更されたペアレンタルコントロール設定の種類を示すフラグです。
struct PctlSettingTypeFlag
{
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<0> Rating;            //! レーティング制限
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<1> SnsPost;           //! SNS投稿
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<2> FreeCommunication; //! ほかの人との自由なコミュニケーション
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<3> PinCode;           //! ペアコン一時解除コード
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<4> PlayTimer;         //! プレイタイマー
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<5> SafetyLevel;       //! 安心レベル
    typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag>::Flag<6> ExemptAppList;     //! 制限対象外リスト
};

//! @brief 変更されたペアレンタルコントロール設定の種類を示すフラグの型です。
typedef ::nn::util::BitFlagSet<32, PctlSettingTypeFlag> PctlSettingTypeFlagSet;

//! @brief プレイタイマーの時間切れにおける制限モードを表す定数値です。
enum PctlPlayTimerModeFlag : uint8_t
{
    PctlPlayTimerModeFlag_Alarm = 0,              //!< アラームモード
    PctlPlayTimerModeFlag_Suspend                 //!< 強制中断モード
};

//! @brief 連携解除の理由を表す定数値です。
enum PctlUnlinkReasonFlag : uint8_t
{
    PctlUnlinkReasonFlag_FromServer = 0,          //!< サーバーから連携解除を通知されました
    PctlUnlinkReasonFlag_IndirectUnlink,          //!< 別の通信処理等によって連携が切れていることが分かったため連携解除しました
    PctlUnlinkReasonFlag_Manual,                  //!< (予約されていますが使用されません) 本体での操作によって連携解除されました
    PctlUnlinkReasonFlag_PinCodeReset             //!< (予約されていますが使用されません) 解除コードの初期化が行われたため連携解除しました
};

//! @brief アラーム通知の設定値を表す列挙子です。
enum PctlAlarmSettingFlag : uint8_t
{
    PctlAlarmSettingFlag_Enabled = 0,             //!< アラーム通知が有効(アラームON)になりました。
    PctlAlarmSettingFlag_Disabled = 1             //!< アラーム通知が無効(アラームOFF)になりました。
};

//! @brief ペアレンタルコントロールの設定変更時の追加情報を保持する構造体です。
struct PctlSettingChangedMessageData
{
    //! @brief 変更された設定の種類を表します。
    PctlSettingTypeFlagSet settingType;
};

//! @brief プレイタイマーの残り時間通知に関する追加情報を保持する構造体です。
struct PctlRemainingTimeMessageData
{
    //! @brief 残り時間を表します(秒単位)。0 の場合はモードによって「時間になった場合」または「時間切れでまもなく中断される場合」を表します。
    //!        また、負数は時間が超過していることを表します(絶対値が超過時間(秒単位)です)。
    int16_t remainingTime;
    //! @brief タイマーの制限モードを表します。
    PctlPlayTimerModeFlag playTimerMode;
};

//! @brief サーバー連携解除通知に関する追加情報を保持する構造体です。
struct PctlUnlinkMessageData
{
    //! @brief 連携解除の理由を表します。
    PctlUnlinkReasonFlag reason;
    uint8_t _reserved[3];
};

//! @brief プレイタイマー開始時に時間が超過している際の通知に関する追加情報を保持する構造体です。
struct PctlTimeExceededOnStartMessageData
{
    //! @brief プレイタイマー開始時点での超過時間を表します(秒単位)。
    //! @details この値は常に 0 以上になります。(PctlRemainingTimeMessageData::remainingTime と符号が異なります。)
    int64_t exceedingTime;
    //! @brief タイマーの制限モードを表します。
    PctlPlayTimerModeFlag playTimerMode;
};

//! @brief アラーム通知設定が変化した際に発行される通知の追加情報を保持する構造体です。
struct PctlAlarmSettingChangedMessageData
{
    //! @brief アラームの設定値を表します。
    PctlAlarmSettingFlag alarmFlag;
    uint8_t _reserved[3];
};

}}}
