﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/hid/hid_NpadColor.h>

namespace nn { namespace ovln { namespace format {

    //! @brief コントローラーの電池残量低下を表すメッセージです。非サポート 互換性維持のため残す
    const uint32_t ControllerBatteryLevelLowDataTag     = 6001;
    //! @brief コントローラーの切断を表すメッセージです。
    const uint32_t ControllerDisconnectionDataTag     = 6002;
    //! @brief コントローラーのペアリング完了を表すメッセージです。
    const uint32_t ControllerPairingCompletedDataTag = 6003;
    //! @brief コントローラーが本体に装着されたことを表すメッセージです。
    const uint32_t ControllerAttachToConsoleDataTag = 6004;
    //! @brief コントローラーの電池残量低下を表すメッセージです。
    const uint32_t ControllerBatteryLevelLowExDataTag     = 6005;


    //! @brief コントローラーの電池状態に関する情報を表す型です。
    struct ControllerPowerInfo
    {
        //! @brief 電池残量情報が有効かどうか。false の場合は電池残量が取得できないデバイスや状態です。
        bool isValid;
        NN_PADDING7;

        //! @brief 電池状態です。
        nn::hid::system::NpadPowerInfo powerInfo;
    };

    //! @brief コントローラーの色情報に関する情報を表す型です。
    struct ControllerColorInfo
    {
        //! @brief 色情報が有効かどうか。false の場合は色情報が取得できないデバイスや状態です。
        bool isValid;
        NN_PADDING7;

        //! @brief 色情報です。
        nn::hid::NpadControllerColor color;
    };

    //! @brief 通知されたコントローラーに関する情報を表す型です。
    struct ControllerInfo
    {
        //! @brief コントローラーの種類です。
        nn::hid::system::NpadDeviceTypeSet deviceType;

        //! @brief Joy-Con 2 本持ちの場合に、どの Joy-Con (Joy-Con (R) / Joy-Con (L) / 両方)にイベントが起きたかを表す型です。
        nn::hid::system::NpadDeviceTypeSet joyType;

        //! @brief Joy-Con 1 本持ちの場合に、縦持ちか横持ちかを表す型です。
        int64_t joyHoldType;

        //! @brief コントローラーのプレイヤーランプの点灯パターンです。
        uint64_t ledPattern;

        //! @brief 電池の状態の情報です。
        union {
            struct {
                ControllerPowerInfo left;
                ControllerPowerInfo right;
            } joyBattery;
            ControllerPowerInfo fullKeyBattery;
        };


        //! @brief コントローラーの色情報です。
        union {
            struct {
                ControllerColorInfo left;
                ControllerColorInfo right;
            } joyColor;
            ControllerColorInfo fullKeyColor;
        };

    public:
        //! @brief Joy-Con の持ち方を取得します。
        nn::hid::NpadJoyHoldType GetJoyHoldType() const NN_NOEXCEPT
        {
            switch (joyHoldType)
            {
            case nn::hid::NpadJoyHoldType_Vertical:
                return nn::hid::NpadJoyHoldType_Vertical;
            case nn::hid::NpadJoyHoldType_Horizontal:
                return nn::hid::NpadJoyHoldType_Horizontal;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };

    //! @brief コントローラーの電池残量低下を表すメッセージの型です。(非サポート。互換維持のために定義のみ残しています)
    struct ControllerBatteryLevelLowData
    {
        //! @brief 電池残量が低下したコントローラーの情報です。
        ControllerInfo controllerInfo;
    };

    //! @brief コントローラーの電池残量低下を表すメッセージの型です
    struct ControllerBatteryLevelLowExData
    {
        //! @brief 電池残量が低下したコントローラーの情報です。
        ControllerInfo controllerInfo;
        //! @brief 電池残量が低下したコントローラーの NpadIdType です
        nn::hid::NpadIdType npadId;
    };

    //! @brief コントローラー切断の理由です
    enum ControllerDisconnectionReason : uint64_t
    {
        ControllerDisconnectionReason_Battery,       //!< 電池残量低下によりコントローラーが切断されました
        ControllerDisconnectionReason_Unknown,       //!< 不明な原因でコントローラーが切断されました。
    };

    //! @brief コントローラーの切断を表すメッセージの型です。
    struct ControllerDisconnectionData
    {
        //! @brief 切断されたコントローラーを表す情報です。
        ControllerInfo controllerInfo;

        //! @brief コントローラー切断の理由です。
        ControllerDisconnectionReason reason;
    };

    //! @brief コントローラーのペアリングの手法です。
    enum ControllerPairingMethod : uint64_t
    {
        ControllerPairingMethod_Joint,       //!< 本体にジョイントしてペアリングされました。
        ControllerPairingMethod_USB,         //!< USB 接続でペアリングされました。
        ControllerPairingMethod_Button,      //!< ボタン押しでペアリングされました。
    };

    //! @brief 挿抜のイベントの種類です
    enum ControllerAttachEventType : uint64_t
    {
        ControllerAttachEventType_ConsoleRailAttached,     //!< 本体レール部にジョイントされました。
        ControllerAttachEventType_ConsoleRailDetached,     //!< 本体レール部から取り外されました
        ControllerAttachEventType_ExternalGripAttached,    //!< 通電式拡張グリップに装着されました。
        ControllerAttachEventType_ExternalGripDetached,    //!< 通電式拡張グリップから取り外されました。
    };

    //! @brief コントローラーのペアリング完了を表す型です。
    struct ControllerPairingCompletedData
    {
        //! @brief コントローラーの種類です。
        nn::hid::system::NpadDeviceTypeSet deviceType;

        //! @brief コントローラーの色情報です。
        ControllerColorInfo color;

        //! @brief ペアリングの手法
        ControllerPairingMethod pairingMethod;
    };

    //! @brief コントローラーが本体に装着されたことを表す型です。
    struct ControllerAttachToConsoleData
    {
        //! @brief コントローラーの種類です。
        nn::hid::system::NpadDeviceTypeSet deviceType;

        //! @brief コントローラーの色情報です。
        ControllerColorInfo color;

        //! @brief イベントの種類
        ControllerAttachEventType eventType;
    };
}}}
