﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace capsrv {

    struct AlbumEntry;

}}

namespace nn { namespace ovln { namespace format {

    //! @brief 画面写真の撮影に成功したことを表すメッセージです。
    const uint32_t ScreenShotSuccessDataTag     = 8001;
    //! @brief 画面写真の撮影に失敗したことを表すメッセージです。
    const uint32_t ScreenShotFailureDataTag     = 8002;
    //! @brief 画面写真の撮影が禁止されている時に撮影が要求されたことを表すメッセージです。
    const uint32_t ScreenShotIsNotPermittedDataTag = 8003;
    //! @brief 画面写真のサムネイルが準備できたことを表すメッセージです。
    const uint32_t ScreenShotThumbnailIsReadyDataTag = 8004;

    //! @brief 撮影理由です。
    enum class ScreenShotReason : uint8_t
    {
        //! @brief 撮影ボタンが押されました。
        CaptureButtonPressed = 0,
        //! @brief アプリケーションが保存 API を呼びました。
        SaveApiCalledByApplication = 1,
    };

    //! @brief 撮影結果です。
    enum class ScreenShotResult : uint8_t
    {
        //! @brief 画面写真の撮影に成功しました。
        Success = 0,
        //! @brief 画面写真の撮影に失敗しました。
        Failure,
        //! @brief 画面写真の撮影が禁止されているため撮影を行いませんでした。
        NotPermitted,
        //! @brief アルバムが一杯のため画面写真の撮影に失敗しました。
        AlbumIsFull,
        //! @brief アルバムのファイル数上限に達しているため画面写真の撮影に失敗しました。
        AlbumFileCountLimit,
        //! @brief サムネイルが取得可能になりました。
        ThumbnailIsReady,
        //! @brief ファイルシステムのエラーのため画面写真の保存に失敗しました。
        AlbumFileSystemError,
        //! @brief 画面キャプチャのエラーのため画面写真の作成に失敗しました。
        ScreenCaptureError,
    };

    //! @brief 撮影結果のフラグです。
    struct ScreenShotResultFlag
    {
        //! @brief 撮影した画面写真のサムネイルを取得できるかを表します。
        typedef nn::util::BitFlagSet<8, ScreenShotResultFlag>::Flag<0> IsThumbnailAvailable;
    };

    //! @brief 撮影結果のフラグの型です。
    typedef nn::util::BitFlagSet<8, ScreenShotResultFlag> ScreenShotResultFlagSet;

    //! @brief 画面写真の撮影結果を通知するメッセージの型です。
    struct ScreenShotCapturedData
    {
        //! @brief 撮影理由です。
        ScreenShotReason reason;

        //! @brief 撮影結果です。
        //! @details
        //!   メッセージのタグに対応した値が設定されます。
        ScreenShotResult result;

        NN_PADDING2;

        //! @brief 撮影結果のフラグです。
        //! @details
        //!   フラグの値は各取得関数で調べることができます。
        //! @see IsThumbailAvailable()
        ScreenShotResultFlagSet flags;

        //! @brief 撮影ファイルのエントリーです。
        //! @detail
        //!   値は GetAlbumEntry() で取得することができます。
        std::aligned_storage<32,8>::type _entry;

    public:
        //! @brief サムネイルを取得できるかを取得します。
        //! @return サムネイルを取得できる場合 true を返します。
        //! @details
        //!   撮影に成功した場合サムネイルがあれば true を返します。
        //!   撮影に失敗した場合には必ず false を返します。
        bool IsThumbnailAvailable() const NN_NOEXCEPT
        {
            return this->flags.Test(ScreenShotResultFlag::IsThumbnailAvailable::Index);
        }

        //! @brief 撮影ファイルのエントリーを取得します。
        //! @details
        //!   撮影に成功した場合保存されたファイルのエントリーを返します。
        //!   アルバムが一杯のため撮影に失敗した場合保存しようとしたファイルの ID を含むエントリーを返します。
        //!   このときエントリーのファイルサイズは負の値となり、エントリーのファイル ID を使ってファイルを読み込むことはできません。
        //!   その他の理由で撮影に失敗した場合に返される値は不定です。
        const nn::capsrv::AlbumEntry& GetAlbumEntry() const
        {
            return *reinterpret_cast<const nn::capsrv::AlbumEntry*>(&_entry);
        }



    };


}}}
