﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Transfer memory 機能に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_TransferMemoryTypes.h>
#include <nn/os/os_TransferMemoryApi.h>

namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   Transfer memory を扱うためのクラスです。
 *
 * @details
 *  Transfer memory は、プロセス間でメモリの受け渡しを行うためのオブジェクトです。
 *
 *  nn::os::TransferMemory クラスは、@ref nn::os::TransferMemoryType オブジェクトと
 *  関連する API の呼び出しをラッピングしたユーティリティクラスです。
 *  本クラスが提供するメンバ関数は、@ref nn::os::TransferMemoryType 型を引数にとる
 *  @ref nn::os::AttachTransferMemory() などの API を発行する形で実装されています。
 */
class TransferMemory
{
    NN_DISALLOW_COPY( TransferMemory );
    NN_DISALLOW_MOVE( TransferMemory );

public:
    //--------------------------------------------------------------------------
    /**
     * @brief   Transfer memory オブジェクトを未初期化で用意します。
     *
     * @details
     *  コンストラクタです。@n
     *  内部の Transfer memory オブジェクトは未初期化状態になります。
     *  この後に Attach() することで、Transfer memory が使用できるようになります。
     */
    TransferMemory() NN_NOEXCEPT
    {
        m_TransferMemory._state = TransferMemoryType::State_Uninitialized;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   Transfer memory オブジェクトを作成し初期化します。
     *
     * @param[in]  addr                Transfer memory にする領域の先頭アドレス
     * @param[in]  size                Transfer memory にする領域のサイズ
     * @param[in]  permission          呼び出し元に許可するメモリアクセス権
     *
     * @details
     *  コンストラクタです。@n
     *  内部の TransferMemoryType オブジェクトを指定されたパラメータで初期化します。@n
     *  詳細は nn::os::CreateTransferMemory() を参照して下さい。
     *
     *  ただし、本クラスでは nn::os::CreateTransferMemory() とは異なり、
     *  ハンドルまたは Transfer memory のリソースが足りない場合アボートします。
     *
     */
    TransferMemory(void* addr, size_t size, MemoryPermission permission) NN_NOEXCEPT
    {
        auto result = CreateTransferMemory(&m_TransferMemory, addr, size, permission);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   指定されたハンドルを使って Transfer memory オブジェクトを初期化します。
     *
     * @param[in]  size            Transfer memory のサイズ
     * @param[in]  handle          Transfer memory のハンドル
     * @param[in]  managed         Transfer memory のハンドル管理フラグ
     *
     * @details
     *  コンストラクタです。@n
     *  指定されたハンドルを使って Transfer memory オブジェクトを初期化します。@n
     *  詳細は nn::os::AttachTransferMemory() を参照して下さい。
     *
     */
    TransferMemory(size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        AttachTransferMemory(&m_TransferMemory, size, handle, managed);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief  TransferMemory オブジェクトを破棄します。
     *
     * @details
     *  デストラクタです。@n
     *  TransferMemory オブジェクトを破棄します。@n
     *  マップされている場合には、自動的にアンマップされます。@n
     *  詳細は nn::os::DestroyTransferMemory() を参照して下さい。
     *
     *  ただし、本クラスでは nn::os::DestroyTransferMemory() とは異なり、
     *  Transfer memory オブジェクトが未初期化の場合、何もしません。
     *
     */
    ~TransferMemory() NN_NOEXCEPT
    {
        if (m_TransferMemory._state == TransferMemoryType::State_Uninitialized)
        {
            return;
        }
        DestroyTransferMemory(&m_TransferMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   既存の Transfer memory ハンドルを関連付けます
     *
     * @param[in]  size            Transfer memory のサイズ
     * @param[in]  handle          Transfer memory のハンドル
     * @param[in]  managed         Transfer memory のハンドル管理フラグ
     *
     * @details
     *  指定されたハンドルを使って Transfer memory オブジェクトを初期化します。@n
     *  詳細は nn::os::AttachTransferMemory() を参照して下さい。
     *
     */
    void Attach(size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        AttachTransferMemory(&m_TransferMemory, size, handle, managed);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   Transfer memory ハンドルを切り離します
     *
     * @return  Transfer memory のハンドルを返します。
     *
     * @details
     *  詳細は nn::os::DetachTransferMemory() を参照して下さい。
     */
    NativeHandle Detach() NN_NOEXCEPT
    {
        return DetachTransferMemory(&m_TransferMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   Transfer memory をマップします。
     *
     * @param[out] pOutAddress     Transfer memory をマップしたアドレス
     * @param[in]  ownerPermission Transfer memory を生成する際に満たすべきアクセス権
     *
     * @retresult
     *   @handleresult{nn::os::ResultInvalidHandle}
     *   @handleresult{nn::os::ResultInvalidTransferMemoryState}
     *   @handleresult{nn::os::ResultInvalidTransferMemorySize}
     *   @handleresult{nn::os::ResultOutOfAddressSpace}
     * @endretresult
     *
     * @details
     *  詳細は nn::os::MapTransferMemory() を参照して下さい。
     */
    nn::Result Map(void** pOutAddress, MemoryPermission ownerPermission) NN_NOEXCEPT
    {
        return MapTransferMemory(pOutAddress, &m_TransferMemory, ownerPermission);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   Transfer memory をアンマップします。
     *
     * @details
     *  詳細は nn::os::UnmapTransferMemory() を参照して下さい。
     */
    void Unmap() NN_NOEXCEPT
    {
        UnmapTransferMemory(&m_TransferMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   TransferMemoryType オブジェクトへの参照を返します。
     *
     * @return  TransferMemoryType オブジェクトへの参照
     *
     * @details
     *  自インスタンスが持つ TransferMemoryType オブジェクトへの参照を返します。
     *  この変換演算子があるため、TransferMemoryType& を引数にとる関数に対して、
     *  TransferMemory& を指定することが可能です。
     */
    NN_IMPLICIT operator TransferMemoryType&() NN_NOEXCEPT
    {
        return m_TransferMemory;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   TransferMemoryType オブジェクトへの const 参照を返します。
     *
     * @return  TransferMemoryType オブジェクトへの const 参照
     *
     * @details
     *  自インスタンスが持つ TransferMemoryType オブジェクトへの const 参照を返します。
     *  この変換演算子があるため、const TransferMemoryType& を引数にとる関数に対して、
     *  TransferMemory& を指定することが可能です。
     */
    NN_IMPLICIT operator const TransferMemoryType&() const NN_NOEXCEPT
    {
        return m_TransferMemory;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   自インスタンスの TransferMemoryType オブジェクトへのポインタを返します。
     *
     * @return  TransferMemoryType オブジェクトへのポインタ
     *
     * @details
     *  自インスタンスが持つ TransferMemoryType オブジェクトへのポインタを返します。
     */
    TransferMemoryType* GetBase() NN_NOEXCEPT
    {
        return &m_TransferMemory;
    }


private:
    TransferMemoryType m_TransferMemory;
};

}}  // namespace nn::os

