﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッドに関する API の宣言（private パッケージ）
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_ThreadTypes.private.h>

namespace nn { namespace os {

//--------------------------------------------------------------------------

struct  ThreadType;

//--------------------------------------------------------------------------
/**
 * @brief   スレッドサスペンドカウント値の最大値です。
 *
 */
const int ThreadSuspendCountMax = 127;


//! @name スレッド関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   指定されたスレッドを中断させます。
 *
 * @param[in] thread    ThreadType オブジェクトへのポインタ
 *
 * @return  本関数を呼び出す前のサスペンドカウント値を返します。
 *
 * @pre
 *  - thread の状態が Started である
 *  - thread != nn::os::GetCurrentThread()
 *  - nn::os::GetThreadSuspendCount() < nn::os::ThreadSuspendCountMax
 *
 * @post
 *  - nn::os::GetThreadSuspendCount() > 0
 *
 * @details
 *  対象スレッドを中断させます。自スレッドを指定することはできません。
 *
 *  より具体的には、各スレッドにはサスペンドカウント値があり、
 *  この値が 0 より大きい場合は対象スレッドの実行が中断されます。
 *  スレッドが作成された時点でのサスペンドカウント値は 0 です。
 *
 *  本 API ではこのサスペンドカウント値をインクリメント（+1）することで
 *  対象スレッドの実行を中断させます。サスペンドカウント値が
 *  nn::os::ThreadSuspendCountMax より大きくなる場合には
 *  内部のアサートに失敗します。
 *
 */
int SuspendThread(ThreadType* thread) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   指定されたスレッドのサスペンドカウント値を減らします。
 *
 * @param[in] thread    ThreadType オブジェクトへのポインタ
 *
 * @return  本関数を呼び出す前のサスペンドカウント値を返します。
 *
 * @pre
 *  - thread の状態が Started である
 *
 * @details
 *  対象スレッドのサスペンドカウンタ値を減らし、その値が 0 になった場合に
 *  対象スレッドの中断を解除し、動作を再開させます。
 *
 *  より具体的には、本 API は最初に現在のサスペンドカウンタ値を検査します。
 *  サスペンドカウント値が 0 の場合、本 API は何も行なわずにリターンします。
 *  サスペンドカウント値が 1 以上の場合、サスペンドカウント値を
 *  デクリメント（-1）し、その結果 0 になった場合に対象スレッドの動作を
 *  再開させます。
 *
 */
int ResumeThread(ThreadType* thread) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   指定されたスレッドの現在のサスペンドカウント値を取得します。
 *
 * @param[in] thread    ThreadType オブジェクトへのポインタ
 *
 * @return  現在のサスペンドカウント値を返します。
 *
 * @pre
 *  - thread の状態が Started である
 *
 * @details
 *  対象スレッドの現在のサスペンドカウント値を取得します。
 *  カウント値は常に 0 ～ nn::os::ThreadSuspendCountMax の範囲となります。
 *
 */
int GetThreadSuspendCount(const ThreadType* thread) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドのコンテキスト情報を取得します。
 *
 * @param[out] pContext 対象スレッドのコンテキスト情報の保存先ポインタ
 * @param[in]  thread   ThreadType オブジェクトへのポインタ
 *
 * @pre
 *  - thread の状態が Started である
 *  - nn::os::GetThreadSuspendCount() > 0
 *
 * @details
 *  対象スレッドのスレッドコンテキスト情報を取得します。
 *  対象スレッドは中断中、つまり、サスペンドカウント値が 0 より大きく
 *  なければなりません。サスペンドカウント値が 0 の場合、本 API は
 *  内部のアサートに失敗します。
 *
 */
void GetThreadContext(ThreadContextInfo* pContext, const ThreadType* thread) NN_NOEXCEPT;


//! @}

//--------------------------------------------------------------------------

}} // namespace nn::os

