﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッドに関する型や定数の宣言
 */

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_ThreadLocalStorageCommon.h>

namespace nn { namespace os {

//-----------------------------------------------------------------------------
/**
 * @brief   ThreadType 構造体内の SDK 専用 TLS 領域のデータ構造を定義する構造体の型です。
 *
 * @details
 *  SDK 開発者は、各 ThreadType 構造体内の TLS 配列領域内の一部を、
 *  静的確保の TLS 領域として予約し、使用することができます。
 *
 *  本構造体は、静的確保の TLS 領域を定義するためのもので、
 *  OS 以外のライブラリモジュール専用の TLS 変数を定義しておくための
 *  領域として用意されています。ただし、ここで定義を追加した分だけ、
 *  nn::os::SdkAllocateTlsSlot() で確保できる SDK 用の TLS スロットは
 *  減少します。
 *
 *  ThreadType* 構造体ポインタから上記 TLS 領域の先頭アドレスを
 *  取得するには nn::os::SdkGetInternalTlsRegion() をご使用下さい。
 *
 *  この領域に TLS メンバを追加したい場合は、以下のことを遵守して下さい。
 *  - 下記 SdkInternalTlsType に追加したいメンバ変数を uintptr_t 相当で追加する
 *  - ビルドおよび実行を確認し、develop ブランチへの Pull Request を作成する
 *  - レビューワに OS 担当者を入れ、OS 担当者の Approve をマージ条件に加える
 *
 *  この SDK 用 TLS 領域は各プラットフォーム環境下で uintptr_t × 16 個分です。
 *
 *  なお、この TLS 領域は nn::os::CreateThread() 時にゼロ初期化されることが
 *  保障されています。また、ここで定義した TLS に関しては、デストラクタ機能は
 *  使用できません。
 *
 */
struct  SdkInternalTlsType
{
    // ここに TLS 化したいメンバ変数を列挙する
    uintptr_t sfInlineContext;
};


static_assert(sizeof(SdkInternalTlsType) <= sizeof(uintptr_t) * SdkTlsSlotCountMax, "");
static_assert(sizeof(SdkInternalTlsType) == 1 /* 空定義の場合は 1 になる */ ||
              sizeof(SdkInternalTlsType) % sizeof(uintptr_t) == 0, "");
static_assert(std::is_pod<SdkInternalTlsType>::value, "");

//-----------------------------------------------------------------------------

}} // namespace nn::os

