﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッド情報に関する公開ヘッダファイル（SDK 開発者向け）
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkResult.h>
#include <nn/os/os_SdkThreadInfoTypes.h>
#include <nn/os/os_ThreadTypes.h>

namespace nn { namespace os {

//! @name CPU プロファイラ関連 API（SDK 開発者専用）
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   同一プロセッサで最後に動作したスレッドの情報を取得します
 *
 * @param[out] pOutThread            最後に動作したスレッド構造体のポインタを格納します
 * @param[out] pOutLastThreadContext 最後に動作したスレッドのコンテキストを格納します
 * @param[out] pOutFlag              最後に動作したスレッドの状態を示すフラグを格納します
 *
 * @handleresult{nn::os::ResultNoThread}
 * @handleresult{nn::os::ResultUnknownThread}
 *
 * @details
 *  同一プロセッサで最後に動作したスレッドの情報を取得します。
 *
 *  pOutThread には最後に動作したスレッドの ThreadType 構造体へのポインタが格納されます。
 *  pOutLastThreadContext には最後に動作したスレッドのコンテキストが格納されます。
 *  ただし、格納されるレジスタは unwind に必要最小限のもののみとなります。
 *  pOutFlag には最後に動作したスレッドの状態を示すフラグが格納されます。
 *
 */
Result GetLastThreadInfo(ThreadType** pOutThread, SdkLastThreadContext* pOutLastThreadContext, Bit32* pOutFlag) NN_NOEXCEPT;

//-----------------------------------------------------------------------------
/**
 * @brief   対象スレッドのスタック情報を取得します
 *
 * @param[out] pOutStackTop  スタックの先頭アドレスの格納先
 * @param[out] pOutStackSize スタックサイズの格納先
 * @param[in]  thread        スタック情報を取得したいスレッド
 *
 * @details
 *  指定したスレッドのスタックの先頭アドレスとサイズを pOutStackTop および pOutStackSize に格納します。
 *  指定したスレッドでファイバ実行中の場合にはファイバのスタック情報を格納します。
 *
 *  指定したスレッドの状態が Started 以外の場合、スタックの情報は全て NULL が返ります。
 *
 *  pOutStackTop または pOutStackSize に NULL を指定した場合、それらの情報は格納されません。
 *
 */
void GetThreadStackInfo(uintptr_t* pOutStackTop, size_t* pOutStackSize, const ThreadType* thread) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::os

