﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   コンパイラもしくは標準ライブラリ実装用の機能
 *
 * @details
 *  このヘッダファイルに含まれる API は全て RYNDA 向けであり、
 *  一般公開はされません。
 *
 */

#pragma once

#include <nn/nn_Common.h>

//! @name   デバッガへの通知を行なう機能
//! @{

//--------------------------------------------------------------------------
/**
 * @brief デバッガへの通知種別を示す列挙型
 *
 */
extern "C" enum nnosNotificationReason
{
    nnosNotificationReason_CppException,    //!< C++ 例外の発生
};


//--------------------------------------------------------------------------
/**
 * @brief デバッガへの通知を行ないます。
 *
 * @param[in] reason デバッガへの通知種別
 * @param[in] data   デバッガへ渡す補助データ
 * @param[in] size   デバッガへ渡す補助データのサイズ
 *
 * @details
 *  デバッガへ通知を行ないます。
 *
 *  詳細な仕様は svc::Break() に従いますが、
 *  reason は然るべき svc::BreakReason に置き換えられてデバッガへ通知されます。
 *  また、data および size はそのままデバッガへ通知されます。
 *
 *  デバッガがアタッチされていない場合は本 API は何も行いません。
 *  また、未知の reason が指定された場合も本 API は何も行いません。
 *
 *  本関数は C リンケージです。@n
 *  また、本関数はコンパイラが生成するコード、もしくは標準 C/C++ ランタイム
 *  ライブラリ内からのみ使用することを想定した関数です。一般公開はされません。
 *
 *  @platformbegin{Windows}
 *   - Windows 環境では本 API は何も行わずにリターンします。
 *
 *  @platformend
 *
 */
extern "C" void nnosNotifyDebugger(nnosNotificationReason reason, uintptr_t data, size_t size);

//! @}



//! @name   プログラムの実行を停止する機能
//! @{

//--------------------------------------------------------------------------
/**
 * @brief プログラムの実行停止種別を示す列挙型
 *
 */
extern "C" enum nnosBreakReason
{
    nnosBreakReason_Panic,          //!< 致命的な状態を検出
    nnosBreakReason_Assert,         //!< アサート違反を検出
};


//--------------------------------------------------------------------------
/**
 * @brief プログラムの実行を停止し、停止種別をデバッガへ通知します。
 *
 * @param[in] reason プログラムの実行停止種別
 * @param[in] data   デバッガへ渡す補助データ
 * @param[in] size   デバッガへ渡す補助データのサイズ
 *
 * @details
 *  プログラムの実行を停止し、停止種別をデバッガへ通知します。
 *  本関数からはリターンしません。
 *
 *  詳細な仕様は svc::Break() に従いますが、
 *  reason は然るべき svc::BreakReason に置き換えられてデバッガへ通知されます。
 *  また、data および size はそのままデバッガへ通知されます。
 *
 *  デバッガがアタッチされていない場合は本 API によってプログラムは終了します。
 *  また、未知の reason が指定された場合は関数の内部でアボートします。
 *
 *  本関数は C リンケージです。@n
 *  また、本関数はコンパイラが生成するコード、もしくは標準 C/C++ ランタイム
 *  ライブラリ内からのみ使用することを想定した関数です。一般公開はされません。
 *
 *  @platformbegin{Windows}
 *   - Windows 環境では全ての引数は無視され、内部で DebugBreak()
 *     関数を呼び出します。詳細は msdn のドキュメントを参照して下さい。
 *
 *  @platformend
 *
 */
extern "C" NN_NORETURN void nnosBreak(nnosBreakReason reason, uintptr_t data, size_t size);

//! @}

