﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   割込みイベントに関する型や定数の宣言
 */

#pragma once

#include <nn/os/os_Config.h>

#include <nn/nn_Common.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_Macro.h>
#include <nn/os/os_InterruptEventCommon.h>
#include <nn/os/detail/os_MacroImpl.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------

namespace nn { namespace os {

namespace detail {
    class  MultiWaitObjectList;
    class  InterruptEventImpl;
}

//--------------------------------------------------------------------------
/**
 * @brief   割込みイベントオブジェクトを定義する構造体です。
 *
 * @details
 * メンバー変数を直接変更することは禁止です。 @n
 * 次の関数を介して、本構造体を利用してください。
 *
 * - @ref InitializeInterruptEvent()
 * - @ref FinalizeInterruptEvent()
 * - @ref WaitInterruptEvent()
 * - @ref TryWaitInterruptEvent()
 * - @ref TimedWaitInterruptEvent()
 * - @ref ClearInterruptEvent()
 * - @ref InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, InterruptEventType* interruptEvent)
 *
 * 上記の関数を利用するには nn/os/os_InterruptEvent.h をインクルードする必要があります。
 *
 * nn::os::InterruptEvent ユーティリティクラスもあわせてご確認ください。
 */
struct  InterruptEventType
{
    //----------------------------------------------------------------------
    //  割込みイベントの状態を表す列挙型です。
    enum State
    {
        State_NotInitialized      = 0,
        State_Initialized         = 1,
    };

    // メンバ変数
    util::TypedStorage<detail::MultiWaitObjectList,NN_OS_DETAIL_CALC_OBJECT_SIZE(0,2),NN_ALIGNOF(void*)> _multiWaitObjectList;

    uint8_t             _clearMode;
    uint8_t             _state;

    // バインドした割込みオブジェクト
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    util::TypedStorage<detail::InterruptEventImpl,NN_OS_DETAIL_CALC_OBJECT_SIZE(0,2),NN_ALIGNOF(void*)> _impl;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    util::TypedStorage<detail::InterruptEventImpl,NN_OS_DETAIL_CALC_OBJECT_SIZE(8,0),NN_ALIGNOF(uint32_t)> _impl;
#endif
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(InterruptEventType);

}} // namespace nn::os

