﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッド関連の型に関する Horizon 環境用の型の宣言
 */

#pragma once

#include <nn/os/os_Config.h>

#include <nn/nn_Common.h>
#include <nn/os/os_Macro.h>
#include <nn/os/detail/os_MacroImpl.h>


namespace nn { namespace os {

#if defined( NN_BUILD_CONFIG_CPU_ARM_V7A ) || \
    defined( NN_BUILD_FOR_DOCUMENT_GENERATION )
//-----------------------------------------------------------------------------
/**
 * @brief   AArch32 動作環境でのスレッドコンテキスト情報を格納する構造体です。
 *
 * @details
 *  AArch32 動作環境において nn::os::ThreadContextInfo へと typedef されます。
 *
 */
struct ThreadContextInfoAarch32
{
    // プロセッサ汎用レジスタ
    union
    {
        struct
        {
            Bit32   r[16];      //!< r0～r15 の値が格納されます
        };
        struct
        {
            Bit32   _padding[9];
            Bit32   sb;         //!< r[ 9] と等価です
            Bit32   sl;         //!< r[10] と等価です
            Bit32   fp;         //!< r[11] と等価です
            Bit32   ip;         //!< r[12] と等価です
            Bit32   sp;         //!< r[13] と等価です
            Bit32   lr;         //!< r[14] と等価です
            Bit32   pc;         //!< r[15] と等価です
        };
    };
    Bit32   cpsr;               //!< cpsr の値が格納されます
    NN_PADDING4;

    // 浮動小数点レジスタ
    Bit64   fpuRegisters[32];   //!< d0～d31 の値が格納されます
    Bit32   fpscr;              //!< fpscr の値が格納されます
    Bit32   fpexc;              //!< fpexc の値が格納されます
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(ThreadContextInfoAarch32);

typedef ThreadContextInfoAarch32    ThreadContextInfo;
#endif

#if defined( NN_BUILD_CONFIG_CPU_ARM_V8A ) || \
    defined( NN_BUILD_FOR_DOCUMENT_GENERATION )
//-----------------------------------------------------------------------------
/**
 * @brief   AArch64 動作環境でのスレッドコンテキスト情報を格納する構造体です。
 *
 * @details
 *  AArch64 動作環境において nn::os::ThreadContextInfo へと typedef されます。
 *
 *  NX32 および NX64 では常にこちらの構造体が使用され、
 *  NX32 の場合には、sp は r[13]、lr は r[14] から取得できます。
 *
 */
struct ThreadContextInfoAarch64
{
    // プロセッサ汎用レジスタ
    union
    {
        struct
        {
            Bit64   r[31];      //!< r0～r30 の値が格納されます
        };
        struct
        {
            Bit64   _padding[29];
            Bit64   fp;         //!< r29 と等価です
            Bit64   lr;         //!< r30 と等価です
        };
    };
    Bit64   sp;                 //!< sp の値が格納されます
    Bit64   pc;                 //!< pc の値が格納されます
    Bit32   pstate;             //!< pstate の値が格納されます
    NN_PADDING4;

    // 浮動小数点レジスタ
    Bit128  v[32];              //!< v0～v31 の値が格納されます
    Bit32   fpcr;               //!< fpcr の値が格納されます
    Bit32   fpsr;               //!< fpsr の値が格納されます
    NN_PADDING8;
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(ThreadContextInfoAarch64);

typedef ThreadContextInfoAarch64    ThreadContextInfo;
#endif


NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(ThreadContextInfo);

//-----------------------------------------------------------------------------

}} // namespace nn::os

