﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/net/osl/osl_CLibraryUtil.h>
#include <nn/svc/svc_MemoryConfigSelect.h>

// TORIAEZU: ここにおく
#define NN_LOG_WARN(...) NN_SDK_LOG(__VA_ARGS__)

enum
{
    NN_NET_MAC_ADDRESS_SIZE = 6,

    NN_NET_OSL_PAGE_SIZE = NN_SVC_MEMORY_PAGE_SIZE,

    /* Use pagesize(=4096) / n to avoid crossing page-boundary for DMA. */
    NN_NET_OSL_MBUF_UNIT_SIZE = (NN_NET_OSL_PAGE_SIZE / 2),
    NN_NET_OSL_MBUF_HEADER_SIZE = sizeof(void *) * 3 + sizeof(uint16_t) * 6 + sizeof(int16_t) * 2 + sizeof(uint8_t) * NN_NET_MAC_ADDRESS_SIZE * 2,
    NN_NET_OSL_MBUF_DATA_SIZE = ((NN_NET_OSL_MBUF_UNIT_SIZE) - (NN_NET_OSL_MBUF_HEADER_SIZE)),
    NN_NET_OSL_MBUF_DATA_MIN_SIZE = ((NN_NET_OSL_MBUF_UNIT_SIZE) - (NN_NET_OSL_MBUF_HEADER_SIZE)),

    // TORIAEZU:
    NN_NET_OSL_MBUF_PACKET_MAX_SIZE = (32 * 1024),

    NN_NET_OSL_M_DONTWAIT = 0,
    NN_NET_OSL_M_WAIT = 1,
    NN_NET_OSL_M_FREELIST = 0x8000,              /**< mbuf is in free list. (for debug */

    NN_NET_OSL_M_BCAST = 0x0200,              /**< ブロードキャスト */
    NN_NET_OSL_M_MCAST = 0x0400,              /**< マルチキャスト */
    NN_NET_OSL_M_LOOPBACK = 0x0800,              /**< ループバック */

    NN_NET_OSL_M_COPYALL = 1000000000,

    NN_NET_OSL_MT_DATA = 1                   /**< Data frames */
};

struct nnnetOslMbuf;
typedef struct nnnetOslMbuf nnnetOslMbuf;

/* mbuf data structure */
typedef struct nnnetOslMbuf
{
    nnnetOslMbuf*       m_next;                         /**< Next cluster */
    nnnetOslMbuf*       m_prev;                         /**< Previous cluster */
    nnnetOslMbuf*       m_nextpkt;                      /**< General purpose link */
    int16_t             m_len;                          /**< Length of stored data */
    uint16_t            m_capacity;
    int16_t             m_top;                          /**< Top offset of stored data */
    uint16_t            m_flags;                        /**< NN_NET_OSL_M_BCAST、NN_NET_OSL_M_MCAST、NN_NET_OSL_M_LOOPBACK の OR */
    uint16_t            m_protocol;                     /**< Protocol id, for socket and related module. */
    uint16_t            m_owner;                        /* Owner module. NN_NET_OSL_M_OWN_XXXX */
    uint16_t            m_name;                         /**< Category name tag. NN_NET_OSL_M_CAT_XXXX */
    uint16_t            m_pool;
    uint8_t             m_dst[NN_NET_MAC_ADDRESS_SIZE]; /**< Destination MAC address. */
    uint8_t             m_src[NN_NET_MAC_ADDRESS_SIZE]; /**< Source MAC address */
    uint8_t             m_data[NN_NET_OSL_MBUF_DATA_MIN_SIZE]; /**< Data buffer */
} nnnetOslMbuf;

/* mbuf in free pool */
typedef struct nnnetOslMbufFree
{
    int32_t                 m_nextFree;                     /**< Next free cluster index */
} nnnetOslMbufFree;

/* size validation */
NN_STATIC_ASSERT(sizeof(nnnetOslMbuf) == NN_NET_OSL_MBUF_UNIT_SIZE);
NN_STATIC_ASSERT(offsetof(nnnetOslMbuf, m_data) == NN_NET_OSL_MBUF_HEADER_SIZE);


// 以下、C 用宣言
NN_EXTERN_C nnnetOslMbuf*   nnnetOslMbuf_getm       (uint32_t name, nnnetOslMbuf *orig, int32_t len, int32_t how, uint8_t type);
NN_EXTERN_C void            nnnetOslMbuf_freem      (nnnetOslMbuf* pMbuf_);
NN_EXTERN_C int32_t         nnnetOslMbuf_adj        (nnnetOslMbuf* pMbuf_, int32_t len);
NN_EXTERN_C int32_t         nnnetOslMbuf_append     (nnnetOslMbuf* pMbuf_, int32_t len, const uint8_t *cp);
NN_EXTERN_C nnnetOslMbuf*   nnnetOslMbuf_prepend_a  (nnnetOslMbuf* pMbuf_, int32_t len, int how);
NN_EXTERN_C nnnetOslMbuf*   nnnetOslMbuf_pullup     (nnnetOslMbuf* pMbuf_, int32_t len);
NN_EXTERN_C nnnetOslMbuf*   nnnetOslMbuf_dup        (nnnetOslMbuf* pMbuf_, int how);
NN_EXTERN_C int32_t         nnnetOslMbuf_copydata   (const nnnetOslMbuf* pMbuf_, int32_t offset, int32_t len, uint8_t *buf);
NN_EXTERN_C int32_t         nnnetOslMbuf_copyback   (nnnetOslMbuf* pMbuf_, int32_t offset, int32_t len, const uint8_t *buf);
NN_EXTERN_C int32_t         nnnetOslMbuf_cat        (nnnetOslMbuf* pMbuf_, nnnetOslMbuf *n);
NN_EXTERN_C nnnetOslMbuf*   nnnetOslMbuf_split      (nnnetOslMbuf* pMbuf_, int32_t len, int how);
NN_EXTERN_C int32_t         nnnetOslMbuf_length     (nnnetOslMbuf* pMbuf_, nnnetOslMbuf **last);
NN_EXTERN_C int32_t         nnnetOslMbuf_apply      (nnnetOslMbuf* pMbuf_, int32_t offset, int32_t len, int32_t (* f)(void* arg, void* data, int32_t len), void* arg);
NN_EXTERN_C void            nnnetOslMbuf_dump       (nnnetOslMbuf* pMbuf_);
NN_EXTERN_C bool            nnnetOslMbuf_align      (nnnetOslMbuf* pMbuf_, uint8_t align, int8_t offset);

NN_C_INLINE int32_t         nnnetOslMbuf_leadingspace(const nnnetOslMbuf* pMbuf_)
{
    return pMbuf_->m_top;
}

NN_C_INLINE int32_t      nnnetOslMbuf_trailingspace(const nnnetOslMbuf* pMbuf_)
{
    return pMbuf_->m_capacity - (pMbuf_->m_top + pMbuf_->m_len);
}

NN_C_INLINE nnnetOslMbuf* nnnetOslMbuf_prepend(nnnetOslMbuf* pMbuf_, int32_t len, int how)
{
    NN_SDK_REQUIRES(len >= 0);
    if (pMbuf_->m_top >= len)
    {
        pMbuf_->m_top -= static_cast<int16_t>(len);
        pMbuf_->m_len += static_cast<int16_t>(len);
        return pMbuf_;
    }
    else
    {
        return nnnetOslMbuf_prepend_a(pMbuf_, len, how);
    }
}

NN_C_INLINE void*     nnnetOslMbuf_tod        (nnnetOslMbuf* pMbuf_)
{
    //todo: Check consitency here.
    return &(pMbuf_->m_data[pMbuf_->m_top]);
}

NN_C_INLINE void      nnnetOslMbuf_setUserPointer(nnnetOslMbuf* pMbuf_, void* p)
{
    pMbuf_->m_nextpkt = (nnnetOslMbuf*)p;
}

NN_C_INLINE void*     nnnetOslMbuf_getUserPointer(nnnetOslMbuf* pMbuf_)
{
    return pMbuf_->m_nextpkt;
}

NN_C_INLINE bool     nnnetOslMbuf_expand(nnnetOslMbuf* pMbuf_, int32_t len, int how)
{
    NN_UNUSED(how);
    if (pMbuf_->m_top + pMbuf_->m_len + len > pMbuf_->m_capacity)
    {
        NN_SDK_LOG("nnnetOslMbuf_expand: out of capacity.(len = %d)\n", len);
        return false;
    }

    pMbuf_->m_len += static_cast<int16_t>(len);
    return true;
}

NN_C_INLINE void     nnnetOslMbuf_reserve(nnnetOslMbuf* pMbuf_, int32_t len)
{
    NN_SDK_ASSERT(pMbuf_->m_len == 0 && pMbuf_->m_top + len <= pMbuf_->m_capacity);
    pMbuf_->m_top += static_cast<int16_t>(len);
}

#ifdef __cplusplus

#include <new>

namespace nn { namespace net { namespace osl {

class MbufPool;

class Mbuf : public nnnetOslMbuf
{
public:
    typedef int32_t FreeIndex;
    static const size_t HEADER_SIZE = NN_NET_OSL_MBUF_HEADER_SIZE;

    void Initialize(size_t unitSize, uint16_t poolId)
    {
        memset(static_cast<nnnetOslMbuf*>(this), 0x00, HEADER_SIZE);
        m_capacity = static_cast<uint16_t>(unitSize - HEADER_SIZE);
        m_pool = poolId;
    }

    explicit Mbuf(FreeIndex nextFree)
    {
        Initialize(0, 0xffff);
        reinterpret_cast<nnnetOslMbufFree*>(this)->m_nextFree = nextFree;
    }

    FreeIndex* GetNextFreeIndexPtr()
    {
        return &(reinterpret_cast<nnnetOslMbufFree*>(this)->m_nextFree);
    }

    void SetNextFreeIndex(FreeIndex freeIndex)
    {
        reinterpret_cast<nnnetOslMbufFree*>(this)->m_nextFree = freeIndex;
    }

    FreeIndex GetNextFreeIndex() const
    {
        return reinterpret_cast<const nnnetOslMbufFree*>(this)->m_nextFree;
    }

    operator nnnetOslMbuf*()
    {
        return reinterpret_cast<nnnetOslMbuf*>(this);
    }

    Mbuf* GetNext()
    {
        return reinterpret_cast<Mbuf*>(m_next);
    }

    const Mbuf* GetNext() const
    {
        return reinterpret_cast<const Mbuf*>(m_next);
    }

    Mbuf* GetPrevious()
    {
        return reinterpret_cast<Mbuf*>(m_prev);
    }

    const Mbuf* GetPrevious() const
    {
        return reinterpret_cast<const Mbuf*>(m_prev);
    }

    void SetNext(Mbuf* pMbuf)
    {
        m_next = reinterpret_cast<nnnetOslMbuf*>(pMbuf);
    }

    void SetPrevious(Mbuf* pMbuf)
    {
        m_prev = reinterpret_cast<nnnetOslMbuf*>(pMbuf);
    }

    bool IsLinking() const
    {
        return m_prev != NULL || m_next != NULL;
    }

    Mbuf* Unlink()
    {
        Mbuf* pNext = GetNext();
        if (pNext)
        {
            SetNext(NULL);
            pNext->SetPrevious(NULL);
            return pNext;
        }
        else
        {
            return NULL;
        }
    }

    void CopyFrom(const Mbuf* pSource);

    Mbuf* GetTail();
    const Mbuf* GetHead() const;

    int16_t SetLengthTrimed(size_t len)
    {
        m_len = (len > m_capacity) ? m_capacity : static_cast<int16_t>(len);
        return m_len;
    }

    int16_t GetLength() const
    {
        NN_SDK_ASSERT(m_len <= m_capacity);
        return m_len;
    }

    int16_t SetLength(int16_t len)
    {
        NN_SDK_ASSERT(len <= m_capacity);
        return m_len = static_cast<int16_t>(len);
    }

    int16_t& Length()
    {
        return m_len;
    }

    uint16_t GetFlags() const
    {
        return m_flags;
    }

    void SetFlags(uint16_t flags)
    {
        m_flags = flags;
    }

    void SetName(uint16_t name)
    {
        m_name = name;
    }

    uint16_t GetName() const
    {
        return m_name;
    }

    void SetProtocol(uint16_t protocol)
    {
        m_protocol = protocol;
    }

    uint16_t GetProtocol() const
    {
        return m_protocol;
    }

    void SetTop(int32_t top)
    {
        NN_SDK_ASSERT(top < m_capacity);
        m_top = static_cast<int16_t>(top);
    }

    size_t GetWritableLength() const
    {
        NN_SDK_ASSERT(m_top < m_capacity);
        return m_capacity - m_top;
    }

    int32_t GetTop() const
    {
        return m_top;
    }

    int32_t GetLeadingSpace() const
    {
        return m_top;
    }

    int32_t GetTrailingSpace() const
    {
        return m_capacity - (m_top + m_len);
    }

    uint8_t* GetTopPtr(int32_t offset = 0)
    {
        NN_SDK_ASSERT(m_top + offset < m_capacity);
        return &m_data[m_top + offset];
    }

    const uint8_t* GetTopPtr(int32_t offset = 0) const
    {
        NN_SDK_ASSERT(m_top + offset < m_capacity);
        return &m_data[m_top + offset];
    }

    bool IsDataAligned(uint8_t align, int8_t offset = 0) const
    {
        return IsAligned(reinterpret_cast<uintptr_t>(GetTopPtr()), align, offset);
    }

    int16_t GetCapacity() const
    {
        return m_capacity;
    }

    int16_t GetUnitSize() const
    {
        return m_capacity + HEADER_SIZE;
    }

    uint16_t GetPoolId() const
    {
        return m_pool;
    }

    static bool IsAligned(uintptr_t address, uint8_t align, int8_t offset)
    {
        NN_SDK_ASSERT(align != 0 && (align & (align - 1)) == 0);
        return ((address + offset) & (align - 1)) == 0;
    }

    // TORIAEZU:
    int32_t GetTotalLength() const
    {
        const nnnetOslMbuf* p = reinterpret_cast<const nnnetOslMbuf*>(this);
        return nnnetOslMbuf_length(const_cast<nnnetOslMbuf*>(p), NULL);
    }

    bool CopyChainTo(uint8_t* pDst, int32_t offset = 0, int32_t len = NN_NET_OSL_M_COPYALL) const
    {
        return nnnetOslMbuf_copydata(reinterpret_cast<const nnnetOslMbuf*>(this), offset, len, pDst) == 0;
    }

    // C 版の速度も重視する
    bool AlignData(uint8_t align, int8_t offset)
    {
        return nnnetOslMbuf_align(reinterpret_cast<nnnetOslMbuf*>(this), align, offset);
    }

    void Link(Mbuf* pNext)
    {
        SetNext(pNext);
        pNext->SetPrevious(this);
    }

    void Dump() const;
    void DumpAll() const;
    void DumpChain() const;

protected:
};

}}}

#endif // __cplusplus

