﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// TORIAEZU: LightAlarm を nos の内部に持ちます

/*! :private
    @file
    @brief      LightAlarm に関する API の宣言

    :include nn/os.h
*/

#include <nn/nn_TimeSpan.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_EventTypes.h>
#include <nn/util/util_IntrusiveList.h>

/*! :private
    @brief アラームを扱う為のクラスです。

    アラームは、指定時間経過後にハンドラを呼び出します。
    この時にハンドラの第1引数はアラームをセットした際の引数、第2引数には false が入って呼ばれます。

    このクラスを使用するには、@ref nn::net::nos::InitializeLightAlarmSystem を先に呼ぶ必要があります。
    アラームハンドラの実行は、複数のインスタンス間で共有されている単一のスレッドで行われます。
    そのため、ハンドラの実行が終わるまで、次のアラームが発現することはありません。
    もし、ハンドラ実行中に発現時間がきた場合は、ハンドラ停止直後に発現します。

    このクラスの使用を終えた際には、@ref nn::os::FinalizeLightAlarmSystem を実行してください。
    クラスで使用していたスレッドとタイマーの終了処理とアラームを管理している待ち行列のクリアが行われ、
    システムのリソースが開放されます。

    アラームにはワンショットアラームのみをサポートしています。
    ワンショットアラームは @ref SetOneShot でセットし、一定時間後にハンドラが呼ばれます。

    ハンドラが動作する前にアラームを停止したい場合は、@ref Cancel を呼び出します。
    ハンドラが動作し始めていた場合は、@ref Cancel を呼んでも、ハンドラの動作が中断されることはありません。
    ハンドラが動作し始める前の場合は、@ref Cancel が呼ばれると、アラームのハンドラが 第2引数に true
    を入れて呼び出されます。

    アラームオブジェクトを複数のスレッドから同時に操作してはいけません。
*/

namespace nn { namespace net { namespace osl {

/*!  :private
    @brief アラームで実行されるハンドラを表す型です。
    @param[in]    param       ハンドラの引数
    @param[in]    canceled   @ref Cancel が呼ばれたハンドラであることを示すフラグ
*/
typedef void (*LightAlarmHandler)(void* param, bool canceled);

void InitializeLightAlarmSystem( void* stackTop, size_t stackSize, int priority );

/*! :private
    @brief アラームシステムを終了します。
*/
void FinalizeLightAlarmSystem();

// 待ち行列に登録するLightAlarmのデータを保存するクラスです。
struct LightAlarm
{
    LightAlarmHandler handler;          //アラームハンドラ
    nn::os::EventType finishedEvent;        //発現完了イベント
    void* parameter;                    //アラームハンドラの引数
    bool  canceled;                     //キャンセルフラグ
    bool  isPeriodic;                   //周期アラームフラグ
    nn::os::Tick  fire;                 //発現時刻 (Tick)
    nn::os::Tick  period;               //発現間隔 (Tick)
    nn::util::IntrusiveListNode listNode;
};

/*! :private
    @brief        アラームオブジェクトの初期化をします。
*/
void InitializeLightAlarm(LightAlarm* p);

/*! :private
    @brief        アラームの終了処理をします。
                  アラームがセット中の場合にこの関数を呼び出されると、
                  内部的に @ref LightAlarmCancel が呼び出されます。
*/
void FinalizeLightAlarm(LightAlarm* p);

/*! :private
    @brief        ワンショットアラームを設定します。
                  time 後にハンドラを一回だけ呼び出します。

    @param[in]    time        ハンドラを呼び出すまでの時間(ns)
    @param[in]    handler     アラームのハンドラ
    @param[in]    param       ハンドラに与えるパラメータ

    @return       無し。
*/
void LightAlarmSetOneShot(LightAlarm* p, nn::TimeSpan time, LightAlarmHandler handler, void* param);

/*! :private
    @brief        周期アラームを設定します。
                  initial 後にハンドラを一回だけ呼び出し、
                  その後 interval 周期でハンドラを呼び出し続けます。

    @param[in]    initial     一度目にハンドラを呼び出すまでの時間
    @param[in]    interval    二度目以降にハンドラを呼び出す間隔
    @param[in]    handler     アラームのハンドラ
    @param[in]    param       ハンドラに与えるパラメータ

    @return       無し。
*/
void LightAlarmSetPeriodic(LightAlarm* p, nn::TimeSpan initial, nn::TimeSpan interval, LightAlarmHandler handler, void* param);

/*! :private
    @brief        設定済みのアラームを取り消します。

                  ハンドラが動作し始めていた場合は、@ref LightAlarmCancel を呼んでも、ハンドラの動作が中断されることはありません。
                  ハンドラが動作し始める前の場合は、@ref LightAlarmCancel が呼ばれると、アラームのハンドラが 第2引数に true
                  を入れて呼び出されます。

    @return       無し。

*/
void LightAlarmCancel(LightAlarm* p);

/*! :private
    @brief        アラームをすぐにセットできる状態であるかどうかを取得します。

    @attention    アラームは、発現した後ハンドラが呼び出される直前に、すぐにセットできる状態になります。

    @return       アラームがセットできる状態であれば true を返し、そうでなければ false を返します。
*/
bool LightAlarmCanSet(LightAlarm* p);

/*! :private
    @brief      アラームをすぐにセットできる状態になるまで待機します。
*/
void LightAlarmWait(LightAlarm* p);

}}}

