﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * NNETモジュールの外部参照用定義
 */

#ifndef NN_NET_COMPATIBLE_NNET_NNET_H_
#define NN_NET_COMPATIBLE_NNET_NNET_H_

#include <nn/net/nos.h>            /* for NOS*** */

#ifdef  __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------------------------*
 * Definitions
 *---------------------------------------------------------------------------*/

#define NNET_ERROR_BASE                          0xfff70000
#define NNET_ERROR_CODE(code)                    ((s32)(NNET_ERROR_BASE|(code)))

/**
 * NNETモジュールのエラーコード
 */
enum {
    NNET_ERR_PARAM        = NNET_ERROR_CODE(1),     /**< パラメータエラー */
    NNET_ERR_EXIST        = NNET_ERROR_CODE(2),     /**< 既に作成済み */
    NNET_ERR_NOMEM        = NNET_ERROR_CODE(3),     /**< メモリ不足 */
    NNET_ERR_RECEIVE      = NNET_ERROR_CODE(4),     /**< 受信失敗 */
    NNET_ERR_SEND         = NNET_ERROR_CODE(5),     /**< 送信失敗 */
    NNET_ERR_RESOURCE     = NNET_ERROR_CODE(6),     /**< OSリソース確保失敗 */
    NNET_ERR_CANCEL       = NNET_ERROR_CODE(7),     /**< キャンセル */
    NNET_ERR_BUSY         = NNET_ERROR_CODE(8),     /**< バッファフル */
    NNET_ERR_STATE        = NNET_ERROR_CODE(9),     /**< ステート不正 */
    NNET_ERR_NONE         = 0                       /**< 正常終了 */
};

/*---------------------------------------------------------------------------*
 * Types/Declarations
 *---------------------------------------------------------------------------*/
#define NNET_TMO_WAITFOREVER    (-1L)   /* 無限待ち */

#define NNET_ETH_ALEN       6       /* MACアドレス長 */
#define NNET_IP_ALEN        4       /* IPv4アドレス長 */

#define NNET_MIN(a, b)   (((a) < (b)) ? (a) : (b))    /**< aがbより小さい場合はaを、そうでなければbを返します。 */
#define NNET_MAX(a, b)   (((a) > (b)) ? (a) : (b))    /**< aがbより大きい場合はaを、そうでなければbを返します。 */


/**
 *  アロケータ
 */
typedef struct NNETAllocator {
    void*    (* alloc) (u32 name, u32 size, u32 align); /**< メモリ確保関数 */
    void     (* free) (u32 name, void* ptr);            /**< メモリ解放関数 */
} NNETAllocator;

/**
 *  抽象化インスタンス
 */
typedef struct NNETInstance {
    struct NNETInstance *lower_ins; /**< 下位インスタンス。0はこのレイヤが最下位であることを示します。 */
    s32 (* startup) (struct NNETInstance *ins, s32 timeout);    /**< startup関数 */
    s32 (* cleanup) (struct NNETInstance *ins, s32 timeout);    /**< cleanup関数 */
    s32 (* abort)   (struct NNETInstance *ins);                 /**< abort関数 */
} NNETInstance;

/**
 *  下位レイヤで得た設定をソケットレイヤに渡すための構造体
 */
typedef struct NNETConfig {
    u8 addr[NNET_IP_ALEN];      /**< IPアドレス */
    u8 gateway[NNET_IP_ALEN];   /**< デフォルトゲートウエイアドレス */
    u8 dns1[NNET_IP_ALEN];      /**< プライマリDNSサーバアドレス */
    u8 dns2[NNET_IP_ALEN];      /**< セカンダリDNSサーバアドレス */
    s32 mtu;                    /**< MTU */
} NNETConfig;


#define NNET_L2TYPE_ETHERNET    1   /**< IPパケット送信前にARPなどアドレッシング処理を行い、Ethernetヘッダで送信するタイプのレイヤです。 */
#define NNET_L2TYPE_PPP         2   /**< そのままIPパケットを送信するタイプのレイヤです。 */

/**
 *  ソケットレイヤとその下位レイヤのためのAPI
 */
typedef struct NNETEthernetAPI {
    NNETInstance *ins;
    s32 (* eth_send) (NNETInstance *ins, u8 *dstMAC, NOSMessageBuf *mbuf);           /**< 送信関数 */
    s32 (* eth_receive) (NNETInstance *ins, uintptr_t receive_id, NOSMessageBuf **mbuf);   /**< 受信関数 */
    s32 (* eth_setupper) (NNETInstance *ins, NNETInstance *upper_ins, uintptr_t *receive_id, u16 list[]); /**< 上位インスタンスの情報を下位にセットするための関数 */
    s32 (* eth_clearupper) (NNETInstance *ins, NNETInstance *upper_ins);             /**< 上位インスタンスの情報を下位から削除するための関数 */
    u8  eth_type;                                                                    /**< NNET_L2TYPE_XXXX */
    s32 (* eth_getmacaddr) (NNETInstance *ins, u8 *mac);                             /**< 自MACアドレス取得関数 */
    s32 (* eth_setmulticast) (NNETInstance *ins, u8 *mac);                           /**< 受信マルチキャストアドレス登録 */
    s32 (* eth_clearmulticast) (NNETInstance *ins, u8 *mac);                         /**< 受信マルチキャストアドレス削除 */
    s32 (* eth_cancel_receive) (NNETInstance *ins, uintptr_t receive_id);                  /**< 下位の受信停止関数 */
    s32 (* eth_getconfig) (NNETInstance *ins, NNETConfig *config);                   /**< 下位の設定取得関数 */
} NNETEthernetAPI;


/**
 *  PPPレイヤとその下位レイヤのためのAPI
 */
typedef struct NNETPPPAPI {
    NNETInstance *ins;
    s32 (* ppp_send) (NNETInstance *ins, NOSMessageBuf *mbuf);                       /**< 送信関数 */
    s32 (* ppp_receive) (NNETInstance *ins, NOSMessageBuf **mbuf);                   /**< 受信関数 */
    s32 (* ppp_setupper) (NNETInstance *ins, NNETInstance *upper_ins);               /**< 上位インスタンスの情報を下位にセットするための関数 */
    s32 (* ppp_clearupper) (NNETInstance *ins, NNETInstance *upper_ins);             /**< 上位インスタンスの情報を下位から削除するための関数 */
    s32 (* ppp_cancel_receive) (NNETInstance *ins);                                  /**< 下位の受信停止関数 */
} NNETPPPAPI;

/*---------------------------------------------------------------------------*
 * Function Prototypes
 *---------------------------------------------------------------------------*/
extern NNETInstance *NNET_GetBottomIF(const NNETInstance *upper_ins);
extern NNETInstance *NNET_GetUpperIFByTop(const NNETInstance *top_ins, const NNETInstance *low_ins);
extern NNETInstance *NNET_GetLowerIF(NNETInstance *upper_ins);
extern s32 NNET_StartupIF(NNETInstance *ins, s32 timeout);
extern s32 NNET_CleanupIF(NNETInstance *ins, s32 timeout);
extern s32 NNET_AbortIF(NNETInstance *ins);

#ifdef NNET_BIG_ENDIAN
#define NNET_Swp2( _d )  d
#else
#define NNET_Swp2( _d )  ((u16)((((_d)>>8)&0x00FF)|(((_d)<<8)&0xFF00)))
#endif
extern u16 NNET_Get2( const void* ptr );
extern u16 NNET_Set2( void* ptr, u16 dat );
extern u16 NNET_And2( void* ptr, u16 dat );
extern u16 NNET_Orr2( void* ptr, u16 dat );
extern u16 NNET_Inc2( void* ptr );
extern u16 NNET_Dec2( void* ptr );
extern u16 NNET_Sub2( void* ptr, u16 dat );

extern u32 NNET_Get4( const void* ptr );
extern u32 NNET_Get4( const void* ptr );
extern u32 NNET_Set4( void* ptr, u32 dat );
extern u32 NNET_Inc4( void* ptr );
extern u32 NNET_Add4( void* ptr, u32 dat );

#include <nn/net/nnet/nnet_ring.h>


/*
    デバッグ用の定義
*/
#include <nn/net/ndebug.h>

#ifdef NDEBUG_ENABLE

#define NNET_DBGASSERT(cond)  NDEBUG_Assert(cond)

#else  /* NDEBUG_ENABLE */

#define NNET_DBGASSERT(cond)

#endif  /* NDEBUG_ENABLE */


#ifdef NDEBUG_PRINT_ENABLE

extern u32 ndebug_mask_nnet;

#define NNET_DBGREPORT(level, log)         \
    do {                                  \
        if( level & ndebug_mask_nnet ) {   \
            NDEBUG_Printf ("[NNET] ");    \
            NDEBUG_Printf log;            \
            NDEBUG_Printf ("\n");         \
        }                                 \
    } while (0)

#define NNET_DBGFUNC(level, func)          \
    do {                                  \
        if( level & ndebug_mask_nnet ) {   \
            func;                         \
        }                                 \
    } while (0)

#else /* NDEBUG_PRINT_ENABLE */

#define NNET_DBGREPORT(level, log)
#define NNET_DBGFUNC(level, func)

#endif /* NDEBUG_PRINT_ENABLE */


#ifdef  __cplusplus
}
#endif

/* NN_NET_COMPATIBLE_NNET_NNET_H_ */
#endif
