﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#ifdef __cplusplus

#include <nn/nn_Common.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_SharedMemoryTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/mbuf/mbuf_Definitions.h>

namespace nn { namespace mbuf
{
    /**
     * @brief       Initialize に指定できる Type の最大数です。
     */
    const int TypeCountMax = 16;

    /**
     * @brief       mbuf プールの用途を定義します。
     */
    struct Type
    {
        //! mbuf ライブラリが使用するイベントです。事前に自動クリアモードで初期化してください。
        nn::os::SystemEventType* pEvent;
    };

    /**
     * @brief         mbuf プールに関連する設定値を保持する構造体です。
     */
    struct Config
    {
        //! mbuf プールで管理するクラスタのバイトサイズです。
        uint32_t unitSize;

        //! mbuf プールで管理するクラスタの数です。
        int32_t count;

        //! mbuf プールの用途です。Type のインデックスに対応させてください。
        int32_t type;
    };

    /**
     * @brief         mbuf ライブラリの初期化に必要なバッファサイズを取得します。
     *
     * @param[in]     configs       mbuf プールの設定です。
     * @param[in]     poolCount     生成する mbuf プールの数です。
     *
     * @return        mbuf ライブラリの初期化に必要なバッファサイズです。
     *
     * @pre
     *                - configs != nullptr
     *                - 1 <= poolCount && poolCount <= PoolCountMax
     */
    size_t GetRequiredBufferSize(const Config configs[], int poolCount) NN_NOEXCEPT;

    /**
     * @brief         mbuf ライブラリを初期化します。
     *
     * @param[in]     types         mbuf の用途です。
     * @param[in]     typeCount     types の要素数です。
     * @param[in]     configs       mbuf プールの設定です。
     * @param[in]     poolCount     生成する mbuf プールの数です。
     * @param[in]     buffer        mbuf ライブラリに提供するバッファです。
     * @param[in]     bufferSize    mbuf ライブラリに提供するバッファのサイズです。
     *
     * @return        mbuf ライブラリの初期化に必要なバッファサイズです。
     *
     * @pre
     *                - configs != nullptr
     *                - 1 <= poolCount && poolCount <= PoolCountMax
     *                - buffer != nullptr
     *                - GetRequiredBufferSize(configs, poolCount) <= bufferSize
     *                - pEventType != nullptr
     *                - mbuf ライブラリが初期化されていないこと
     *
     * @details       新しく mbuf プールを作成する場合、
     *                この関数を使用して mbuf ライブラリを初期化ください。
     */
    void Initialize(const Type types[], int typeCount,
                    const Config configs[], int poolCount,
                    void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief         mbuf ライブラリを初期化します。
     *
     * @param[in]     types         mbuf の用途です。
     * @param[in]     typeCount     types の要素数です。
     * @param[in]     buffer        mbuf ライブラリに提供するバッファです。
     * @param[in]     bufferSize    mbuf ライブラリに提供するバッファのサイズです。
     *
     * @return        mbuf ライブラリの初期化に必要なバッファサイズです。
     *
     * @pre
     *                - buffer != nullptr
     *                - pEventType != nullptr
     *                - mbuf ライブラリが初期化されていないこと
     *
     * @details       既に他のプロセスによって mbuf プールが構築されている場合、
     *                この関数を使用して mbuf ライブラリを初期化してください。
     */
    void Initialize(const Type types[], int typeCount,
                    void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief         mbuf ライブラリを解放します。
     *
     * @pre
     *                - mbuf ライブラリが初期化されていること
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief         mbuf chain をインデックス化します。
     *
     * @param[in]     pMbuf         対象の mbuf チェインです。
     *
     * @return        mbuf chain のインデックスです。
     *
     * @pre
     *                - pMbuf != nullptr
     *                - pMbuf が mbuf チェインの先頭であること
     *                - mbuf ライブラリが初期化されていること
     *
     * @details       プロセス間で mbuf を共有するために共有メモリを使用している場合であっても、
     *                それぞれのプロセスで共有メモリがマップされているアドレスは異なります。
     *                そのため、 mbuf チェインのポインタを IPC で直接やりとりしても、
     *                正しく mbuf チェインを受け渡しできません。
     *
     *                IPC で mbuf チェインを受け渡すには、事前にこの関数で、
     *                mbuf チェインをプロセス間で共有可能なインデックスに変換してください。
     *                インデックス化された mbuf チェインは一時的に利用できなくなりますが、
     *                @ref ConvertFromIndex 関数で復元することができます。
     */
    int ConvertToIndex(Mbuf* pMbuf) NN_NOEXCEPT;

    /**
     * @brief         インデックスから mbuf chain を復元します。
     *
     * @param[in]     int           mbuf チェインのインデックスです。
     *
     * @return        インデックスから復元された mbuf chain です。
     *
     * @pre
     *                - mbuf ライブラリが初期化されていること
     *
     * @details       @ref ConvertToIndex() 関数でインデックス化された mbuf chain を復元します。
     *                詳細については @ref ConvertToIndex() 関数のリファレンスを参照してください。
     */
    Mbuf* ConvertFromIndex(int index) NN_NOEXCEPT;

}} // end of namespace nn::mbuf

#endif
