﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/detail/mbuf_TypesPrivate.h>

namespace nn { namespace mbuf { namespace detail
{
    /**
     * @brief       mbuf プールのメタ情報です。
     */
    struct MbufPoolMetaInfo
    {
        /**
         * @brief       メタ情報を初期化します。
         * @param[in]   unit        mbuf プールで管理するクラスタのバイトサイズです。
         * @param[in]   count       mbuf プールで管理するクラスタの数です。
         * @param[in]   usage       mbuf プールの用途です。
         */
        void Clear(size_t unit, int count, int usage) NN_NOEXCEPT
        {
            this->magic = MAGIC_NUMBER;
            this->unitSize = static_cast<uint32_t>(unit);
            this->maxCount = count;
            this->type = usage;
            this->usedCount = 0;
            this->taggedFreeIndex = 0;
        }

        /**
         * @brief       メタ情報の有効性を検証します。
         * @return      メタ情報が有効であれば true です。
         */
        bool IsValid() const NN_NOEXCEPT
        {
            return magic == MAGIC_NUMBER &&
                   MbufUnitSizeMin <= unitSize && unitSize <= MbufUnitSizeMax &&
                   PoolCapacityMin <= maxCount && maxCount <= PoolCapacityMax;
        }

        /**
         * @brief       mbuf クラスタの使用数をインクリメントします。
         * @return      変更前の値です。
         */
        int IncrementUsed() NN_NOEXCEPT
        {
            int count = usedCount.fetch_add(1);
            NN_SDK_ASSERT_MINMAX(count, 0, maxCount - 1);
            return count;
        }

        /**
         * @brief       mbuf クラスタの使用数をデクリメントします。
         * @return      変更前の値です。
         */
        int DecrementUsed() NN_NOEXCEPT
        {
            int count = usedCount.fetch_sub(1);
            NN_SDK_ASSERT_MINMAX(count, 1, maxCount);
            return count;
        }

        uint32_t                magic;
        uint32_t                unitSize;
        int32_t                 maxCount;
        int32_t                 type;
        std::atomic<int32_t>    usedCount;
        std::atomic<uint32_t>   taggedFreeIndex;

        // mbuf プールの末尾に付加するマジックナンバーで ASCII で pool です。
        static const Bit32 MAGIC_NUMBER = 0x6c6f6f70;
    };

    /**
     * @brief       mbuf プールを管理するクラスです。
     */
    class MbufPool
    {
    public:
        static const Bit16 InvalidIndex = 0xffff;

        /**
         * @brief       コンストラクタです。
         */
        MbufPool() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~MbufPool() NN_NOEXCEPT;

        /**
         * @brief       mbuf プールを初期化します。
         * @param[in]   id          mbuf プールの一意な識別子です。
         * @param[in]   unitSize    mbuf プールを構成するクラスタのバイトサイズです。
         * @param[in]   count       mbuf プールを構成するクラスタの数です。
         * @param[in]   type        mbuf プールの用途です。
         * @param[in]   base        mbuf プールに割り当てるバッファの先頭アドレスです。
         * @param[in]   isServer    自身で mbuf プールを管理する場合に true です。
         */
        void Initialize(
            int id, size_t unitSize, int count, int type,
            uintptr_t base, bool isServer) NN_NOEXCEPT;

        /**
         * @brief       mbuf プールを解放します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief       mbuf プールを構成するクラスタの総数を取得します。
         * @return      mbuf プールを構成するクラスタの総数です。
         */
        int GetClusterCount() const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            return m_pMetaInfo->maxCount;
        }

        /**
         * @brief       割り当て可能な未使用のクラスタの総数を取得します。
         * @return      割り当て可能な未使用のクラスタの総数です。
         */
        int GetFreeCount() const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            return m_pMetaInfo->maxCount - m_pMetaInfo->usedCount;
        }

        /**
         * @brief       mbuf クラスタのバイトサイズを取得します。
         * @return      mbuf クラスタのバイトサイズです。
         */
        size_t GetUnitSize() const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            return m_pMetaInfo->unitSize;
        }

        /**
         * @brief       mbuf クラスタに格納可能なデータサイズを取得します。
         * @return      mbuf クラスタに格納可能なデータサイズです。
         */
        size_t GetDataSize() const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            return m_pMetaInfo->unitSize - MbufHeaderSize;
        }

        /**
         * @brief       mbuf プールのバイトサイズを取得します。
         * @return      mbuf プールのバイトサイズです。
         */
        inline size_t GetPoolSize() const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            return GetRequiredMemorySize(m_pMetaInfo->unitSize, m_pMetaInfo->maxCount);
        }

        /**
         * @brief       mbuf クラスタの割り当てを試行します。
         * @param[in]   type        用途です。
         * @retval      nullptr     mbuf クラスタの割り当てに失敗しました。
         * @retval      上記以外    新しく割り当てられた mbuf クラスタです。
         */
        Mbuf* TryAllocate(int type) NN_NOEXCEPT;

        /**
         * @brief       mbuf クラスタを解放します。
         * @param[in]   pMbuf       対象の mbuf クラスタです。
         */
        void Free(Mbuf* pMbuf) NN_NOEXCEPT;

        /**
         * @brief       プール内のインデックスから mbuf クラスタのポインタに変換します。
         * @param[in]   プール内のインデックスです。
         * @return      インデックスに対応する mbuf クラスタへのポインタです。
         */
        Mbuf* GetPtrFromIndex(int index) NN_NOEXCEPT;

        /**
         * @brief       プール内のインデックスから mbuf クラスタのポインタに変換します。
         * @param[in]   プール内のインデックスです。
         * @return      インデックスに対応する mbuf クラスタへのポインタです。
         */
        const Mbuf* GetPtrFromIndex(int index) const NN_NOEXCEPT;

        /**
         * @brief       mbuf クラスタに対応するプール内のインデックスを取得します。
         * @param[in]   対象の mbuf クラスタです。
         * @return      mbuf クラスタに対応するプール内のインデックスです。
         */
        int GetIndexFromPtr(const Mbuf* pMbuf) const NN_NOEXCEPT;

        /**
         * @brief       mbuf プールの生成に必要なバッファサイズを計算します。
         * @param[in]   unitSize    mbuf プールで管理するクラスタのバイトサイズです。
         * @param[in]   count       mbuf プールで管理するクラスタの数です。
         * @return      mbuf プールの生成に必要なバッファサイズです。
         */
        static size_t GetRequiredMemorySize(size_t unitSize, int count) NN_NOEXCEPT;

        /**
         * @brief       mbuf プールごとの空きmbuf unit数を返します。
         * @param[in]   type    mbuf プール識別子です。
         * @return      指定した mbuf プールの空きmbuf unit数です。指定したプールが存在しない場合-1が返ります。
         */
        int GetFreeCountByType( int type ) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMetaInfo);
            if( type == m_pMetaInfo->type )
            {
                return m_pMetaInfo->maxCount - m_pMetaInfo->usedCount;
            }
            return -1;
        }


    private:

        /**
         * @brief       mbuf クラスタのフリーリストを生成します。
         */
        void ConstructFreeChain() NN_NOEXCEPT;

        /**
         * @brief       全ての mbuf クラスタが解放され、クリーンな状態であることを検証します。
         * @return      クリーンな状態であれば true です。
         */
        bool IsClean() const NN_NOEXCEPT;

        MbufPoolMetaInfo*   m_pMetaInfo;
        uintptr_t           m_Base;
        Bit16               m_Id;
    };

}}} // end of namespace nn::mbuf::detail
