﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/lr/lr_Path.h>

namespace nn { namespace lr {

    class IRegisteredLocationResolver;
    class RegisteredLocationResolver
    {
        NN_DISALLOW_COPY(RegisteredLocationResolver);

    public:

        /**
        * @brief    コンストラクタです。
        */
        RegisteredLocationResolver() NN_NOEXCEPT{}

        /**
        * @brief    コンストラクタです。
        */
        explicit RegisteredLocationResolver(sf::SharedPointer<IRegisteredLocationResolver> interface) NN_NOEXCEPT : m_Interface(interface) {}

        RegisteredLocationResolver(RegisteredLocationResolver&& rvalue) NN_NOEXCEPT
        {
            m_Interface = std::move(rvalue.m_Interface);
        }

        RegisteredLocationResolver& operator=(RegisteredLocationResolver&& rvalue) NN_NOEXCEPT
        {
            RegisteredLocationResolver(std::move(rvalue)).swap(*this);

            return *this;
        }

        void swap(RegisteredLocationResolver& other) NN_NOEXCEPT
        {
            std::swap(m_Interface, other.m_Interface);

        }
        /**
         * @brief   あらかじめ RegisterProgramPath で登録したプログラムのプログラムパスを取得します。
         *
         * @detail  該当するプログラム ID が登録されていない場合、ResultProgramNotFound を返します。
         *
         * @param[out]   outValue    プログラムパスを取得するバッファ
         * @param[in]    id          プログラムパスを取得するプログラムの ID
         *
         * @return   処理の結果が返ります。
         * @retval   ResultProgramNotFound   指定されたプログラムが登録されていません。
         */
        Result ResolveProgramPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
         * @brief   プログラム ID と、それに対応するプログラムのパスを lr に登録します。
         *
         * @detail  登録したパスは、ResolveProgramPath で取得できます。
         *          同時に登録可能なプログラムの数は 1 つです。
         *          登録されたプログラムがある状態で Register を呼んだ場合、設定は上書きされます。
         *
         * @param[in]    id    プログラムの ID
         * @param[in]    path  対応するパス
         *
         * @return   処理の結果が返ります。
         */
        Result RegisterProgramPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT;

        /**
         * @brief   登録済みのプログラム ID とパスのペアを削除します。
         *
         * @detail  該当するプログラム ID が登録されていないときも成功を返します。
         *
         * @param[in]    id    プログラムの ID
         *
         * @return   処理の結果が返ります。
         */
        Result UnregisterProgramPath(ncm::ProgramId id) NN_NOEXCEPT;

        /**
         * @brief    指定されたプログラム ID で解決されるプログラムパスの設定を上書きします。
         *
         * @detail   上書き可能な設定の個数は固定個（個数未定）です。
         *
         * @param[in]    id              上書き設定対象のプログラムの ID
         * @param[in]    redirectedPath  上書き設定するパス
         *
         * @pre
         *           - 開発環境で実行される
         */
        void RedirectProgramPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
         * @brief   あらかじめ RegisterHtmlDocumentPath で登録したプログラムのプログラムパスを取得します。
         *
         * @detail  該当するプログラム ID が登録されていない場合、ResultHtmlDocumentNotFound を返します。
         *
         * @param[out]   outValue    プログラムパスを取得するバッファ
         * @param[in]    id          プログラムパスを取得するプログラムの ID
         *
         * @return   処理の結果が返ります。
         * @retval   ResultHtmlDocumentNotFound   指定されたプログラムが登録されていません。
         */
        Result ResolveHtmlDocumentPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
         * @brief   プログラム ID と、それに対応するプログラムのパスを lr に登録します。
         *
         * @detail  登録したパスは、ResolveHtmlDocumentPath で取得できます。
         *          同時に登録可能なプログラムの数は 1 つです。
         *          登録されたプログラムがある状態で Register を呼んだ場合、設定は上書きされます。
         *
         * @param[in]    id    プログラムの ID
         * @param[in]    path  対応するパス
         *
         * @return   処理の結果が返ります。
         */
        Result RegisterHtmlDocumentPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT;

        /**
         * @brief   登録済みのプログラム ID とパスのペアを削除します。
         *
         * @detail  該当するプログラム ID が登録されていないときも成功を返します。
         *
         * @param[in]    id    プログラムの ID
         *
         * @return   処理の結果が返ります。
         */
        Result UnregisterHtmlDocumentPath(ncm::ProgramId id) NN_NOEXCEPT;

        /**
         * @brief    指定されたプログラム ID で解決されるプログラムパスの設定を上書きします。
         *
         * @detail   上書き可能な設定の個数は固定個（個数未定）です。
         *
         * @param[in]    id              上書き設定対象のプログラムの ID
         * @param[in]    redirectedPath  上書き設定するパス
         *
         * @pre
         *           - 開発環境で実行される
         */
        void RedirectHtmlDocumentPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

    private:
        sf::SharedPointer<IRegisteredLocationResolver> m_Interface;
    };
}}
